# CashBook ERP - cPanel Deployment Guide

## Architecture

```
Domain: cashbook.yourcompany.com
├── public_html/               (Frontend - Static HTML/JS/CSS)
│   ├── index.html
│   ├── _next/
│   └── .htaccess              (SPA routing)
│
└── api/                       (Backend - PHP API)
    ├── public/index.php       (Entry point)
    ├── src/                   (Application code)
    ├── vendor/                (Dependencies)
    └── .htaccess              (URL rewriting)
```

Both frontend and backend run on the **same domain** (`cashbook.yourcompany.com`):
- Frontend accessible at `https://cashbook.yourcompany.com/`
- Backend API accessible at `https://cashbook.yourcompany.com/api/*`
- No CORS headers needed (same-domain requests)

---

## Prerequisites

- cPanel with PHP 8.2+ support
- PostgreSQL database (if available on your host)
  - Alternative: MySQL (requires schema conversion - see MySQL Migration section)
- SSH access (recommended) or File Manager access
- Composer for PHP dependency management

---

## Step 1: Set Up Frontend (Static Export)

### 1.1 Build the Frontend

On your local machine or development server:

```bash
cd frontend
npm install
npm run export
```

This generates a `frontend/out/` directory with static files.

### 1.2 Upload Frontend to cPanel

1. In cPanel **File Manager**, open `public_html`
2. Delete any existing files (if starting fresh)
3. Upload all contents of `frontend/out/` to `public_html/`

**Directory structure in public_html:**
```
index.html
_next/
login/
register/
dashboard/
.htaccess
```

### 1.3 Upload .htaccess

1. If not already present, upload `.htaccess.frontend` (rename to `.htaccess`) to `public_html/`
2. This enables SPA routing for Next.js pages

```apache
# Routes all requests to index.html except real files/directories
RewriteRule ^ index.html [QSA,L]
```

---

## Step 2: Set Up Backend (PHP API)

### 2.1 Create Backend Directory

1. In cPanel **File Manager**, create a new folder `api` inside `public_html`
2. Directory: `public_html/api/`

### 2.2 Upload Backend Code

1. Upload all files from `backend/` to `public_html/api/`, **except:**
   - `Dockerfile`
   - `docker/`
   - `.dockerignore`
   - `docker-compose*.yml`

**Directory structure in public_html/api:**
```
public/
  index.php          (API entry point)
src/
  Controllers/
  Core/
  Middleware/
  Util/
config/
  app.php
  database.cpanel.php
routes/
  api.php
vendor/             (dependencies)
.htaccess           (URL rewriting)
.env.cpanel
composer.json
```

### 2.3 Upload .htaccess for Backend

Copy `backend/.htaccess` to `public_html/api/.htaccess`:

```apache
RewriteRule ^ public/index.php [QSA,L]
```

---

## Step 3: Configure Database

### 3.1 Create PostgreSQL Database (cPanel > PostgreSQL Databases)

1. Log into cPanel
2. Go to **PostgreSQL Databases**
3. Create a new database: `cashbook_prod`
4. Create a new user: `cashbook_user`
5. Assign user to database with all privileges
6. Note the connection credentials:
   - Host: `localhost` (or Unix socket if available)
   - Database: `cashbook_prod`
   - User: `cashbook_user`
   - Password: `[your-strong-password]`

### 3.2 Import Database Schema

**Option A: Via cPanel phpPgAdmin**

1. Go to **PhpPgAdmin** in cPanel
2. Select database `cashbook_prod`
3. Go to **SQL** tab
4. Copy & paste contents of `backend/database/schema.cpanel.sql`
5. Execute

**Option B: Via SSH**

```bash
cd ~/public_html/api
psql -h localhost -U cashbook_user -d cashbook_prod < ../database/schema.cpanel.sql
```

---

## Step 4: Configure Environment

### 4.1 Create .env File for Backend

In `public_html/api/.env` (create if doesn't exist):

```env
# Application
APP_NAME=CashBook ERP
APP_ENV=production
APP_DEBUG=false
APP_URL=https://cashbook.yourcompany.com
FRONTEND_URL=https://cashbook.yourcompany.com

# Database (from cPanel credentials above)
DB_HOST=localhost
DB_PORT=5432
DB_NAME=cashbook_prod
DB_USER=cashbook_user
DB_PASS=[your-strong-password-from-step-3]

# JWT Auth
JWT_SECRET=[generate-a-strong-64-character-random-string]
JWT_EXPIRY=3600
JWT_REFRESH_EXPIRY=604800

# Payment Gateways (optional)
PAYSTACK_SECRET_KEY=sk_live_xxxxx
HUBTEL_CLIENT_ID=
HUBTEL_CLIENT_SECRET=

# Email/SMTP (optional)
SMTP_HOST=
SMTP_PORT=587
SMTP_USERNAME=
SMTP_PASSWORD=
```

**Generate secure JWT_SECRET:**
```bash
openssl rand -base64 48
```

### 4.2 Update Backend Config

Edit `public_html/api/config/app.php`:

```php
// Make sure APP_ENV is 'production'
define('APP_ENV', $_ENV['APP_ENV'] ?? 'production');
define('APP_DEBUG', false);  // Never true in production
```

### 4.3 Set File Permissions

Via SSH:

```bash
cd ~/public_html/api
chmod 755 public
chmod 644 public/index.php
chmod 755 src vendor config routes
```

---

## Step 5: Install PHP Dependencies

### Via SSH (Recommended)

```bash
cd ~/public_html/api
composer install --optimize-autoloader --no-dev
```

### Via cPanel Terminal

1. cPanel > **Terminal** (if available)
2. Run same commands as above

**Note:** If Composer is not installed on your server, download `vendor.zip` locally:
```bash
cd backend
composer install --optimize-autoloader --no-dev
zip -r vendor.zip vendor/
```
Then upload `vendor.zip` to `public_html/api/` and extract.

---

## Step 6: Verify Setup

### Test Frontend

1. Open `https://cashbook.yourcompany.com/`
2. You should see the login page
3. Test that navigation works (click buttons, go to different pages)

### Test Backend API

1. Open `https://cashbook.yourcompany.com/api/health`
2. Should return JSON:
   ```json
   {
     "status": "ok",
     "app": "CashBook ERP",
     "version": "1.0.0",
     "timestamp": "2026-02-19T..."
   }
   ```

### Test API Login

```bash
curl -X POST https://cashbook.yourcompany.com/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "company_name": "Test Company",
    "tin": "C0000000000",
    "name": "Admin User",
    "email": "admin@test.com",
    "password": "Password123!"
  }'
```

Should return a success response with user/company IDs.

---

## Step 7: SSL Certificate (HTTPS)

1. In cPanel, go to **SSL/TLS Manager**
2. Select "Auto-generate a self-signed certificate" or use Let's Encrypt (recommended)
3. Force HTTPS:
   - In `.htaccess` (public_html):
     ```apache
     RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [R=301,L]
     ```

---

## Step 8: Database Backups

### Automated Backup (cPanel)

1. Go to **Backup** in cPanel
2. Enable automated PostgreSQL backups if available

### Manual Backup (SSH)

```bash
pg_dump -h localhost -U cashbook_user -d cashbook_prod > backup_$(date +%Y%m%d).sql
gzip backup_*.sql
```

### Restore from Backup

```bash
gunzip backup_2026-02-19.sql.gz
psql -h localhost -U cashbook_user -d cashbook_prod < backup_2026-02-19.sql
```

---

## Troubleshooting

### Issue: 404 errors on SPA routes (e.g., `/dashboard`)

**Solution:** Make sure `.htaccess` is in `public_html` and `mod_rewrite` is enabled.

```bash
# Via cPanel Terminal:
a2enmod rewrite
systemctl restart apache2
```

### Issue: API returns 500 error

**Check logs:**
```bash
# Frontend
tail -f ~/public_html/error_log

# Backend
tail -f ~/public_html/api/error_log
# OR check PHP error log:
tail -f /var/log/apache2/error_log
```

### Issue: Database connection fails

**Verify credentials:**
```bash
psql -h localhost -U cashbook_user -d cashbook_prod
```

If it fails check:
1. User exists: `\du` in psql
2. Database exists: `\l` in psql
3. Permissions: `GRANT ALL ON DATABASE cashbook_prod TO cashbook_user;`

### Issue: UUID function not found

**If you see `ERROR: function uuid_generate_v4() does not exist`:**

The cPanel PostgreSQL doesn't have uuid-ossp extension. The code now falls back to PHP UUID generation automatically. Make sure `src/Util/UUID.php` is present.

### Issue: File upload fails

Check `public_html/api/public` directory permissions:
```bash
chmod 777 ~/public_html/api/public/uploads/
```

---

## File Uploads

By default, uploaded files go to `public_html/api/public/uploads/`.

To serve them via the frontend:

1. In cPanel, symlink uploads to frontend:
   ```bash
   cd ~/public_html
   ln -s api/public/uploads/ uploads
   ```

2. Frontend can now access: `https://cashbook.yourcompany.com/uploads/filename`

---

## Performance Optimization

### 1. Enable PHP Opcache

In cPanel > **MultiPHP Manager**:
- Select domain
- Ensure "Opcache" is enabled

### 2. Compress Responses

Already configured in `.htaccess` files (gzip).

### 3. Cache Static Assets

Already configured in `.htaccess` files (30-day cache for images/JS/CSS).

### 4. Database Indexes

Schema includes all necessary indexes. Monitor query performance:

```bash
# SSH access:
psql -U cashbook_user -d cashbook_prod -c "\d+ audit_log"  # Check indexes
```

---

## Regular Maintenance

### Update Composer Dependencies (Monthly)

```bash
cd ~/public_html/api
composer update
```

### Monitor Error Logs (Weekly)

```bash
tail -f ~/public_html/error_log
tail -f ~/public_html/api/error_log
```

### Check Database Size

```bash
psql -U cashbook_user -d cashbook_prod -c "SELECT pg_size_pretty(pg_database_size('cashbook_prod'));"
```

### Rotate Audit Logs (Monthly)

```bash
# Keep only last 6 months of audit logs
psql -U cashbook_user -d cashbook_prod -c "DELETE FROM audit_log WHERE created_at < NOW() - INTERVAL '6 months';"
```

---

## Security Checklist

- [ ] Change `JWT_SECRET` to a strong random value
- [ ] Change `DB_PASS` to a strong random value
- [ ] Set `APP_DEBUG=false` in production
- [ ] Enable HTTPS (SSL certificate)
- [ ] Disable directory listing in `.htaccess`
- [ ] Remove sensitive files (`.env.example`, `README.md`) from public folders
- [ ] Set strict file permissions (644 for files, 755 for directories)
- [ ] Regularly review error logs
- [ ] Enable database backups

---

## MySQL Alternative (If PostgreSQL Not Available)

If your cPanel host doesn't support PostgreSQL, you'll need to:

1. Create a MySQL database equivalent of the schema
2. Replace UUID with AUTO_INCREMENT or CHAR(36)
3. Replace TIMESTAMP WITH TIME ZONE with DATETIME
4. Update database config to use MySQL PDO driver

Contact support if you need help converting the schema to MySQL.

---

## Support & Issues

For issues, check:
1. Error logs (see Troubleshooting section)
2. Database connection (see Troubleshooting section)
3. File permissions (see Troubleshooting section)
4. DNS/domain propagation (wait up to 24 hours after domain setup)

---

**Deployment completed!** Your CashBook ERP is now running on cPanel.
