# CashBook ERP - cPanel Production Deployment Summary

**Date**: February 19, 2026  
**Status**: ✅ Ready for cPanel Deployment  
**Architecture**: Single Domain (Frontend + Backend on same domain)

---

## Changes Made for cPanel Compatibility

### 1. Frontend (Next.js → Static HTML)

**File**: `frontend/next.config.js`
```javascript
output: 'export'  // Changed from 'standalone'
images: {
  unoptimized: true,  // Required for static export
}
```

**File**: `frontend/package.json`
```json
"export": "next build"  // Generates static files in 'out/' directory
```

**Result**: 
- Generates static HTML/JS/CSS files in `frontend/out/`
- No Node.js server needed for production
- Perfect for cPanel hosting (Apache only)

---

### 2. Backend (PHP Compatibility)

#### UUID Generation (cPanel-friendly)

**New File**: `backend/src/Util/UUID.php`
- PHP UUID generation without PostgreSQL `uuid-ossp` extension
- Falls back to random_bytes() for secure UUIDs
- Validates UUID format

**Updated**: `backend/src/Core/Controller.php`
```php
protected function generateId(): string {
  // Try PostgreSQL gen_random_uuid() first
  // Falls back to PHP UUID::v4Secure() if not available
}
```

#### Database Configuration

**New File**: `backend/config/database.cpanel.php`
- Unix socket support (common on cPanel)
- TCP connection fallback
- Compatible with cPanel's PostgreSQL setup

#### Database Schema

**New File**: `backend/database/schema.cpanel.sql`
- Uses `gen_random_uuid()` (no `uuid-ossp` extension needed)
- Simplified for cPanel restrictions
- All required indexes included
- Ready for immediate import

---

### 3. Apache Configuration (.htaccess)

#### Frontend Routing

**File**: `.htaccess.frontend` (rename to `.htaccess` in public_html/)
```apache
RewriteRule ^ index.html [QSA,L]  # SPA routing
```
- Routes all requests to `index.html` (except real files)
- Enables Next.js SPA routing on cPanel
- Includes security headers and gzip compression

#### Backend Routing

**File**: `backend/.htaccess`
```apache
RewriteRule ^ public/index.php [QSA,L]  # Routes to PHP entry point
```
- Routes all API requests to `public/index.php`
- Blocks access to sensitive files (.env, vendor/, etc.)
- Includes security headers and PHP settings

---

### 4. Environment Configuration

**New Files**:
- `.env.cpanel.example` - Template for cPanel secrets
- `backend/.env.cpanel.example` - Backend secrets template

**Variables**:
```env
APP_ENV=production
APP_DEBUG=false
DB_HOST=localhost              # cPanel PostgreSQL
DB_PORT=5432
JWT_SECRET=[strong-64-chars]
```

---

## Deployment Architecture

```
Domain: cashbook.yourcompany.com
├── public_html/ (Frontend - Static)
│   ├── index.html
│   ├── _next/static/
│   ├── login/index.html
│   ├── dashboard/index.html
│   ├── .htaccess (SPA routing)
│   └── (no Node.js needed)
│
└── public_html/api/ (Backend - PHP)
    ├── public/index.php (API entry point)
    ├── src/ (Controllers, Middleware, etc.)
    ├── vendor/ (Composer dependencies)
    ├── .htaccess (URL rewriting)
    ├── .env (Secrets)
    └── (Uses Apache, no Docker)

Database: PostgreSQL (in cPanel)
├── Database: cashbook_prod
├── User: cashbook_user
└── Schema: Imported from schema.cpanel.sql
```

**Same Domain**:
- Frontend calls API at `/api/auth/login`, `/api/invoices`, etc.
- No CORS headers needed (same-domain requests)
- All traffic goes through Apache on port 80/443

---

## Quick Start (Step-by-Step)

### 1. Build Frontend Static Export
```bash
cd frontend
npm run export
# Creates: frontend/out/ with all static files
```

### 2. Log into cPanel
```
https://your-domain.com:2083
Username: your-cpanel-username
```

### 3. Upload Frontend
- **File Manager** → `public_html/`
- Delete existing files
- Upload all contents of `frontend/out/`
- Upload `.htaccess.frontend` (rename to `.htaccess`)

### 4. Create API Directory
- **File Manager** → Create folder `api` in `public_html`
- Set permissions: `755`

### 5. Upload Backend
- **File Manager** → `public_html/api/`
- Upload all backend files **except**:
  - Dockerfile
  - docker/
  - .dockerignore
  - docker-compose*
  - Makefile (if any)

### 6. Set Up Database
- **PostgreSQL Databases** → Create `cashbook_prod`
- Create user `cashbook_user` with strong password
- **PhpPgAdmin** → Import `schema.cpanel.sql`

### 7. Configure Environment
- **File Manager** → `public_html/api/.env`
- Copy from `.env.cpanel.example`
- Update:
  - `APP_URL=https://your-domain.com`
  - `DB_PASS=[from-step-6]`
  - `JWT_SECRET=[generate-with-openssl]`

### 8. Test
- Open `https://your-domain.com/api/health`
- Should return: `{"status":"ok","app":"CashBook ERP",...}`
- Open `https://your-domain.com/`
- Should show login page

---

## Files Created for cPanel

| File | Purpose |
|------|---------|
| `frontend/next.config.js` | Static export config |
| `backend/src/Util/UUID.php` | UUID without PostgreSQL extensions |
| `backend/config/database.cpanel.php` | cPanel database config |
| `backend/database/schema.cpanel.sql` | cPanel-compatible schema |
| `backend/.htaccess` | API routing |
| `.htaccess.frontend` | Frontend SPA routing |
| `backend/.env.cpanel.example` | Environment template |
| `CPANEL_DEPLOYMENT.md` | Detailed deployment guide |
| `CPANEL_CHECKLIST.md` | Pre/post deployment checklist |
| `deploy-cpanel.sh` | Linux deployment script |
| `deploy-cpanel.bat` | Windows deployment script |

---

## Key Features for cPanel

✅ **No Docker** - Pure Apache + PHP  
✅ **No Node.js in Production** - Static HTML export  
✅ **Single Domain** - Both frontend and backend at same URL  
✅ **No CORS Complexity** - Same-domain requests  
✅ **UUID Compatibility** - Works without PostgreSQL extensions  
✅ **Apache Rewriting** - Clean URLs without query strings  
✅ **Security Headers** - Built into .htaccess  
✅ **Gzip Compression** - Automatic in .htaccess  
✅ **Asset Caching** - Long cache for JS/CSS/images  

---

## Post-Deployment

### Database Backups
```bash
# Via SSH
pg_dump -U cashbook_user -d cashbook_prod > backup.sql
gzip backup.sql
```

### Monitor Errors
```bash
# Check:
tail -f ~/public_html/error_log
tail -f ~/public_html/api/error_log
```

### Update PHP Dependencies
```bash
cd ~/public_html/api
composer update
```

---

## Common Issues & Solutions

### **Issue**: Frontend pages return 404
**Solution**: 
1. Ensure `.htaccess` is in `public_html/`
2. Enable mod_rewrite in cPanel > Apache Modules
3. Run: `a2enmod rewrite && systemctl restart apache2`

### **Issue**: API returns 500 error
**Solution**:
1. Check error logs: `tail -f ~/public_html/api/error_log`
2. Verify `.env` file with correct DB credentials
3. Test DB connection: `psql -U cashbook_user -d cashbook_prod`

### **Issue**: Database connection fails
**Solution**:
1. Verify credentials in `.env` match cPanel Database
2. Check PostgreSQL is running: `systemctl status postgresql`
3. Grant privileges: `GRANT ALL ON DATABASE cashbook_prod TO cashbook_user;`

### **Issue**: UUID not found error
**Solution**:
- The code now falls back to PHP UUIDs automatically
- If still failing, ensure `src/Util/UUID.php` exists

---

## Support Files

📄 **CPANEL_DEPLOYMENT.md** - Complete deployment guide (10+ pages)  
📄 **CPANEL_CHECKLIST.md** - Pre/post deployment checklist  
🔧 **deploy-cpanel.sh** - Automated setup (Linux/macOS)  
🔧 **deploy-cpanel.bat** - Setup script (Windows)  

---

## Security Checklist

- [ ] Change `JWT_SECRET` to strong random value
- [ ] Change `DB_PASS` to strong random value (20+ chars)
- [ ] Set `APP_DEBUG=false`
- [ ] Enable HTTPS (Let's Encrypt)
- [ ] Add security headers (in .htaccess)
- [ ] Disable directory listing
- [ ] Remove sensitive files from public folders
- [ ] Set strict file permissions (644 for files, 755 for directories)
- [ ] Regular database backups
- [ ] Monitor error logs

---

## Validation

✅ Static export generated: `frontend/out/` (20 pages)  
✅ Backend prepared: All PHP files ready  
✅ Database schema: `schema.cpanel.sql` ready  
✅ Configuration files: `.htaccess` and `.env` templates ready  
✅ Documentation: Comprehensive guides created  

---

**Your CashBook ERP is now ready for production deployment on cPanel!**

Next: Follow CPANEL_DEPLOYMENT.md for step-by-step instructions.
