# CashBook - Cloud Accounting & Bookkeeping for Ghana SMEs

A full-stack cloud-based accounting, invoicing, POS, and payroll application built specifically for Ghanaian small and medium enterprises. Compliant with Ghana Revenue Authority (GRA) tax systems including VAT, NHIL, GETFund Levy, PAYE, and SSNIT.

---

## Features

- **Double-Entry Accounting** — Chart of accounts, journal entries, trial balance, general ledger
- **Invoicing & Billing** — Create, send, track invoices with Ghana tax calculations
- **Point of Sale (POS)** — Session-based POS with barcode scanning, split payments, daily reports
- **Payroll** — Ghana PAYE (7 brackets), SSNIT Tier 1/2/3, automated payslip generation
- **Inventory Management** — Stock tracking, low-stock alerts, barcode lookup, categories
- **Fixed Asset Management** — Depreciation schedules, disposal tracking, journal integration
- **Tax Compliance** — VAT (Standard 15% / Flat Rate 4%), NHIL, GETFund, COVID Levy, withholding tax
- **Financial Reports** — Income statement, balance sheet, cash flow, AR aging, tax summary
- **Contact Management** — Customers, suppliers, transaction history
- **Admin Dashboard** — User management, audit log, bank accounts, data export
- **Payment Gateways** — Paystack and Hubtel integration (Ghana)

## Tech Stack

| Layer | Technology |
|-------|-----------|
| Backend | PHP 8.2+, Custom MVC Framework |
| Database | PostgreSQL 16 |
| Frontend | Next.js 14, TypeScript, Tailwind CSS |
| Auth | JWT (access + refresh tokens) |
| Charts | Chart.js / react-chartjs-2 |
| PDF/Excel | TCPDF, PhpSpreadsheet |
| Deployment | Docker + Nginx (single domain) |

---

## Project Structure

```
cash/
├── backend/
│   ├── config/           # App & database configuration
│   ├── database/         # PostgreSQL schema & migrations
│   ├── docker/           # Nginx & Supervisor configs
│   ├── public/           # API entry point (index.php)
│   ├── routes/           # API route definitions
│   ├── src/
│   │   ├── Controllers/  # Business logic controllers
│   │   ├── Core/         # Framework (Router, Request, Response)
│   │   └── Middleware/   # Auth & role middleware
│   ├── .env.example
│   ├── composer.json
│   └── Dockerfile
├── frontend/
│   ├── src/
│   │   ├── app/          # Next.js App Router pages
│   │   ├── components/   # Reusable UI components
│   │   ├── context/      # React context (Auth)
│   │   └── lib/          # API client, utilities
│   ├── Dockerfile        # Multi-stage production build
│   ├── vercel.json       # Vercel deployment (alternative)
│   └── package.json
├── nginx/
│   └── default.conf      # Production reverse proxy config
├── docker-compose.yml          # Development
├── docker-compose.prod.yml     # Production (single domain)
├── .env.production.example     # Production env template
└── README.md
```

---

## Quick Start

### Prerequisites

- Docker & Docker Compose (v2+)
- Node.js 18+ and npm

### 1. Clone & Configure

```bash
git clone <repo-url> cashbook
cd cashbook

# Backend environment
cp backend/.env.example backend/.env
# Edit backend/.env with your database credentials, JWT secret, etc.

# Frontend environment
cp frontend/.env.local.example frontend/.env.local
```

### 2. Start Backend (Docker)

```bash
# Build and start all backend services (PHP API + PostgreSQL)
docker compose up -d --build
```

This starts:
- **PHP API** on `http://localhost:8000`
- **PostgreSQL** on `localhost:5432`
- Database schema is auto-applied on first run

#### Useful Backend Commands

```bash
# Check container status
docker compose ps

# View API logs (live)
docker compose logs -f app

# View database logs
docker compose logs -f db

# Stop all services
docker compose down

# Stop and remove all data (reset database)
docker compose down -v

# Rebuild after code changes
docker compose up -d --build

# Open a shell inside the PHP container
docker compose exec app sh

# Run composer commands inside the container
docker compose exec app composer install
docker compose exec app composer dump-autoload

# Connect to PostgreSQL directly
docker compose exec db psql -U cashbook -d cashbook

# Restart only the API container
docker compose restart app
```

#### Running Backend Without Docker (Local PHP)

If you prefer running PHP locally without Docker:

```bash
# Requirements: PHP 8.2+, PostgreSQL 16, Composer

# 1. Install PHP dependencies
cd backend
composer install

# 2. Set up PostgreSQL
# Create the database and user
psql -U postgres -c "CREATE USER cashbook WITH PASSWORD 'cashbook_secret';"
psql -U postgres -c "CREATE DATABASE cashbook OWNER cashbook;"

# 3. Apply the schema
psql -U cashbook -d cashbook -f database/schema.sql

# 4. Configure environment
cp .env.example .env
# Edit .env with your local database settings:
#   DB_HOST=localhost
#   DB_PORT=5432
#   DB_NAME=cashbook
#   DB_USER=cashbook
#   DB_PASSWORD=cashbook_secret

# 5. Start the PHP built-in server
php -S localhost:8000 -t public
```

#### Verify Backend is Running

```bash
# Health check - should return a JSON response
curl http://localhost:8000/api/health

# Or test auth endpoint
curl -X POST http://localhost:8000/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{"company_name":"Test Co","tin":"C0000000000","name":"Admin","email":"admin@test.com","password":"Password123!"}'
```

### 3. Start Frontend (Development)

```bash
cd frontend
npm install
npm run dev
```

Frontend runs at `http://localhost:3000`

### 4. First Login

1. Open `http://localhost:3000`
2. Click **Get Started** → Register a new company
3. Provide company name, TIN, your details
4. You're in! The system creates your chart of accounts and tax rates automatically.

---

## Backend API

Base URL: `http://localhost:8000/api`

### Authentication

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/auth/register` | Register company + admin |
| POST | `/auth/login` | Login, returns JWT |
| POST | `/auth/refresh` | Refresh access token |
| GET | `/auth/profile` | Current user profile |

### Accounting

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/accounts` | List chart of accounts |
| POST | `/accounts` | Create account |
| GET/POST | `/journal-entries` | List / create journal entries |
| POST | `/journal-entries/:id/post` | Post journal entry |
| GET | `/trial-balance` | Trial balance report |
| GET | `/general-ledger` | General ledger |

### Invoicing

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET/POST | `/invoices` | List / create invoices |
| GET/PUT | `/invoices/:id` | View / update invoice |
| POST | `/invoices/:id/payment` | Record payment |

### POS

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/pos/sessions/open` | Open POS session |
| POST | `/pos/sessions/close` | Close session |
| POST | `/pos/sales` | Create sale |
| GET | `/pos/daily-report` | Daily report |

### Payroll

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET/POST | `/employees` | List / add employees |
| POST | `/payroll/run` | Run payroll period |
| POST | `/payroll/:id/approve` | Approve & post payroll |

### Tax

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/tax/rates` | Ghana tax rates |
| POST | `/tax/calculate` | Calculate VAT/NHIL/GETFund |
| POST | `/tax/calculate-paye` | Calculate PAYE |
| GET/POST | `/tax/filings` | Tax filing management |

### Reports

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/reports/income-statement` | Profit & loss |
| GET | `/reports/balance-sheet` | Balance sheet |
| GET | `/reports/cash-flow` | Cash flow statement |
| GET | `/reports/tax-summary` | Tax summary |
| GET | `/reports/ar-aging` | AR aging report |
| GET | `/reports/dashboard` | Dashboard KPIs |

---

## Ghana Tax Compliance

### VAT (Value Added Tax)
- **Standard Rate**: 15% on taxable supply
- **Flat Rate**: 4% for qualifying businesses
- **Compound calculation**: NHIL (2.5%) + GETFund (2.5%) + COVID Levy (1%) applied first, then VAT on the total

### PAYE (Pay As You Earn)
Monthly brackets:
| Annual Income (GHS) | Rate |
|---------------------|------|
| 0 – 5,880 | 0% |
| 5,881 – 7,920 | 5% |
| 7,921 – 12,000 | 10% |
| 12,001 – 16,320 | 17.5% |
| 16,321 – 24,000 | 25% |
| 24,001 – 60,000 | 30% |
| Above 60,000 | 35% |

### SSNIT
- Employee contribution: 5.5% of basic salary
- Employer contribution: 13% of basic salary
- Tier 2: 5% (managed by trustees)
- Tier 3: Voluntary

---

## Deployment

### Production — Single Domain (Docker + Nginx)

The production setup runs the entire stack behind nginx on a single domain.
All traffic goes through nginx: `/api/*` → PHP backend, everything else → Next.js.

**Architecture:**
```
Client → nginx (:80/443)
              ├── /api/*      → PHP backend (:8000 internal)
              └── /*          → Next.js frontend (:3000 internal)
```

#### 1. Configure Environment

```bash
cp .env.production.example .env
nano .env
```

Set these values:
```env
APP_URL=https://yourdomain.com
POSTGRES_PASSWORD=<strong-random-64-chars>
JWT_SECRET=<strong-random-64-chars>
```

#### 2. Build & Deploy

```bash
docker compose -f docker-compose.prod.yml up -d --build
```

This starts 4 containers:
| Container | Service | Port |
|-----------|---------|------|
| `cashbook-proxy` | Nginx reverse proxy | 80 (public) |
| `cashbook-api` | PHP 8.2 + PostgreSQL client | 8000 (internal) |
| `cashbook-web` | Next.js standalone server | 3000 (internal) |
| `cashbook-db` | PostgreSQL 16 | 5432 (internal) |

#### 3. Enable SSL (Recommended)

Edit `nginx/default.conf` — uncomment the SSL lines and mount certificates:

```bash
# In docker-compose.prod.yml, uncomment the SSL volume:
# - ./nginx/ssl:/etc/nginx/ssl:ro

# Place your certs:
mkdir -p nginx/ssl
cp fullchain.pem nginx/ssl/
cp privkey.pem nginx/ssl/
```

For Let's Encrypt with Certbot:
```bash
certbot certonly --standalone -d yourdomain.com
cp /etc/letsencrypt/live/yourdomain.com/fullchain.pem nginx/ssl/
cp /etc/letsencrypt/live/yourdomain.com/privkey.pem nginx/ssl/
docker compose -f docker-compose.prod.yml restart nginx
```

#### 4. Verify Deployment

```bash
curl https://yourdomain.com/api/health
# → {"status":"ok","app":"CashBook ERP","version":"1.0.0",...}

curl https://yourdomain.com/
# → HTML of the login page
```

#### Production Commands

```bash
# View logs
docker compose -f docker-compose.prod.yml logs -f

# Restart a specific service
docker compose -f docker-compose.prod.yml restart app

# Update and rebuild
git pull
docker compose -f docker-compose.prod.yml up -d --build

# Database backup
docker compose -f docker-compose.prod.yml exec db \
  pg_dump -U cashbook cashbook > backup_$(date +%Y%m%d).sql

# Database restore
cat backup.sql | docker compose -f docker-compose.prod.yml exec -T db \
  psql -U cashbook cashbook

# Scale (when behind a load balancer)
docker compose -f docker-compose.prod.yml up -d --scale app=3
```

#### Production Features

- **Rate limiting**: 30 req/s for API, 5 req/s for auth endpoints
- **Gzip compression**: Enabled for text, JSON, JS, CSS, SVG
- **Security headers**: X-Frame-Options, X-Content-Type-Options, XSS-Protection, Referrer-Policy
- **Log rotation**: JSON driver with max 10MB × 5 files per container
- **Health checks**: nginx `/health`, PostgreSQL `pg_isready`
- **No exposed ports**: Only nginx port 80/443 is published; db, api, frontend are internal only
- **OPcache**: PHP OPcache enabled with production-optimized settings
- **Standalone Next.js**: Minimal production image (~150MB) using output: 'standalone'

---

### Alternative — Vercel (Frontend) + VPS (Backend)

If you prefer hosting the frontend on Vercel separately:

```bash
cd frontend
npx vercel
```

Set Vercel environment variables:
- `NEXT_PUBLIC_API_URL` = `https://api.yourdomain.com/api`
- `BACKEND_URL` = `https://api.yourdomain.com`

Then run backend on your VPS:
```bash
docker compose up -d --build
```

---

## Environment Variables

### Production (`.env` at project root)

```env
APP_URL=https://cashbook.yourdomain.com
HTTP_PORT=80
POSTGRES_DB=cashbook
POSTGRES_USER=cashbook
POSTGRES_PASSWORD=<strong-random-password>
JWT_SECRET=<strong-random-string-64-chars>
JWT_EXPIRY=3600
JWT_REFRESH_EXPIRY=604800
PAYSTACK_SECRET_KEY=sk_live_xxxx
HUBTEL_CLIENT_ID=
HUBTEL_CLIENT_SECRET=
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USERNAME=your@email.com
SMTP_PASSWORD=app-password
```

### Development — Backend (`backend/.env`)

```env
DB_HOST=localhost
DB_PORT=5432
DB_NAME=cashbook
DB_USER=cashbook
DB_PASSWORD=your_password
JWT_SECRET=your-secret-key-min-32-characters
JWT_EXPIRY=3600
JWT_REFRESH_EXPIRY=604800
```

### Development — Frontend (`frontend/.env.local`)

```env
NEXT_PUBLIC_API_URL=/api
BACKEND_INTERNAL_URL=http://localhost:8000
```

---

## License

Proprietary — Pineweb Consult. All rights reserved.
