<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class AssetController extends Controller
{
    /**
     * GET /assets
     */
    public function index(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $status = $request->query('status'); // active, disposed, fully_depreciated
        $category = $request->query('category');

        $where = "fa.company_id = :cid";
        $params = ['cid' => $companyId];

        if ($status) {
            $where .= " AND fa.status = :status";
            $params['status'] = $status;
        }
        if ($category) {
            $where .= " AND fa.category = :cat";
            $params['cat'] = $category;
        }

        $stmt = $this->db->prepare(
            "SELECT fa.*, coa.account_name as asset_account_name
             FROM fixed_assets fa
             LEFT JOIN chart_of_accounts coa ON fa.asset_account_id = coa.id
             WHERE $where ORDER BY fa.purchase_date DESC"
        );
        $stmt->execute($params);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /assets
     */
    public function create(Request $request): void
    {
        $data = $request->validate([
            'asset_name' => 'required',
            'category' => 'required',
            'purchase_date' => 'required',
            'purchase_cost' => 'required|numeric',
            'useful_life_months' => 'required|numeric'
        ]);

        $companyId = $request->getCompanyId();
        $depMethod = $request->input('depreciation_method', 'straight_line');
        $salvageValue = (float) $request->input('salvage_value', 0);
        $purchaseCost = (float) $data['purchase_cost'];

        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare(
                "INSERT INTO fixed_assets (company_id, asset_name, description, category, purchase_date, purchase_cost,
                 salvage_value, useful_life_months, depreciation_method, current_value, asset_account_id, depreciation_account_id, status)
                 VALUES (:cid, :name, :desc, :cat, :pdate, :cost, :sv, :life, :method, :cv, :aaid, :daid, 'active')
                 RETURNING *"
            );
            $stmt->execute([
                'cid' => $companyId,
                'name' => $data['asset_name'],
                'desc' => $request->input('description'),
                'cat' => $data['category'],
                'pdate' => $data['purchase_date'],
                'cost' => $purchaseCost,
                'sv' => $salvageValue,
                'life' => (int) $data['useful_life_months'],
                'method' => $depMethod,
                'cv' => $purchaseCost,
                'aaid' => $request->input('asset_account_id'),
                'daid' => $request->input('depreciation_account_id')
            ]);
            $asset = $stmt->fetch();

            // Generate depreciation schedule
            $this->generateDepreciationSchedule($asset);

            $this->db->commit();
            $this->auditLog($request, 'asset_created', 'fixed_assets', $asset['id']);
            Response::created($asset, 'Asset created');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to create asset: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /assets/{id}
     */
    public function show(Request $request, string $id): void
    {
        $companyId = $request->getCompanyId();
        $stmt = $this->db->prepare(
            "SELECT fa.*, coa.account_name as asset_account_name, coa2.account_name as depreciation_account_name
             FROM fixed_assets fa
             LEFT JOIN chart_of_accounts coa ON fa.asset_account_id = coa.id
             LEFT JOIN chart_of_accounts coa2 ON fa.depreciation_account_id = coa2.id
             WHERE fa.id = :id AND fa.company_id = :cid"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $asset = $stmt->fetch();

        if (!$asset) {
            Response::notFound('Asset not found');
            return;
        }

        // Get depreciation schedule
        $schedStmt = $this->db->prepare(
            "SELECT * FROM asset_depreciation_schedule WHERE asset_id = :aid ORDER BY period_date"
        );
        $schedStmt->execute(['aid' => $id]);
        $asset['depreciation_schedule'] = $schedStmt->fetchAll();

        Response::success($asset);
    }

    /**
     * POST /assets/{id}/depreciate
     * Run depreciation for current period
     */
    public function runDepreciation(Request $request, string $id): void
    {
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT * FROM fixed_assets WHERE id = :id AND company_id = :cid AND status = 'active'"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $asset = $stmt->fetch();

        if (!$asset) {
            Response::notFound('Active asset not found');
            return;
        }

        // Find next unprocessed depreciation period
        $nextStmt = $this->db->prepare(
            "SELECT * FROM asset_depreciation_schedule WHERE asset_id = :aid AND is_processed = FALSE ORDER BY period_date LIMIT 1"
        );
        $nextStmt->execute(['aid' => $id]);
        $period = $nextStmt->fetch();

        if (!$period) {
            Response::error('No pending depreciation periods', 400);
            return;
        }

        $this->db->beginTransaction();
        try {
            // Mark as processed
            $upd = $this->db->prepare(
                "UPDATE asset_depreciation_schedule SET is_processed = TRUE WHERE id = :id"
            );
            $upd->execute(['id' => $period['id']]);

            // Update asset current value
            $newValue = (float)$asset['current_value'] - (float)$period['depreciation_amount'];
            $updAsset = $this->db->prepare(
                "UPDATE fixed_assets SET current_value = :cv, accumulated_depreciation = accumulated_depreciation + :dep, updated_at = NOW() WHERE id = :id"
            );
            $updAsset->execute([
                'cv' => max(round($newValue, 2), (float)$asset['salvage_value']),
                'dep' => $period['depreciation_amount'],
                'id' => $id
            ]);

            // Check if fully depreciated
            if ($newValue <= (float)$asset['salvage_value']) {
                $this->db->prepare("UPDATE fixed_assets SET status = 'fully_depreciated' WHERE id = :id")
                    ->execute(['id' => $id]);
            }

            // Create journal entry for depreciation
            if ($asset['asset_account_id'] && $asset['depreciation_account_id']) {
                $jeStmt = $this->db->prepare(
                    "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, reference, source, status, created_by)
                     VALUES (:cid, :num, :date, :desc, :ref, 'system', 'posted', :uid) RETURNING id"
                );
                $entryNum = $this->generateNumber($companyId, 'JE', 'journal_entries', 'entry_number');
                $jeStmt->execute([
                    'cid' => $companyId,
                    'num' => $entryNum,
                    'date' => $period['period_date'],
                    'desc' => "Depreciation: {$asset['asset_name']}",
                    'ref' => "DEP-{$asset['id']}",
                    'uid' => $request->getUserId()
                ]);
                $jeId = $jeStmt->fetch()['id'];

                // Debit depreciation expense
                $this->db->prepare(
                    "INSERT INTO journal_entry_lines (journal_entry_id, account_id, description, debit, credit) VALUES (:jeid, :aid, :desc, :amt, 0)"
                )->execute([
                    'jeid' => $jeId, 'aid' => $asset['depreciation_account_id'],
                    'desc' => "Depreciation expense - {$asset['asset_name']}", 'amt' => $period['depreciation_amount']
                ]);

                // Credit accumulated depreciation (asset account)
                $this->db->prepare(
                    "INSERT INTO journal_entry_lines (journal_entry_id, account_id, description, debit, credit) VALUES (:jeid, :aid, :desc, 0, :amt)"
                )->execute([
                    'jeid' => $jeId, 'aid' => $asset['asset_account_id'],
                    'desc' => "Accumulated depreciation - {$asset['asset_name']}", 'amt' => $period['depreciation_amount']
                ]);
            }

            $this->db->commit();
            $this->auditLog($request, 'asset_depreciated', 'fixed_assets', $id);
            Response::success([
                'period' => $period['period_date'],
                'depreciation_amount' => $period['depreciation_amount'],
                'new_book_value' => max(round($newValue, 2), (float)$asset['salvage_value'])
            ], 'Depreciation recorded');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * POST /assets/{id}/dispose
     */
    public function dispose(Request $request, string $id): void
    {
        $companyId = $request->getCompanyId();
        $disposalPrice = (float) $request->input('disposal_price', 0);
        $disposalDate = $request->input('disposal_date', date('Y-m-d'));

        $stmt = $this->db->prepare(
            "SELECT * FROM fixed_assets WHERE id = :id AND company_id = :cid AND status IN ('active','fully_depreciated')"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $asset = $stmt->fetch();

        if (!$asset) {
            Response::notFound('Asset not found or already disposed');
            return;
        }

        $gainLoss = $disposalPrice - (float) $asset['current_value'];

        $this->db->prepare(
            "UPDATE fixed_assets SET status = 'disposed', disposal_date = :dd, disposal_price = :dp, updated_at = NOW() WHERE id = :id"
        )->execute(['dd' => $disposalDate, 'dp' => $disposalPrice, 'id' => $id]);

        $this->auditLog($request, 'asset_disposed', 'fixed_assets', $id);
        Response::success([
            'asset' => $asset['asset_name'],
            'book_value' => (float) $asset['current_value'],
            'disposal_price' => $disposalPrice,
            'gain_loss' => round($gainLoss, 2)
        ], $gainLoss >= 0 ? 'Asset disposed at a gain' : 'Asset disposed at a loss');
    }

    /**
     * Generate straight-line depreciation schedule
     */
    private function generateDepreciationSchedule(array $asset): void
    {
        $depreciable = (float)$asset['purchase_cost'] - (float)$asset['salvage_value'];
        $months = (int)$asset['useful_life_months'];
        if ($months <= 0) return;

        $monthlyDep = round($depreciable / $months, 2);
        $startDate = new \DateTime($asset['purchase_date']);
        $remaining = (float)$asset['purchase_cost'];

        for ($i = 1; $i <= $months; $i++) {
            $periodDate = clone $startDate;
            $periodDate->modify("+{$i} months");

            $dep = ($i === $months) ? round($remaining - (float)$asset['salvage_value'], 2) : $monthlyDep;
            $remaining -= $dep;

            $this->db->prepare(
                "INSERT INTO asset_depreciation_schedule (asset_id, period_number, period_date, depreciation_amount, accumulated_depreciation, book_value)
                 VALUES (:aid, :pn, :pd, :da, :ad, :bv)"
            )->execute([
                'aid' => $asset['id'],
                'pn' => $i,
                'pd' => $periodDate->format('Y-m-d'),
                'da' => $dep,
                'ad' => round((float)$asset['purchase_cost'] - $remaining, 2),
                'bv' => round($remaining, 2)
            ]);
        }
    }
}
