<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;
use Firebase\JWT\JWT;

class AuthController extends Controller
{
    /**
     * POST /auth/register
     * Register a new company and admin user
     */
    public function register(Request $request): void
    {
        $data = $request->validate([
            'company_name' => 'required|min:2',
            'email' => 'required|email',
            'password' => 'required|min:8',
            'first_name' => 'required|min:2',
            'last_name' => 'required|min:2',
            'phone' => 'required'
        ]);

        // Check if email already exists
        $stmt = $this->db->prepare("SELECT id FROM users WHERE email = :email");
        $stmt->execute(['email' => $data['email']]);
        if ($stmt->fetch()) {
            Response::error('Email already registered', 409);
            return;
        }

        try {
            $this->db->beginTransaction();

            // Create tenant
            $slug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $data['company_name']));
            $slug = trim($slug, '-') . '-' . substr(uniqid(), -5);
            $tenantStmt = $this->db->prepare(
                "INSERT INTO tenants (name, slug, plan, max_businesses, max_users)
                 VALUES (:name, :slug, 'starter', 3, 10)
                 RETURNING id"
            );
            $tenantStmt->execute([
                'name' => $data['company_name'],
                'slug' => $slug
            ]);
            $tenant = $tenantStmt->fetch();
            $tenantId = $tenant['id'];

            // Create company (first business)
            $companyStmt = $this->db->prepare(
                "INSERT INTO companies (tenant_id, name, email, phone, city, region) 
                 VALUES (:tenant_id, :name, :email, :phone, :city, :region) 
                 RETURNING id"
            );
            $companyStmt->execute([
                'tenant_id' => $tenantId,
                'name' => $data['company_name'],
                'email' => $data['email'],
                'phone' => $data['phone'],
                'city' => $request->input('city', 'Accra'),
                'region' => $request->input('region', 'Greater Accra')
            ]);
            $company = $companyStmt->fetch();
            $companyId = $company['id'];

            // Create admin user
            $passwordHash = password_hash($data['password'], PASSWORD_BCRYPT, ['cost' => 12]);
            $userStmt = $this->db->prepare(
                "INSERT INTO users (company_id, tenant_id, email, password_hash, first_name, last_name, phone, role) 
                 VALUES (:company_id, :tenant_id, :email, :password_hash, :first_name, :last_name, :phone, 'admin') 
                 RETURNING id"
            );
            $userStmt->execute([
                'company_id' => $companyId,
                'tenant_id' => $tenantId,
                'email' => $data['email'],
                'password_hash' => $passwordHash,
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'phone' => $data['phone']
            ]);
            $user = $userStmt->fetch();

            // Link user to business
            $this->db->prepare(
                "INSERT INTO user_businesses (user_id, company_id, role, is_default) VALUES (:uid, :cid, 'admin', TRUE)"
            )->execute(['uid' => $user['id'], 'cid' => $companyId]);

            // Create default chart of accounts
            $this->db->prepare("SELECT create_default_chart_of_accounts(:company_id)")
                ->execute(['company_id' => $companyId]);

            // Create default tax rates for the company
            $this->db->prepare(
                "INSERT INTO tax_rates (company_id, tax_name, tax_code, rate, tax_type, effective_date, description)
                 SELECT :company_id, tax_name, tax_code, rate, tax_type, effective_date, description
                 FROM tax_rates WHERE company_id IS NULL"
            )->execute(['company_id' => $companyId]);

            $this->db->commit();

            // Generate JWT
            $token = $this->generateToken($user['id'], $companyId, $tenantId, $data['email'], 'admin', $data['first_name'], $data['last_name']);
            $refreshToken = $this->generateRefreshToken($user['id'], $companyId, $tenantId);

            Response::created([
                'token' => $token,
                'refresh_token' => $refreshToken,
                'expires_in' => (int) ($_ENV['JWT_EXPIRY'] ?? 3600),
                'user' => [
                    'id' => $user['id'],
                    'email' => $data['email'],
                    'first_name' => $data['first_name'],
                    'last_name' => $data['last_name'],
                    'role' => 'admin',
                    'tenant_id' => $tenantId,
                    'company' => [
                        'id' => $companyId,
                        'name' => $data['company_name'],
                        'currency' => 'GHS'
                    ]
                ]
            ], 'Registration successful');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Registration failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * POST /auth/login
     */
    public function login(Request $request): void
    {
        $data = $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        $stmt = $this->db->prepare(
            "SELECT u.id, u.company_id, u.tenant_id, u.email, u.password_hash, u.first_name, u.last_name, 
                    u.role, u.is_active, c.name as company_name, c.currency,
                    t.name as tenant_name, t.is_active as tenant_active
             FROM users u 
             JOIN companies c ON u.company_id = c.id 
             LEFT JOIN tenants t ON u.tenant_id = t.id
             WHERE u.email = :email"
        );
        $stmt->execute(['email' => $data['email']]);
        $user = $stmt->fetch();

        if (!$user || !password_verify($data['password'], $user['password_hash'])) {
            Response::error('Invalid email or password', 401);
            return;
        }

        if (!$user['is_active']) {
            Response::error('Account is deactivated', 403);
            return;
        }

        if (isset($user['tenant_active']) && !$user['tenant_active']) {
            Response::error('Your organization is inactive. Contact support.', 403);
            return;
        }

        // Update last login
        $this->db->prepare("UPDATE users SET last_login = NOW() WHERE id = :id")
            ->execute(['id' => $user['id']]);

        // Generate JWT
        $token = $this->generateToken(
            $user['id'], $user['company_id'], $user['tenant_id'], $user['email'],
            $user['role'], $user['first_name'], $user['last_name']
        );

        // Generate refresh token
        $refreshToken = $this->generateRefreshToken($user['id'], $user['company_id'], $user['tenant_id']);

        $this->auditLog($user['company_id'], $user['id'], 'login', 'user', $user['id']);

        Response::success([
            'token' => $token,
            'refresh_token' => $refreshToken,
            'expires_in' => (int) ($_ENV['JWT_EXPIRY'] ?? 3600),
            'user' => [
                'id' => $user['id'],
                'email' => $user['email'],
                'first_name' => $user['first_name'],
                'last_name' => $user['last_name'],
                'role' => $user['role'],
                'tenant_id' => $user['tenant_id'],
                'company' => [
                    'id' => $user['company_id'],
                    'name' => $user['company_name'],
                    'currency' => $user['currency']
                ]
            ]
        ], 'Login successful');
    }

    /**
     * POST /auth/refresh
     */
    public function refresh(Request $request): void
    {
        $refreshToken = $request->input('refresh_token');
        if (!$refreshToken) {
            Response::error('Refresh token required', 400);
            return;
        }

        try {
            $decoded = JWT::decode($refreshToken, new \Firebase\JWT\Key($_ENV['JWT_SECRET'] . '_refresh', 'HS256'));
            $payload = (array) $decoded;

            // Verify user still active
            $stmt = $this->db->prepare("SELECT id, company_id, tenant_id, email, first_name, last_name, role, is_active FROM users WHERE id = :id");
            $stmt->execute(['id' => $payload['id']]);
            $user = $stmt->fetch();

            if (!$user || !$user['is_active']) {
                Response::unauthorized('Invalid session');
                return;
            }

            $token = $this->generateToken($user['id'], $user['company_id'], $user['tenant_id'], $user['email'], $user['role'], $user['first_name'], $user['last_name']);
            $newRefreshToken = $this->generateRefreshToken($user['id'], $user['company_id'], $user['tenant_id']);

            Response::success([
                'token' => $token,
                'refresh_token' => $newRefreshToken,
                'expires_in' => (int) ($_ENV['JWT_EXPIRY'] ?? 3600)
            ]);
        } catch (\Exception $e) {
            Response::unauthorized('Invalid refresh token');
        }
    }

    /**
     * GET /auth/me
     */
    public function me(Request $request): void
    {
        $userId = $request->getUserId();
        $stmt = $this->db->prepare(
            "SELECT u.id, u.email, u.first_name, u.last_name, u.phone, u.role, u.last_login,
                    u.tenant_id, t.name as tenant_name,
                    c.id as company_id, c.name as company_name, c.tin_number, c.vat_registered,
                    c.currency, c.logo_url, c.digital_address
             FROM users u 
             JOIN companies c ON u.company_id = c.id
             LEFT JOIN tenants t ON u.tenant_id = t.id
             WHERE u.id = :id"
        );
        $stmt->execute(['id' => $userId]);
        $user = $stmt->fetch();

        if (!$user) {
            Response::notFound('User not found');
            return;
        }

        Response::success($user);
    }

    /**
     * POST /auth/change-password
     */
    public function changePassword(Request $request): void
    {
        $data = $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8'
        ]);

        $stmt = $this->db->prepare("SELECT password_hash FROM users WHERE id = :id");
        $stmt->execute(['id' => $request->getUserId()]);
        $user = $stmt->fetch();

        if (!password_verify($data['current_password'], $user['password_hash'])) {
            Response::error('Current password is incorrect', 400);
            return;
        }

        $newHash = password_hash($data['new_password'], PASSWORD_BCRYPT, ['cost' => 12]);
        $this->db->prepare("UPDATE users SET password_hash = :hash, updated_at = NOW() WHERE id = :id")
            ->execute(['hash' => $newHash, 'id' => $request->getUserId()]);

        $this->auditLog($request->getCompanyId(), $request->getUserId(), 'update', 'password', $request->getUserId());

        Response::success(null, 'Password changed successfully');
    }

    private function generateToken(string $userId, string $companyId, ?string $tenantId, string $email, string $role, string $firstName, string $lastName): string
    {
        $payload = [
            'iss' => APP_URL,
            'iat' => time(),
            'exp' => time() + (int) ($_ENV['JWT_EXPIRY'] ?? 3600),
            'id' => $userId,
            'company_id' => $companyId,
            'tenant_id' => $tenantId,
            'email' => $email,
            'role' => $role,
            'first_name' => $firstName,
            'last_name' => $lastName
        ];
        return JWT::encode($payload, $_ENV['JWT_SECRET'], 'HS256');
    }

    private function generateRefreshToken(string $userId, string $companyId, ?string $tenantId): string
    {
        $payload = [
            'iss' => APP_URL,
            'iat' => time(),
            'exp' => time() + (int) ($_ENV['JWT_REFRESH_EXPIRY'] ?? 604800),
            'id' => $userId,
            'company_id' => $companyId,
            'tenant_id' => $tenantId,
            'type' => 'refresh'
        ];
        return JWT::encode($payload, $_ENV['JWT_SECRET'] . '_refresh', 'HS256');
    }

    /**
     * Public wrappers for token generation (used by BusinessController for business switching)
     */
    public function generateTokenPublic(string $userId, string $companyId, ?string $tenantId, string $email, string $role, string $firstName, string $lastName): string
    {
        return $this->generateToken($userId, $companyId, $tenantId, $email, $role, $firstName, $lastName);
    }

    public function generateRefreshTokenPublic(string $userId, string $companyId, ?string $tenantId): string
    {
        return $this->generateRefreshToken($userId, $companyId, $tenantId);
    }
}
