<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class FileAttachmentController extends Controller
{
    private string $uploadDir;
    private array $allowedMimes = [
        'image/jpeg', 'image/png', 'image/gif', 'image/webp',
        'application/pdf',
        'application/msword',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'application/vnd.ms-excel',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'text/csv', 'text/plain',
    ];
    private int $maxSize = 10 * 1024 * 1024; // 10MB

    public function __construct()
    {
        parent::__construct();
        $this->uploadDir = __DIR__ . '/../../public/uploads';
        if (!is_dir($this->uploadDir)) {
            mkdir($this->uploadDir, 0755, true);
        }
    }

    /**
     * POST /api/attachments
     * Upload a file attachment
     */
    public function upload(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $userId = $request->getUserId();

        $entityType = $_POST['entity_type'] ?? '';
        $entityId   = $_POST['entity_id'] ?? '';

        if (!$entityType || !$entityId) {
            Response::error('entity_type and entity_id are required', 422);
            return;
        }

        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            Response::error('No file uploaded or upload error', 422);
            return;
        }

        $file = $_FILES['file'];

        // Validate mime type
        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file['tmp_name']);
        if (!in_array($mimeType, $this->allowedMimes)) {
            Response::error('File type not allowed. Allowed: images, PDF, Word, Excel, CSV, text', 422);
            return;
        }

        // Validate file size
        if ($file['size'] > $this->maxSize) {
            Response::error('File too large. Maximum size is 10MB', 422);
            return;
        }

        // Generate unique filename
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $fileName = uniqid('att_') . '_' . time() . '.' . $ext;

        // Create company-specific subdirectory
        $companyDir = $this->uploadDir . '/' . $companyId;
        if (!is_dir($companyDir)) {
            mkdir($companyDir, 0755, true);
        }

        $filePath = $companyDir . '/' . $fileName;
        if (!move_uploaded_file($file['tmp_name'], $filePath)) {
            Response::error('Failed to save file', 500);
            return;
        }

        // Save to database
        $id = $this->generateId();
        $stmt = $this->db->prepare(
            "INSERT INTO file_attachments (id, company_id, entity_type, entity_id, file_name, original_name, file_path, file_size, mime_type, uploaded_by)
             VALUES (:id, :cid, :etype, :eid, :fname, :oname, :fpath, :fsize, :mime, :uid)
             RETURNING id, entity_type, entity_id, file_name, original_name, file_size, mime_type, created_at"
        );
        $stmt->execute([
            'id'    => $id,
            'cid'   => $companyId,
            'etype' => $entityType,
            'eid'   => $entityId,
            'fname' => $fileName,
            'oname' => $file['name'],
            'fpath' => 'uploads/' . $companyId . '/' . $fileName,
            'mime'  => $mimeType,
            'fsize' => $file['size'],
            'uid'   => $userId,
        ]);

        Response::success($stmt->fetch(), 201);
    }

    /**
     * GET /api/attachments?entity_type=invoice&entity_id=xxx
     * List attachments for an entity
     */
    public function index(Request $request): void
    {
        $companyId  = $request->getCompanyId();
        $entityType = $request->query('entity_type', '');
        $entityId   = $request->query('entity_id', '');

        $sql = "SELECT id, entity_type, entity_id, file_name, original_name, file_size, mime_type, created_at
                FROM file_attachments WHERE company_id = :cid";
        $params = ['cid' => $companyId];

        if ($entityType) {
            $sql .= " AND entity_type = :etype";
            $params['etype'] = $entityType;
        }
        if ($entityId) {
            $sql .= " AND entity_id = :eid";
            $params['eid'] = $entityId;
        }

        $sql .= " ORDER BY created_at DESC";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        Response::success($stmt->fetchAll());
    }

    /**
     * GET /api/attachments/{id}/download
     * Download/serve a file
     */
    public function download(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $id = $request->param('id');

        $stmt = $this->db->prepare(
            "SELECT * FROM file_attachments WHERE id = :id AND company_id = :cid"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $attachment = $stmt->fetch();

        if (!$attachment) {
            Response::error('Attachment not found', 404);
            return;
        }

        $fullPath = __DIR__ . '/../../public/' . $attachment['file_path'];
        if (!file_exists($fullPath)) {
            Response::error('File not found on server', 404);
            return;
        }

        header('Content-Type: ' . $attachment['mime_type']);
        header('Content-Disposition: attachment; filename="' . $attachment['original_name'] . '"');
        header('Content-Length: ' . filesize($fullPath));
        readfile($fullPath);
        exit;
    }

    /**
     * DELETE /api/attachments/{id}
     */
    public function delete(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $id = $request->param('id');

        $stmt = $this->db->prepare(
            "SELECT * FROM file_attachments WHERE id = :id AND company_id = :cid"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $attachment = $stmt->fetch();

        if (!$attachment) {
            Response::error('Attachment not found', 404);
            return;
        }

        // Delete physical file
        $fullPath = __DIR__ . '/../../public/' . $attachment['file_path'];
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }

        // Delete record
        $del = $this->db->prepare("DELETE FROM file_attachments WHERE id = :id");
        $del->execute(['id' => $id]);

        Response::success(['deleted' => true]);
    }
}
