<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class InvoiceController extends Controller
{
    /**
     * GET /invoices
     */
    public function index(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE i.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('type')) {
            $where .= " AND i.invoice_type = :type";
            $params['type'] = $request->query('type');
        }
        if ($request->query('status')) {
            $where .= " AND i.status = :status";
            $params['status'] = $request->query('status');
        }
        if ($request->query('contact_id')) {
            $where .= " AND i.contact_id = :contact_id";
            $params['contact_id'] = $request->query('contact_id');
        }
        if ($request->query('date_from')) {
            $where .= " AND i.invoice_date >= :date_from";
            $params['date_from'] = $request->query('date_from');
        }
        if ($request->query('date_to')) {
            $where .= " AND i.invoice_date <= :date_to";
            $params['date_to'] = $request->query('date_to');
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM invoices i $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT i.*, c.name as contact_name, c.email as contact_email
                FROM invoices i
                JOIN contacts c ON i.contact_id = c.id
                $where ORDER BY i.invoice_date DESC
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $invoices = $stmt->fetchAll();

        Response::paginated($invoices, $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * POST /invoices
     */
    public function create(Request $request): void
    {
        $data = $request->validate([
            'contact_id' => 'required',
            'invoice_type' => 'required|in:sales,purchase',
            'invoice_date' => 'required',
            'due_date' => 'required'
        ]);

        $items = $request->input('items', []);
        if (empty($items)) {
            Response::error('At least one item is required', 422);
            return;
        }

        $companyId = $request->getCompanyId();

        try {
            $this->db->beginTransaction();

            $prefix = $data['invoice_type'] === 'sales' ? 'INV' : 'BILL';
            $invoiceNumber = $this->generateNumber($prefix, 'invoices', 'invoice_number', $companyId);

            // Calculate totals with Ghana tax
            $taxCalc = $this->calculateGhanaTax($companyId, $items, $request->input('discount_type'), $request->input('discount_value'));

            $stmt = $this->db->prepare(
                "INSERT INTO invoices (company_id, contact_id, invoice_number, invoice_type, invoice_date, due_date,
                    reference, subtotal, discount_type, discount_value, discount_amount, tax_amount,
                    nhil_amount, getfund_amount, covid_levy_amount, vat_amount, total_amount, balance_due,
                    notes, terms, currency, status, created_by)
                 VALUES (:company_id, :contact_id, :invoice_number, :invoice_type, :invoice_date, :due_date,
                    :reference, :subtotal, :discount_type, :discount_value, :discount_amount, :tax_amount,
                    :nhil_amount, :getfund_amount, :covid_levy_amount, :vat_amount, :total_amount, :balance_due,
                    :notes, :terms, :currency, :status, :created_by)
                 RETURNING *"
            );
            $stmt->execute([
                'company_id' => $companyId,
                'contact_id' => $data['contact_id'],
                'invoice_number' => $invoiceNumber,
                'invoice_type' => $data['invoice_type'],
                'invoice_date' => $data['invoice_date'],
                'due_date' => $data['due_date'],
                'reference' => $request->input('reference'),
                'subtotal' => $taxCalc['subtotal'],
                'discount_type' => $request->input('discount_type'),
                'discount_value' => $request->input('discount_value', 0),
                'discount_amount' => $taxCalc['discount_amount'],
                'tax_amount' => $taxCalc['total_tax'],
                'nhil_amount' => $taxCalc['nhil'],
                'getfund_amount' => $taxCalc['getfund'],
                'covid_levy_amount' => $taxCalc['covid_levy'],
                'vat_amount' => $taxCalc['vat'],
                'total_amount' => $taxCalc['total'],
                'balance_due' => $taxCalc['total'],
                'notes' => $request->input('notes'),
                'terms' => $request->input('terms'),
                'currency' => $request->input('currency', 'GHS'),
                'status' => $request->input('status', 'draft'),
                'created_by' => $request->getUserId()
            ]);
            $invoice = $stmt->fetch();

            // Insert items
            $itemStmt = $this->db->prepare(
                "INSERT INTO invoice_items (invoice_id, product_id, description, quantity, unit_price, discount_percent, tax_rate, tax_amount, line_total, account_id, sort_order)
                 VALUES (:invoice_id, :product_id, :description, :quantity, :unit_price, :discount_percent, :tax_rate, :tax_amount, :line_total, :account_id, :sort_order)"
            );

            foreach ($items as $idx => $item) {
                $lineTotal = ($item['quantity'] * $item['unit_price']) * (1 - ($item['discount_percent'] ?? 0) / 100);
                $itemStmt->execute([
                    'invoice_id' => $invoice['id'],
                    'product_id' => $item['product_id'] ?? null,
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'discount_percent' => $item['discount_percent'] ?? 0,
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'tax_amount' => $item['tax_amount'] ?? 0,
                    'line_total' => $lineTotal,
                    'account_id' => $item['account_id'] ?? null,
                    'sort_order' => $idx
                ]);
            }

            // Update contact outstanding balance
            $this->db->prepare(
                "UPDATE contacts SET outstanding_balance = outstanding_balance + :amount WHERE id = :id"
            )->execute(['amount' => $taxCalc['total'], 'id' => $data['contact_id']]);

            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'create', 'invoice', $invoice['id'], null, $invoice);
            Response::created($invoice, 'Invoice created successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to create invoice: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /invoices/{id}
     */
    public function show(Request $request): void
    {
        $invoiceId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT i.*, c.name as contact_name, c.email as contact_email, c.phone as contact_phone,
                    c.address as contact_address, c.tin_number as contact_tin,
                    co.name as company_name, co.address as company_address, co.tin_number as company_tin,
                    co.vat_number as company_vat, co.phone as company_phone, co.email as company_email,
                    co.logo_url as company_logo, co.digital_address as company_digital_address
             FROM invoices i
             JOIN contacts c ON i.contact_id = c.id
             JOIN companies co ON i.company_id = co.id
             WHERE i.id = :id AND i.company_id = :company_id"
        );
        $stmt->execute(['id' => $invoiceId, 'company_id' => $companyId]);
        $invoice = $stmt->fetch();

        if (!$invoice) {
            Response::notFound('Invoice not found');
            return;
        }

        // Get items
        $itemStmt = $this->db->prepare(
            "SELECT ii.*, p.name as product_name, p.sku
             FROM invoice_items ii
             LEFT JOIN products p ON ii.product_id = p.id
             WHERE ii.invoice_id = :invoice_id ORDER BY ii.sort_order"
        );
        $itemStmt->execute(['invoice_id' => $invoiceId]);
        $invoice['items'] = $itemStmt->fetchAll();

        // Get payments
        $payStmt = $this->db->prepare(
            "SELECT pa.amount, p.payment_number, p.payment_date, p.payment_method, p.reference
             FROM payment_allocations pa
             JOIN payments p ON pa.payment_id = p.id
             WHERE pa.invoice_id = :invoice_id ORDER BY p.payment_date"
        );
        $payStmt->execute(['invoice_id' => $invoiceId]);
        $invoice['payments'] = $payStmt->fetchAll();

        Response::success($invoice);
    }

    /**
     * PUT /invoices/{id}
     */
    public function update(Request $request): void
    {
        $invoiceId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM invoices WHERE id = :id AND company_id = :cid AND status IN ('draft', 'sent')");
        $stmt->execute(['id' => $invoiceId, 'cid' => $companyId]);
        $invoice = $stmt->fetch();

        if (!$invoice) {
            Response::error('Invoice not found or cannot be edited', 404);
            return;
        }

        $items = $request->input('items', []);
        $taxCalc = $this->calculateGhanaTax($companyId, $items, $request->input('discount_type'), $request->input('discount_value'));

        try {
            $this->db->beginTransaction();

            $this->db->prepare(
                "UPDATE invoices SET contact_id = :contact_id, invoice_date = :invoice_date, due_date = :due_date,
                    reference = :reference, subtotal = :subtotal, discount_type = :discount_type,
                    discount_value = :discount_value, discount_amount = :discount_amount,
                    tax_amount = :tax_amount, nhil_amount = :nhil_amount, getfund_amount = :getfund_amount,
                    covid_levy_amount = :covid_levy_amount, vat_amount = :vat_amount,
                    total_amount = :total_amount, balance_due = :total_amount - amount_paid,
                    notes = :notes, terms = :terms, updated_at = NOW()
                 WHERE id = :id"
            )->execute([
                'id' => $invoiceId,
                'contact_id' => $request->input('contact_id', $invoice['contact_id']),
                'invoice_date' => $request->input('invoice_date', $invoice['invoice_date']),
                'due_date' => $request->input('due_date', $invoice['due_date']),
                'reference' => $request->input('reference'),
                'subtotal' => $taxCalc['subtotal'],
                'discount_type' => $request->input('discount_type'),
                'discount_value' => $request->input('discount_value', 0),
                'discount_amount' => $taxCalc['discount_amount'],
                'tax_amount' => $taxCalc['total_tax'],
                'nhil_amount' => $taxCalc['nhil'],
                'getfund_amount' => $taxCalc['getfund'],
                'covid_levy_amount' => $taxCalc['covid_levy'],
                'vat_amount' => $taxCalc['vat'],
                'total_amount' => $taxCalc['total'],
                'notes' => $request->input('notes'),
                'terms' => $request->input('terms')
            ]);

            // Replace items
            $this->db->prepare("DELETE FROM invoice_items WHERE invoice_id = :id")->execute(['id' => $invoiceId]);

            $itemStmt = $this->db->prepare(
                "INSERT INTO invoice_items (invoice_id, product_id, description, quantity, unit_price, discount_percent, tax_rate, tax_amount, line_total, account_id, sort_order)
                 VALUES (:invoice_id, :product_id, :description, :quantity, :unit_price, :discount_percent, :tax_rate, :tax_amount, :line_total, :account_id, :sort_order)"
            );

            foreach ($items as $idx => $item) {
                $lineTotal = ($item['quantity'] * $item['unit_price']) * (1 - ($item['discount_percent'] ?? 0) / 100);
                $itemStmt->execute([
                    'invoice_id' => $invoiceId,
                    'product_id' => $item['product_id'] ?? null,
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'discount_percent' => $item['discount_percent'] ?? 0,
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'tax_amount' => $item['tax_amount'] ?? 0,
                    'line_total' => $lineTotal,
                    'account_id' => $item['account_id'] ?? null,
                    'sort_order' => $idx
                ]);
            }

            $this->db->commit();
            Response::success(null, 'Invoice updated successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to update invoice: ' . $e->getMessage(), 500);
        }
    }

    /**
     * POST /invoices/{id}/send
     */
    public function send(Request $request): void
    {
        $invoiceId = $request->param('id');
        $companyId = $request->getCompanyId();

        $this->db->prepare("UPDATE invoices SET status = 'sent', updated_at = NOW() WHERE id = :id AND company_id = :cid")
            ->execute(['id' => $invoiceId, 'cid' => $companyId]);

        // TODO: Send email with invoice PDF
        Response::success(null, 'Invoice sent');
    }

    /**
     * POST /payments
     */
    public function recordPayment(Request $request): void
    {
        $data = $request->validate([
            'payment_type' => 'required|in:received,made',
            'payment_date' => 'required',
            'amount' => 'required|numeric',
            'payment_method' => 'required'
        ]);

        $companyId = $request->getCompanyId();
        $invoiceIds = $request->input('invoice_ids', []);

        try {
            $this->db->beginTransaction();

            $paymentNumber = $this->generateNumber('PAY', 'payments', 'payment_number', $companyId);

            $stmt = $this->db->prepare(
                "INSERT INTO payments (company_id, contact_id, payment_number, payment_type, payment_date, amount,
                    payment_method, mobile_money_provider, mobile_money_number, reference, bank_account_id, notes, created_by)
                 VALUES (:company_id, :contact_id, :payment_number, :payment_type, :payment_date, :amount,
                    :payment_method, :mobile_money_provider, :mobile_money_number, :reference, :bank_account_id, :notes, :created_by)
                 RETURNING *"
            );
            $stmt->execute([
                'company_id' => $companyId,
                'contact_id' => $request->input('contact_id'),
                'payment_number' => $paymentNumber,
                'payment_type' => $data['payment_type'],
                'payment_date' => $data['payment_date'],
                'amount' => $data['amount'],
                'payment_method' => $data['payment_method'],
                'mobile_money_provider' => $request->input('mobile_money_provider'),
                'mobile_money_number' => $request->input('mobile_money_number'),
                'reference' => $request->input('reference'),
                'bank_account_id' => $request->input('bank_account_id'),
                'notes' => $request->input('notes'),
                'created_by' => $request->getUserId()
            ]);
            $payment = $stmt->fetch();

            // Allocate to invoices
            $remainingAmount = (float) $data['amount'];
            foreach ($invoiceIds as $invoiceId) {
                if ($remainingAmount <= 0) break;

                $invStmt = $this->db->prepare("SELECT balance_due FROM invoices WHERE id = :id AND company_id = :cid");
                $invStmt->execute(['id' => $invoiceId, 'cid' => $companyId]);
                $inv = $invStmt->fetch();

                if (!$inv) continue;

                $allocAmount = min($remainingAmount, (float) $inv['balance_due']);

                $this->db->prepare(
                    "INSERT INTO payment_allocations (payment_id, invoice_id, amount) VALUES (:pid, :iid, :amount)"
                )->execute(['pid' => $payment['id'], 'iid' => $invoiceId, 'amount' => $allocAmount]);

                // Update invoice
                $this->db->prepare(
                    "UPDATE invoices SET amount_paid = amount_paid + :amount, 
                        balance_due = balance_due - :amount2,
                        status = CASE WHEN balance_due - :amount3 <= 0 THEN 'paid' ELSE 'partially_paid' END,
                        updated_at = NOW()
                     WHERE id = :id"
                )->execute([
                    'amount' => $allocAmount, 'amount2' => $allocAmount, 'amount3' => $allocAmount,
                    'id' => $invoiceId
                ]);

                $remainingAmount -= $allocAmount;
            }

            // Create journal entry for payment
            $this->createPaymentJournalEntry($companyId, $payment, $data['payment_type'], $request->getUserId());

            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'create', 'payment', $payment['id'], null, $payment);
            Response::created($payment, 'Payment recorded successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to record payment: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /payments
     */
    public function getPayments(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE p.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('type')) {
            $where .= " AND p.payment_type = :type";
            $params['type'] = $request->query('type');
        }
        if ($request->query('method')) {
            $where .= " AND p.payment_method = :method";
            $params['method'] = $request->query('method');
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM payments p $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT p.*, c.name as contact_name
                FROM payments p LEFT JOIN contacts c ON p.contact_id = c.id
                $where ORDER BY p.payment_date DESC
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * Calculate Ghana taxes (NHIL, GETFund, VAT)
     */
    private function calculateGhanaTax(string $companyId, array $items, ?string $discountType = null, $discountValue = 0): array
    {
        $subtotal = 0;
        foreach ($items as $item) {
            $lineTotal = ($item['quantity'] * $item['unit_price']) * (1 - ($item['discount_percent'] ?? 0) / 100);
            $subtotal += $lineTotal;
        }

        // Apply overall discount
        $discountAmount = 0;
        if ($discountType === 'percent' && $discountValue > 0) {
            $discountAmount = $subtotal * ($discountValue / 100);
        } elseif ($discountType === 'fixed' && $discountValue > 0) {
            $discountAmount = (float) $discountValue;
        }
        $taxableBase = $subtotal - $discountAmount;

        // Get company's tax registration status
        $compStmt = $this->db->prepare("SELECT vat_registered, nhil_registered, getfund_registered FROM companies WHERE id = :id");
        $compStmt->execute(['id' => $companyId]);
        $company = $compStmt->fetch();

        // Ghana Composite Tax Calculation:
        // Step 1: NHIL (2.5%) and GETFund (2.5%) on the taxable value (before VAT)
        // Step 2: VAT (15%) on (taxable value + NHIL + GETFund)
        $nhil = 0;
        $getfund = 0;
        $vat = 0;

        if ($company['vat_registered']) {
            if ($company['nhil_registered']) {
                $nhil = round($taxableBase * 0.025, 2);
            }
            if ($company['getfund_registered']) {
                $getfund = round($taxableBase * 0.025, 2);
            }

            // VAT is calculated on taxable base + levies
            $vatBase = $taxableBase + $nhil + $getfund;
            $vat = round($vatBase * 0.15, 2);
        }

        $totalTax = $nhil + $getfund + $vat;
        $total = round($taxableBase + $totalTax, 2);

        return [
            'subtotal' => round($subtotal, 2),
            'discount_amount' => round($discountAmount, 2),
            'taxable_base' => round($taxableBase, 2),
            'nhil' => $nhil,
            'getfund' => $getfund,
            'covid_levy' => 0,
            'vat' => $vat,
            'total_tax' => $totalTax,
            'total' => $total
        ];
    }

    private function createPaymentJournalEntry(string $companyId, array $payment, string $type, string $userId): void
    {
        $entryNumber = $this->generateNumber('JE', 'journal_entries', 'entry_number', $companyId);

        // Get bank/cash account
        $bankAccountId = $payment['bank_account_id'];
        if (!$bankAccountId) {
            $cashStmt = $this->db->prepare(
                "SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '1000' LIMIT 1"
            );
            $cashStmt->execute(['cid' => $companyId]);
            $bankAccountId = $cashStmt->fetchColumn();
        }

        // Get AR/AP account
        $arApCode = $type === 'received' ? '1200' : '2000';
        $arApStmt = $this->db->prepare(
            "SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code LIMIT 1"
        );
        $arApStmt->execute(['cid' => $companyId, 'code' => $arApCode]);
        $arApAccountId = $arApStmt->fetchColumn();

        $jeStmt = $this->db->prepare(
            "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, reference, source, source_id, total_debit, total_credit, status, created_by, posted_at)
             VALUES (:cid, :entry_number, :date, :desc, :ref, 'payment', :source_id, :amount, :amount2, 'posted', :user_id, NOW())
             RETURNING id"
        );
        $jeStmt->execute([
            'cid' => $companyId,
            'entry_number' => $entryNumber,
            'date' => $payment['payment_date'],
            'desc' => "Payment {$payment['payment_number']}",
            'ref' => $payment['reference'],
            'source_id' => $payment['id'],
            'amount' => $payment['amount'],
            'amount2' => $payment['amount'],
            'user_id' => $userId
        ]);
        $je = $jeStmt->fetch();

        $lineStmt = $this->db->prepare(
            "INSERT INTO journal_entry_lines (journal_entry_id, account_id, debit_amount, credit_amount, description)
             VALUES (:je_id, :account_id, :debit, :credit, :desc)"
        );

        if ($type === 'received') {
            // Debit Bank, Credit AR
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $bankAccountId, 'debit' => $payment['amount'], 'credit' => 0, 'desc' => 'Payment received']);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $arApAccountId, 'debit' => 0, 'credit' => $payment['amount'], 'desc' => 'Payment received']);
        } else {
            // Debit AP, Credit Bank
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $arApAccountId, 'debit' => $payment['amount'], 'credit' => 0, 'desc' => 'Payment made']);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $bankAccountId, 'debit' => 0, 'credit' => $payment['amount'], 'desc' => 'Payment made']);
        }

        // Update payment with journal entry
        $this->db->prepare("UPDATE payments SET journal_entry_id = :je_id WHERE id = :id")
            ->execute(['je_id' => $je['id'], 'id' => $payment['id']]);
    }

    /**
     * Void an invoice - marks it as voided and reverses accounting entries
     */
    public function voidInvoice(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $id = $request->param('id');

        $stmt = $this->db->prepare(
            "SELECT * FROM invoices WHERE id = :id AND company_id = :cid"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $invoice = $stmt->fetch();

        if (!$invoice) {
            Response::notFound('Invoice not found');
            return;
        }

        if ($invoice['status'] === 'voided') {
            Response::error('Invoice is already voided', 400);
            return;
        }

        if ($invoice['status'] === 'paid') {
            Response::error('Cannot void a paid invoice. Refund payments first.', 400);
            return;
        }

        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare(
                "UPDATE invoices SET status = 'voided', updated_at = NOW()
                 WHERE id = :id AND company_id = :cid RETURNING *"
            );
            $stmt->execute(['id' => $id, 'cid' => $companyId]);
            $voided = $stmt->fetch();

            $this->auditLog($companyId, $request->getUserId(), 'void', 'invoices', $id, $invoice, $voided);

            $this->db->commit();
            Response::success($voided, 'Invoice voided successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to void invoice: ' . $e->getMessage(), 500);
        }
    }
}
