<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class POSController extends Controller
{
    /**
     * POST /pos/sessions/open
     */
    public function openSession(Request $request): void
    {
        $data = $request->validate([
            'terminal_id' => 'required',
            'opening_balance' => 'required|numeric'
        ]);

        $companyId = $request->getCompanyId();

        // Check no open session for this cashier
        $check = $this->db->prepare(
            "SELECT id FROM pos_sessions WHERE cashier_id = :uid AND status = 'open'"
        );
        $check->execute(['uid' => $request->getUserId()]);
        if ($check->fetch()) {
            Response::error('You already have an open POS session. Close it first.', 400);
            return;
        }

        $stmt = $this->db->prepare(
            "INSERT INTO pos_sessions (company_id, terminal_id, cashier_id, opening_balance, status)
             VALUES (:company_id, :terminal_id, :cashier_id, :opening_balance, 'open')
             RETURNING *"
        );
        $stmt->execute([
            'company_id' => $companyId,
            'terminal_id' => $data['terminal_id'],
            'cashier_id' => $request->getUserId(),
            'opening_balance' => $data['opening_balance']
        ]);

        $session = $stmt->fetch();
        $this->auditLog($companyId, $request->getUserId(), 'open', 'pos_session', $session['id']);
        Response::created($session, 'POS session opened');
    }

    /**
     * POST /pos/sessions/{id}/close
     */
    public function closeSession(Request $request): void
    {
        $sessionId = $request->param('id');
        $companyId = $request->getCompanyId();

        $data = $request->validate([
            'closing_balance' => 'required|numeric'
        ]);

        $stmt = $this->db->prepare("SELECT * FROM pos_sessions WHERE id = :id AND company_id = :cid AND status = 'open'");
        $stmt->execute(['id' => $sessionId, 'cid' => $companyId]);
        $session = $stmt->fetch();

        if (!$session) {
            Response::error('Session not found or already closed', 404);
            return;
        }

        // Calculate totals
        $salesStmt = $this->db->prepare(
            "SELECT 
                COALESCE(SUM(total_amount), 0) as total_sales,
                COALESCE(SUM(CASE WHEN payment_method = 'cash' THEN total_amount ELSE 0 END), 0) as total_cash,
                COALESCE(SUM(CASE WHEN payment_method = 'mobile_money' THEN total_amount ELSE 0 END), 0) as total_mobile_money,
                COALESCE(SUM(CASE WHEN payment_method = 'card' THEN total_amount ELSE 0 END), 0) as total_card,
                COALESCE(SUM(CASE WHEN status = 'refunded' THEN total_amount ELSE 0 END), 0) as total_refunds
             FROM pos_sales WHERE session_id = :sid AND status != 'voided'"
        );
        $salesStmt->execute(['sid' => $sessionId]);
        $totals = $salesStmt->fetch();

        $expectedCash = (float) $session['opening_balance'] + (float) $totals['total_cash'] - (float) $totals['total_refunds'];
        $cashDiff = (float) $data['closing_balance'] - $expectedCash;

        $this->db->prepare(
            "UPDATE pos_sessions SET closing_balance = :closing, total_sales = :sales, total_cash = :cash,
                total_mobile_money = :momo, total_card = :card, total_refunds = :refunds,
                cash_difference = :diff, status = 'closed', closed_at = NOW(), notes = :notes
             WHERE id = :id"
        )->execute([
            'closing' => $data['closing_balance'],
            'sales' => $totals['total_sales'],
            'cash' => $totals['total_cash'],
            'momo' => $totals['total_mobile_money'],
            'card' => $totals['total_card'],
            'refunds' => $totals['total_refunds'],
            'diff' => $cashDiff,
            'notes' => $request->input('notes'),
            'id' => $sessionId
        ]);

        $this->auditLog($companyId, $request->getUserId(), 'close', 'pos_session', $sessionId);
        Response::success([
            'session_id' => $sessionId,
            'total_sales' => $totals['total_sales'],
            'total_cash' => $totals['total_cash'],
            'total_mobile_money' => $totals['total_mobile_money'],
            'total_card' => $totals['total_card'],
            'total_refunds' => $totals['total_refunds'],
            'expected_cash' => $expectedCash,
            'closing_balance' => $data['closing_balance'],
            'cash_difference' => $cashDiff
        ], 'POS session closed');
    }

    /**
     * GET /pos/sessions/current
     */
    public function currentSession(Request $request): void
    {
        $stmt = $this->db->prepare(
            "SELECT ps.*, pt.terminal_name, u.first_name || ' ' || u.last_name as cashier_name
             FROM pos_sessions ps
             JOIN pos_terminals pt ON ps.terminal_id = pt.id
             JOIN users u ON ps.cashier_id = u.id
             WHERE ps.cashier_id = :uid AND ps.status = 'open'"
        );
        $stmt->execute(['uid' => $request->getUserId()]);
        $session = $stmt->fetch();

        if (!$session) {
            Response::success(null, 'No open session');
            return;
        }

        // Get today's sales summary
        $salesStmt = $this->db->prepare(
            "SELECT COUNT(*) as sale_count, COALESCE(SUM(total_amount), 0) as total
             FROM pos_sales WHERE session_id = :sid AND status != 'voided'"
        );
        $salesStmt->execute(['sid' => $session['id']]);
        $session['sales_summary'] = $salesStmt->fetch();

        Response::success($session);
    }

    /**
     * POST /pos/sales
     */
    public function createSale(Request $request): void
    {
        $data = $request->validate([
            'session_id' => 'required',
            'payment_method' => 'required|in:cash,mobile_money,card,split',
            'items' => 'required'
        ]);

        $companyId = $request->getCompanyId();
        $items = $request->input('items');

        if (empty($items)) {
            Response::error('At least one item is required', 422);
            return;
        }

        // Verify open session
        $sessStmt = $this->db->prepare("SELECT * FROM pos_sessions WHERE id = :id AND status = 'open'");
        $sessStmt->execute(['id' => $data['session_id']]);
        if (!$sessStmt->fetch()) {
            Response::error('POS session is not open', 400);
            return;
        }

        try {
            $this->db->beginTransaction();

            $saleNumber = $this->generateNumber('POS', 'pos_sales', 'sale_number', $companyId);

            // Calculate totals with Ghana tax
            $subtotal = 0;
            $discountTotal = 0;

            foreach ($items as $item) {
                $lineTotal = $item['quantity'] * $item['unit_price'];
                $discount = $lineTotal * (($item['discount_percent'] ?? 0) / 100);
                $subtotal += $lineTotal;
                $discountTotal += $discount;
            }

            $taxableBase = $subtotal - $discountTotal;

            // Check if company is VAT registered
            $compStmt = $this->db->prepare("SELECT vat_registered FROM companies WHERE id = :id");
            $compStmt->execute(['id' => $companyId]);
            $company = $compStmt->fetch();

            $nhil = 0; $getfund = 0; $vat = 0;
            if ($company['vat_registered']) {
                $nhil = round($taxableBase * 0.025, 2);
                $getfund = round($taxableBase * 0.025, 2);
                $vatBase = $taxableBase + $nhil + $getfund;
                $vat = round($vatBase * 0.15, 2);
            }

            $totalTax = $nhil + $getfund + $vat;
            $totalAmount = round($taxableBase + $totalTax, 2);
            $amountTendered = (float) ($request->input('amount_tendered') ?? $totalAmount);
            $changeGiven = max(0, $amountTendered - $totalAmount);

            $stmt = $this->db->prepare(
                "INSERT INTO pos_sales (company_id, session_id, sale_number, customer_id, customer_name, subtotal,
                    discount_amount, tax_amount, nhil_amount, getfund_amount, covid_levy_amount, vat_amount,
                    total_amount, amount_tendered, change_given, payment_method, mobile_money_provider,
                    mobile_money_number, notes, created_by)
                 VALUES (:company_id, :session_id, :sale_number, :customer_id, :customer_name, :subtotal,
                    :discount_amount, :tax_amount, :nhil_amount, :getfund_amount, :covid_levy_amount, :vat_amount,
                    :total_amount, :amount_tendered, :change_given, :payment_method, :mobile_money_provider,
                    :mobile_money_number, :notes, :created_by)
                 RETURNING *"
            );
            $stmt->execute([
                'company_id' => $companyId,
                'session_id' => $data['session_id'],
                'sale_number' => $saleNumber,
                'customer_id' => $request->input('customer_id'),
                'customer_name' => $request->input('customer_name'),
                'subtotal' => $subtotal,
                'discount_amount' => $discountTotal,
                'tax_amount' => $totalTax,
                'nhil_amount' => $nhil,
                'getfund_amount' => $getfund,
                'covid_levy_amount' => 0,
                'vat_amount' => $vat,
                'total_amount' => $totalAmount,
                'amount_tendered' => $amountTendered,
                'change_given' => $changeGiven,
                'payment_method' => $data['payment_method'],
                'mobile_money_provider' => $request->input('mobile_money_provider'),
                'mobile_money_number' => $request->input('mobile_money_number'),
                'notes' => $request->input('notes'),
                'created_by' => $request->getUserId()
            ]);
            $sale = $stmt->fetch();

            // Insert sale items and update inventory
            $itemStmt = $this->db->prepare(
                "INSERT INTO pos_sale_items (sale_id, product_id, product_name, quantity, unit_price, cost_price, discount_percent, discount_amount, tax_amount, line_total)
                 VALUES (:sale_id, :product_id, :product_name, :quantity, :unit_price, :cost_price, :discount_percent, :discount_amount, :tax_amount, :line_total)"
            );

            foreach ($items as $item) {
                // Get product details
                $prodStmt = $this->db->prepare("SELECT name, cost_price, track_inventory FROM products WHERE id = :id");
                $prodStmt->execute(['id' => $item['product_id']]);
                $product = $prodStmt->fetch();

                $lineTotal = $item['quantity'] * $item['unit_price'];
                $discountAmt = $lineTotal * (($item['discount_percent'] ?? 0) / 100);
                $netLine = $lineTotal - $discountAmt;

                $itemStmt->execute([
                    'sale_id' => $sale['id'],
                    'product_id' => $item['product_id'],
                    'product_name' => $product['name'] ?? $item['product_name'] ?? '',
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'cost_price' => $product['cost_price'] ?? 0,
                    'discount_percent' => $item['discount_percent'] ?? 0,
                    'discount_amount' => $discountAmt,
                    'tax_amount' => 0,
                    'line_total' => $netLine
                ]);

                // Update inventory
                if ($product && $product['track_inventory']) {
                    $this->db->prepare(
                        "UPDATE products SET quantity_on_hand = quantity_on_hand - :qty, updated_at = NOW() WHERE id = :id"
                    )->execute(['qty' => $item['quantity'], 'id' => $item['product_id']]);

                    // Record inventory movement
                    $this->db->prepare(
                        "INSERT INTO inventory_movements (company_id, product_id, movement_type, quantity, unit_cost, total_cost, reference_type, reference_id, created_by)
                         VALUES (:cid, :pid, 'sale', :qty, :cost, :total, 'pos_sale', :ref_id, :uid)"
                    )->execute([
                        'cid' => $companyId,
                        'pid' => $item['product_id'],
                        'qty' => -$item['quantity'],
                        'cost' => $product['cost_price'] ?? 0,
                        'total' => ($product['cost_price'] ?? 0) * $item['quantity'],
                        'ref_id' => $sale['id'],
                        'uid' => $request->getUserId()
                    ]);
                }
            }

            // Handle split payments
            if ($data['payment_method'] === 'split') {
                $splitPayments = $request->input('split_payments', []);
                foreach ($splitPayments as $sp) {
                    $this->db->prepare(
                        "INSERT INTO pos_split_payments (sale_id, payment_method, amount, reference, mobile_money_provider, mobile_money_number)
                         VALUES (:sale_id, :method, :amount, :ref, :provider, :number)"
                    )->execute([
                        'sale_id' => $sale['id'],
                        'method' => $sp['payment_method'],
                        'amount' => $sp['amount'],
                        'ref' => $sp['reference'] ?? null,
                        'provider' => $sp['mobile_money_provider'] ?? null,
                        'number' => $sp['mobile_money_number'] ?? null
                    ]);
                }
            }

            // Create journal entry for POS sale
            $this->createPOSSaleJournalEntry($companyId, $sale, $request->getUserId());

            $this->db->commit();

            // Return sale with items for receipt printing
            $sale['items'] = $items;
            $sale['change_given'] = $changeGiven;

            Response::created($sale, 'Sale completed');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Sale failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /pos/sales
     */
    public function getSales(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE ps.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('session_id')) {
            $where .= " AND ps.session_id = :session_id";
            $params['session_id'] = $request->query('session_id');
        }
        if ($request->query('date_from')) {
            $where .= " AND ps.sale_date >= :date_from";
            $params['date_from'] = $request->query('date_from');
        }
        if ($request->query('date_to')) {
            $where .= " AND ps.sale_date <= :date_to";
            $params['date_to'] = $request->query('date_to');
        }
        if ($request->query('payment_method')) {
            $where .= " AND ps.payment_method = :method";
            $params['method'] = $request->query('payment_method');
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM pos_sales ps $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT ps.*, u.first_name || ' ' || u.last_name as cashier_name
                FROM pos_sales ps LEFT JOIN users u ON ps.created_by = u.id
                $where ORDER BY ps.sale_date DESC
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * GET /pos/sales/{id}
     */
    public function getSale(Request $request): void
    {
        $saleId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT ps.*, u.first_name || ' ' || u.last_name as cashier_name,
                    co.name as company_name, co.phone as company_phone, co.address as company_address,
                    co.tin_number as company_tin, co.vat_number as company_vat
             FROM pos_sales ps
             LEFT JOIN users u ON ps.created_by = u.id
             JOIN companies co ON ps.company_id = co.id
             WHERE ps.id = :id AND ps.company_id = :cid"
        );
        $stmt->execute(['id' => $saleId, 'cid' => $companyId]);
        $sale = $stmt->fetch();

        if (!$sale) {
            Response::notFound('Sale not found');
            return;
        }

        // Get items
        $itemStmt = $this->db->prepare("SELECT * FROM pos_sale_items WHERE sale_id = :id");
        $itemStmt->execute(['id' => $saleId]);
        $sale['items'] = $itemStmt->fetchAll();

        // Get split payments
        if ($sale['payment_method'] === 'split') {
            $spStmt = $this->db->prepare("SELECT * FROM pos_split_payments WHERE sale_id = :id");
            $spStmt->execute(['id' => $saleId]);
            $sale['split_payments'] = $spStmt->fetchAll();
        }

        Response::success($sale);
    }

    /**
     * POST /pos/sales/{id}/refund
     */
    public function refundSale(Request $request): void
    {
        $saleId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM pos_sales WHERE id = :id AND company_id = :cid AND status IN ('completed', 'partially_refunded')");
        $stmt->execute(['id' => $saleId, 'cid' => $companyId]);
        $sale = $stmt->fetch();

        if (!$sale) {
            Response::error('Sale not found or already fully refunded', 404);
            return;
        }

        $reason = $request->input('reason', 'No reason provided');
        $returnItems = $request->input('items', []);

        // Get all sale items
        $itemStmt = $this->db->prepare("SELECT * FROM pos_sale_items WHERE sale_id = :id");
        $itemStmt->execute(['id' => $saleId]);
        $saleItems = $itemStmt->fetchAll();

        // Build map of sale items
        $saleItemMap = [];
        foreach ($saleItems as $si) {
            $saleItemMap[$si['id']] = $si;
        }

        // Get previously returned quantities
        $prevReturns = $this->db->prepare(
            "SELECT sale_item_id, COALESCE(SUM(quantity_returned), 0) as already_returned
             FROM pos_returns WHERE sale_id = :sid GROUP BY sale_item_id"
        );
        $prevReturns->execute(['sid' => $saleId]);
        $returnedMap = [];
        foreach ($prevReturns->fetchAll() as $pr) {
            $returnedMap[$pr['sale_item_id']] = (float) $pr['already_returned'];
        }

        try {
            $this->db->beginTransaction();

            $isFullRefund = empty($returnItems);
            $totalRefundAmount = 0;

            if ($isFullRefund) {
                // Full refund: return all items not yet returned
                $itemsToReturn = [];
                foreach ($saleItems as $item) {
                    $alreadyReturned = $returnedMap[$item['id']] ?? 0;
                    $remainingQty = (float) $item['quantity'] - $alreadyReturned;
                    if ($remainingQty > 0) {
                        $itemsToReturn[] = [
                            'sale_item_id' => $item['id'],
                            'product_id' => $item['product_id'],
                            'product_name' => $item['product_name'],
                            'quantity' => $remainingQty,
                            'unit_price' => (float) $item['unit_price'],
                            'line_total' => $remainingQty * (float) $item['unit_price'],
                        ];
                    }
                }
            } else {
                // Partial refund: validate items
                $itemsToReturn = [];
                foreach ($returnItems as $ri) {
                    $saleItemId = $ri['sale_item_id'] ?? null;
                    $qty = (float) ($ri['quantity'] ?? 0);

                    if (!$saleItemId || $qty <= 0) continue;

                    if (!isset($saleItemMap[$saleItemId])) {
                        $this->db->rollBack();
                        Response::error("Sale item not found: $saleItemId", 422);
                        return;
                    }

                    $saleItem = $saleItemMap[$saleItemId];
                    $alreadyReturned = $returnedMap[$saleItemId] ?? 0;
                    $maxReturnable = (float) $saleItem['quantity'] - $alreadyReturned;

                    if ($qty > $maxReturnable) {
                        $this->db->rollBack();
                        Response::error("Cannot return {$qty} of {$saleItem['product_name']}. Max returnable: {$maxReturnable}", 422);
                        return;
                    }

                    $itemsToReturn[] = [
                        'sale_item_id' => $saleItemId,
                        'product_id' => $saleItem['product_id'],
                        'product_name' => $saleItem['product_name'],
                        'quantity' => $qty,
                        'unit_price' => (float) $saleItem['unit_price'],
                        'line_total' => $qty * (float) $saleItem['unit_price'],
                    ];
                }
            }

            if (empty($itemsToReturn)) {
                $this->db->rollBack();
                Response::error('No items to return', 422);
                return;
            }

            // Generate return number
            $returnNumber = $this->generateNumber('RET', 'pos_returns', 'return_number', $companyId);

            // Calculate refund amount with proportional tax
            $refundSubtotal = 0;
            foreach ($itemsToReturn as $item) {
                $refundSubtotal += $item['line_total'];
            }

            // Calculate proportional tax on refunded amount
            $saleSubtotal = (float) $sale['subtotal'];
            $proportion = $saleSubtotal > 0 ? $refundSubtotal / $saleSubtotal : 0;
            $refundNhil = round((float) $sale['nhil_amount'] * $proportion, 2);
            $refundGetfund = round((float) $sale['getfund_amount'] * $proportion, 2);
            $refundVat = round((float) $sale['vat_amount'] * $proportion, 2);
            $refundTax = $refundNhil + $refundGetfund + $refundVat;
            $totalRefundAmount = $refundSubtotal + $refundTax;

            // Insert return records and restore inventory
            foreach ($itemsToReturn as $item) {
                $this->db->prepare(
                    "INSERT INTO pos_returns (company_id, sale_id, sale_item_id, return_number, product_id, product_name,
                        quantity_returned, unit_price, refund_amount, reason, created_by)
                     VALUES (:cid, :sid, :siid, :rn, :pid, :pname, :qty, :price, :refund, :reason, :uid)"
                )->execute([
                    'cid' => $companyId,
                    'sid' => $saleId,
                    'siid' => $item['sale_item_id'],
                    'rn' => $returnNumber,
                    'pid' => $item['product_id'],
                    'pname' => $item['product_name'],
                    'qty' => $item['quantity'],
                    'price' => $item['unit_price'],
                    'refund' => $item['line_total'],
                    'reason' => $reason,
                    'uid' => $request->getUserId()
                ]);

                // Restore inventory
                $this->db->prepare(
                    "UPDATE products SET quantity_on_hand = quantity_on_hand + :qty WHERE id = :id"
                )->execute(['qty' => $item['quantity'], 'id' => $item['product_id']]);

                // Record inventory movement
                $this->db->prepare(
                    "INSERT INTO inventory_movements (company_id, product_id, movement_type, quantity, reference_type, reference_id, notes, created_by)
                     VALUES (:cid, :pid, 'return', :qty, 'pos_return', :ref, :notes, :uid)"
                )->execute([
                    'cid' => $companyId,
                    'pid' => $item['product_id'],
                    'qty' => $item['quantity'],
                    'ref' => $saleId,
                    'notes' => "POS Return: {$item['product_name']} x{$item['quantity']} - {$reason}",
                    'uid' => $request->getUserId()
                ]);
            }

            // Check if all items are now fully returned
            $allReturned = true;
            foreach ($saleItems as $si) {
                $prevRet = $returnedMap[$si['id']] ?? 0;
                $newlyReturned = 0;
                foreach ($itemsToReturn as $itr) {
                    if ($itr['sale_item_id'] === $si['id']) {
                        $newlyReturned = $itr['quantity'];
                        break;
                    }
                }
                if (($prevRet + $newlyReturned) < (float) $si['quantity']) {
                    $allReturned = false;
                    break;
                }
            }

            $newStatus = $allReturned ? 'refunded' : 'partially_refunded';

            // Update sale status and totals
            $this->db->prepare(
                "UPDATE pos_sales SET status = :status, 
                    notes = COALESCE(notes, '') || E'\nReturn ({$returnNumber}): {$reason}'
                 WHERE id = :id"
            )->execute(['status' => $newStatus, 'id' => $saleId]);

            // Create reversal journal entry for refund
            $this->createRefundJournalEntry($companyId, $sale, $totalRefundAmount, $refundNhil, $refundGetfund, $refundVat, $returnNumber, $request->getUserId());

            $this->db->commit();
            $this->auditLog($companyId, $request->getUserId(), 'refund', 'pos_sale', $saleId);

            Response::success([
                'return_number' => $returnNumber,
                'items_returned' => $itemsToReturn,
                'refund_subtotal' => $refundSubtotal,
                'refund_tax' => $refundTax,
                'total_refund' => $totalRefundAmount,
                'sale_status' => $newStatus,
            ], 'Return processed successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Refund failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /pos/returns
     */
    public function getReturns(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE pr.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('sale_id')) {
            $where .= " AND pr.sale_id = :sale_id";
            $params['sale_id'] = $request->query('sale_id');
        }
        if ($request->query('date_from')) {
            $where .= " AND pr.created_at >= :date_from";
            $params['date_from'] = $request->query('date_from');
        }
        if ($request->query('date_to')) {
            $where .= " AND pr.created_at <= :date_to || ' 23:59:59'";
            $params['date_to'] = $request->query('date_to');
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM pos_returns pr $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT pr.*, ps.sale_number, u.first_name || ' ' || u.last_name as processed_by_name
                FROM pos_returns pr
                LEFT JOIN pos_sales ps ON pr.sale_id = ps.id
                LEFT JOIN users u ON pr.created_by = u.id
                $where ORDER BY pr.created_at DESC
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * GET /pos/terminals
     */
    public function getTerminals(Request $request): void
    {
        $stmt = $this->db->prepare("SELECT * FROM pos_terminals WHERE company_id = :cid AND is_active = TRUE ORDER BY terminal_name");
        $stmt->execute(['cid' => $request->getCompanyId()]);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /pos/terminals
     */
    public function createTerminal(Request $request): void
    {
        $data = $request->validate(['terminal_name' => 'required']);

        $stmt = $this->db->prepare(
            "INSERT INTO pos_terminals (company_id, terminal_name, location) VALUES (:cid, :name, :location) RETURNING *"
        );
        $stmt->execute([
            'cid' => $request->getCompanyId(),
            'name' => $data['terminal_name'],
            'location' => $request->input('location')
        ]);
        Response::created($stmt->fetch());
    }

    /**
     * GET /pos/daily-report
     */
    public function dailyReport(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $date = $request->query('date', date('Y-m-d'));

        $stmt = $this->db->prepare(
            "SELECT 
                COUNT(*) as total_transactions,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN total_amount ELSE 0 END), 0) as total_sales,
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'cash' THEN total_amount ELSE 0 END), 0) as cash_sales,
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'mobile_money' THEN total_amount ELSE 0 END), 0) as momo_sales,
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'card' THEN total_amount ELSE 0 END), 0) as card_sales,
                COALESCE(SUM(CASE WHEN status = 'refunded' THEN total_amount ELSE 0 END), 0) as refunds,
                COALESCE(SUM(discount_amount), 0) as total_discounts,
                COALESCE(SUM(vat_amount), 0) as total_vat,
                COALESCE(SUM(nhil_amount), 0) as total_nhil,
                COALESCE(SUM(getfund_amount), 0) as total_getfund
             FROM pos_sales WHERE company_id = :cid AND DATE(sale_date) = :date"
        );
        $stmt->execute(['cid' => $companyId, 'date' => $date]);
        $report = $stmt->fetch();

        // Top products
        $topStmt = $this->db->prepare(
            "SELECT psi.product_name, SUM(psi.quantity) as qty_sold, SUM(psi.line_total) as revenue
             FROM pos_sale_items psi
             JOIN pos_sales ps ON psi.sale_id = ps.id
             WHERE ps.company_id = :cid AND DATE(ps.sale_date) = :date AND ps.status = 'completed'
             GROUP BY psi.product_name ORDER BY revenue DESC LIMIT 10"
        );
        $topStmt->execute(['cid' => $companyId, 'date' => $date]);
        $report['top_products'] = $topStmt->fetchAll();

        Response::success($report);
    }

    /**
     * Create reversal journal entry for POS refund
     */
    private function createRefundJournalEntry(string $companyId, array $sale, float $totalRefund, float $nhil, float $getfund, float $vat, string $returnNumber, string $userId): void
    {
        $entryNumber = $this->generateNumber('JE', 'journal_entries', 'entry_number', $companyId);

        $cashAccountCode = match ($sale['payment_method']) {
            'mobile_money' => '1800',
            'card' => '1100',
            default => '1000'
        };

        $cashStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code");
        $cashStmt->execute(['cid' => $companyId, 'code' => $cashAccountCode]);
        $cashAccountId = $cashStmt->fetchColumn();

        $salesStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '4010'");
        $salesStmt->execute(['cid' => $companyId]);
        $salesAccountId = $salesStmt->fetchColumn();

        if (!$cashAccountId || !$salesAccountId) return;

        $revenueRefund = $totalRefund - ($nhil + $getfund + $vat);

        $jeStmt = $this->db->prepare(
            "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, source, source_id, total_debit, total_credit, status, created_by, posted_at)
             VALUES (:cid, :num, CURRENT_DATE, :desc, 'pos_refund', :sid, :amount, :amount2, 'posted', :uid, NOW())
             RETURNING id"
        );
        $jeStmt->execute([
            'cid' => $companyId, 'num' => $entryNumber,
            'desc' => "POS Refund {$returnNumber}",
            'sid' => $sale['id'],
            'amount' => $totalRefund, 'amount2' => $totalRefund,
            'uid' => $userId
        ]);
        $je = $jeStmt->fetch();

        $lineStmt = $this->db->prepare(
            "INSERT INTO journal_entry_lines (journal_entry_id, account_id, debit_amount, credit_amount, description)
             VALUES (:je_id, :account_id, :debit, :credit, :desc)"
        );

        // Reverse: Debit Sales Revenue, Credit Cash/Bank
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $salesAccountId, 'debit' => $revenueRefund, 'credit' => 0, 'desc' => "POS Refund {$returnNumber}"]);
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $cashAccountId, 'debit' => 0, 'credit' => $totalRefund, 'desc' => "POS Refund {$returnNumber}"]);

        // Reverse tax entries
        if ($vat > 0) {
            $vatStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '2100'");
            $vatStmt->execute(['cid' => $companyId]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $vatStmt->fetchColumn(), 'debit' => $vat, 'credit' => 0, 'desc' => 'VAT Reversal']);
        }
        if ($nhil > 0) {
            $nhilStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '2110'");
            $nhilStmt->execute(['cid' => $companyId]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $nhilStmt->fetchColumn(), 'debit' => $nhil, 'credit' => 0, 'desc' => 'NHIL Reversal']);
        }
        if ($getfund > 0) {
            $gfStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '2120'");
            $gfStmt->execute(['cid' => $companyId]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $gfStmt->fetchColumn(), 'debit' => $getfund, 'credit' => 0, 'desc' => 'GETFund Reversal']);
        }
    }

    private function createPOSSaleJournalEntry(string $companyId, array $sale, string $userId): void
    {

        $entryNumber = $this->generateNumber('JE', 'journal_entries', 'entry_number', $companyId);

        // Determine debit account (cash, bank, momo)
        $cashAccountCode = match ($sale['payment_method']) {
            'mobile_money' => '1800',
            'card' => '1100',
            default => '1000'
        };

        $cashStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code");
        $cashStmt->execute(['cid' => $companyId, 'code' => $cashAccountCode]);
        $cashAccountId = $cashStmt->fetchColumn();

        $salesStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '4010'");
        $salesStmt->execute(['cid' => $companyId]);
        $salesAccountId = $salesStmt->fetchColumn();

        $jeStmt = $this->db->prepare(
            "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, source, source_id, total_debit, total_credit, status, created_by, posted_at)
             VALUES (:cid, :num, CURRENT_DATE, :desc, 'pos', :sid, :amount, :amount2, 'posted', :uid, NOW())
             RETURNING id"
        );
        $jeStmt->execute([
            'cid' => $companyId, 'num' => $entryNumber,
            'desc' => "POS Sale {$sale['sale_number']}",
            'sid' => $sale['id'],
            'amount' => $sale['total_amount'], 'amount2' => $sale['total_amount'],
            'uid' => $userId
        ]);
        $je = $jeStmt->fetch();

        $lineStmt = $this->db->prepare(
            "INSERT INTO journal_entry_lines (journal_entry_id, account_id, debit_amount, credit_amount, description)
             VALUES (:je_id, :account_id, :debit, :credit, :desc)"
        );

        // Debit Cash/Bank
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $cashAccountId, 'debit' => $sale['total_amount'], 'credit' => 0, 'desc' => 'POS Sale']);
        // Credit Sales Revenue
        $saleRevenue = $sale['total_amount'] - $sale['tax_amount'];
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $salesAccountId, 'debit' => 0, 'credit' => $saleRevenue, 'desc' => 'POS Sale Revenue']);

        // Credit Tax accounts if applicable
        if ($sale['vat_amount'] > 0) {
            $vatStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '2100'");
            $vatStmt->execute(['cid' => $companyId]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $vatStmt->fetchColumn(), 'debit' => 0, 'credit' => $sale['vat_amount'], 'desc' => 'VAT Output']);
        }
        if ($sale['nhil_amount'] > 0) {
            $nhilStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '2110'");
            $nhilStmt->execute(['cid' => $companyId]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $nhilStmt->fetchColumn(), 'debit' => 0, 'credit' => $sale['nhil_amount'], 'desc' => 'NHIL']);
        }
        if ($sale['getfund_amount'] > 0) {
            $gfStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = '2120'");
            $gfStmt->execute(['cid' => $companyId]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $gfStmt->fetchColumn(), 'debit' => 0, 'credit' => $sale['getfund_amount'], 'desc' => 'GETFund Levy']);
        }

        // Update sale with journal entry
        $this->db->prepare("UPDATE pos_sales SET journal_entry_id = :je WHERE id = :id")->execute(['je' => $je['id'], 'id' => $sale['id']]);
    }
}
