<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class TaxController extends Controller
{
    /**
     * GET /tax/rates
     */
    public function getRates(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $stmt = $this->db->prepare(
            "SELECT * FROM tax_rates WHERE (company_id = :cid OR company_id IS NULL) AND is_active = TRUE ORDER BY tax_type, tax_name"
        );
        $stmt->execute(['cid' => $companyId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /tax/calculate
     * Calculate Ghana composite tax
     */
    public function calculateTax(Request $request): void
    {
        $amount = (float) $request->input('amount', 0);
        $scheme = $request->input('scheme', 'standard'); // standard or flat_rate
        $includeNHIL = (bool) $request->input('include_nhil', true);
        $includeGETFund = (bool) $request->input('include_getfund', true);

        if ($scheme === 'flat_rate') {
            // VAT Flat Rate Scheme - 4% on turnover
            $vat = round($amount * 0.04, 2);
            Response::success([
                'scheme' => 'flat_rate',
                'taxable_amount' => $amount,
                'vat_rate' => '4%',
                'vat_amount' => $vat,
                'total' => round($amount + $vat, 2)
            ]);
            return;
        }

        // Standard VAT scheme
        // Step 1: NHIL (2.5%) and GETFund (2.5%) on taxable value
        $nhil = $includeNHIL ? round($amount * 0.025, 2) : 0;
        $getfund = $includeGETFund ? round($amount * 0.025, 2) : 0;

        // VAT (15%) on (taxable value + NHIL + GETFund) — compound
        $vatBase = $amount + $nhil + $getfund;
        $vat = round($vatBase * 0.15, 2);

        $totalTax = $nhil + $getfund + $vat;
        $grandTotal = round($amount + $totalTax, 2);

        Response::success([
            'scheme' => 'standard',
            'taxable_amount' => $amount,
            'nhil' => ['rate' => '2.5%', 'amount' => $nhil],
            'getfund' => ['rate' => '2.5%', 'amount' => $getfund],
            'vat' => ['rate' => '15%', 'base' => $vatBase, 'amount' => $vat],
            'total_tax' => $totalTax,
            'effective_tax_rate' => $amount > 0 ? round(($totalTax / $amount) * 100, 2) . '%' : '0%',
            'grand_total' => $grandTotal
        ]);
    }

    /**
     * POST /tax/calculate-paye
     */
    public function calculatePAYE(Request $request): void
    {
        $monthlyIncome = (float) $request->input('monthly_income', 0);
        $ssnitRate = (float) $request->input('ssnit_rate', 5.5);

        $ssnitDeduction = round($monthlyIncome * ($ssnitRate / 100), 2);
        $taxableIncome = $monthlyIncome - $ssnitDeduction;

        // Ghana PAYE Monthly Tax Brackets
        $brackets = [
            ['limit' => 490,       'rate' => 0,    'label' => 'First GHS 490'],
            ['limit' => 110,       'rate' => 5,    'label' => 'Next GHS 110'],
            ['limit' => 130,       'rate' => 10,   'label' => 'Next GHS 130'],
            ['limit' => 3166.67,   'rate' => 17.5, 'label' => 'Next GHS 3,166.67'],
            ['limit' => 16103.33,  'rate' => 25,   'label' => 'Next GHS 16,103.33'],
            ['limit' => 30000,     'rate' => 30,   'label' => 'Next GHS 30,000'],
            ['limit' => PHP_FLOAT_MAX, 'rate' => 35, 'label' => 'Exceeding GHS 50,000'],
        ];

        $tax = 0;
        $remaining = $taxableIncome;
        $breakdown = [];

        foreach ($brackets as $bracket) {
            if ($remaining <= 0) break;
            $taxableInBracket = min($remaining, $bracket['limit']);
            $taxInBracket = round($taxableInBracket * ($bracket['rate'] / 100), 2);
            $tax += $taxInBracket;
            $remaining -= $taxableInBracket;

            if ($taxableInBracket > 0) {
                $breakdown[] = [
                    'bracket' => $bracket['label'],
                    'rate' => $bracket['rate'] . '%',
                    'taxable_amount' => $taxableInBracket,
                    'tax' => $taxInBracket
                ];
            }
        }

        Response::success([
            'monthly_income' => $monthlyIncome,
            'ssnit_deduction' => $ssnitDeduction,
            'taxable_income' => round($taxableIncome, 2),
            'paye_tax' => round($tax, 2),
            'net_income' => round($monthlyIncome - $ssnitDeduction - $tax, 2),
            'effective_rate' => $monthlyIncome > 0 ? round(($tax / $monthlyIncome) * 100, 2) . '%' : '0%',
            'breakdown' => $breakdown
        ]);
    }

    /**
     * GET /tax/filings
     */
    public function getFilings(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $stmt = $this->db->prepare(
            "SELECT * FROM tax_filings WHERE company_id = :cid ORDER BY period_start DESC"
        );
        $stmt->execute(['cid' => $companyId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /tax/filings
     */
    public function createFiling(Request $request): void
    {
        $data = $request->validate([
            'tax_type' => 'required|in:vat,paye,withholding',
            'period_start' => 'required',
            'period_end' => 'required'
        ]);

        $companyId = $request->getCompanyId();

        $filingPeriod = date('Y-m', strtotime($data['period_start']));

        // Auto-calculate based on type
        $outputTax = 0;
        $inputTax = 0;

        if ($data['tax_type'] === 'vat') {
            // Output VAT from invoices + POS
            $outStmt = $this->db->prepare(
                "SELECT COALESCE(SUM(vat_amount + nhil_amount + getfund_amount), 0)
                 FROM invoices WHERE company_id = :cid AND invoice_type = 'sales'
                 AND invoice_date BETWEEN :from AND :to AND status != 'voided'"
            );
            $outStmt->execute(['cid' => $companyId, 'from' => $data['period_start'], 'to' => $data['period_end']]);
            $outputTax = (float) $outStmt->fetchColumn();

            $posStmt = $this->db->prepare(
                "SELECT COALESCE(SUM(vat_amount + nhil_amount + getfund_amount), 0)
                 FROM pos_sales WHERE company_id = :cid AND DATE(sale_date) BETWEEN :from AND :to AND status = 'completed'"
            );
            $posStmt->execute(['cid' => $companyId, 'from' => $data['period_start'], 'to' => $data['period_end']]);
            $outputTax += (float) $posStmt->fetchColumn();

            // Input VAT from purchase invoices
            $inStmt = $this->db->prepare(
                "SELECT COALESCE(SUM(vat_amount), 0)
                 FROM invoices WHERE company_id = :cid AND invoice_type = 'purchase'
                 AND invoice_date BETWEEN :from AND :to AND status != 'voided'"
            );
            $inStmt->execute(['cid' => $companyId, 'from' => $data['period_start'], 'to' => $data['period_end']]);
            $inputTax = (float) $inStmt->fetchColumn();
        } elseif ($data['tax_type'] === 'paye') {
            $payeStmt = $this->db->prepare(
                "SELECT COALESCE(SUM(ps.paye_tax), 0)
                 FROM payslips ps JOIN payroll_periods pp ON ps.payroll_period_id = pp.id
                 WHERE pp.company_id = :cid AND pp.start_date >= :from AND pp.end_date <= :to"
            );
            $payeStmt->execute(['cid' => $companyId, 'from' => $data['period_start'], 'to' => $data['period_end']]);
            $outputTax = (float) $payeStmt->fetchColumn();
        }

        $taxPayable = $outputTax - $inputTax;

        $stmt = $this->db->prepare(
            "INSERT INTO tax_filings (company_id, tax_type, filing_period, period_start, period_end, output_tax, input_tax, tax_payable, status, created_by)
             VALUES (:cid, :type, :period, :start, :end, :output, :input, :payable, 'draft', :uid)
             RETURNING *"
        );
        $stmt->execute([
            'cid' => $companyId,
            'type' => $data['tax_type'],
            'period' => $filingPeriod,
            'start' => $data['period_start'],
            'end' => $data['period_end'],
            'output' => round($outputTax, 2),
            'input' => round($inputTax, 2),
            'payable' => round($taxPayable, 2),
            'uid' => $request->getUserId()
        ]);

        Response::created($stmt->fetch(), 'Tax filing created');
    }

    /**
     * GET /tax/withholding-summary
     */
    public function withholdingSummary(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $dateFrom = $request->query('date_from', date('Y-01-01'));
        $dateTo = $request->query('date_to', date('Y-m-d'));

        $stmt = $this->db->prepare(
            "SELECT c.name as supplier_name, c.tin_number,
                    SUM(i.subtotal) as total_amount,
                    SUM(i.tax_amount) as withholding_tax
             FROM invoices i JOIN contacts c ON i.contact_id = c.id
             WHERE i.company_id = :cid AND i.invoice_type = 'purchase'
               AND i.invoice_date BETWEEN :from AND :to AND i.status != 'voided'
             GROUP BY c.id, c.name, c.tin_number ORDER BY c.name"
        );
        $stmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        Response::success($stmt->fetchAll());
    }
}
