<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class TransactionController extends Controller
{
    /**
     * GET /transactions
     */
    public function index(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE t.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('type')) {
            $where .= " AND t.transaction_type = :type";
            $params['type'] = $request->query('type');
        }
        if ($request->query('category')) {
            $where .= " AND t.category = :category";
            $params['category'] = $request->query('category');
        }
        if ($request->query('payment_method')) {
            $where .= " AND t.payment_method = :method";
            $params['method'] = $request->query('payment_method');
        }
        if ($request->query('date_from')) {
            $where .= " AND t.transaction_date >= :date_from";
            $params['date_from'] = $request->query('date_from');
        }
        if ($request->query('date_to')) {
            $where .= " AND t.transaction_date <= :date_to";
            $params['date_to'] = $request->query('date_to');
        }
        if ($request->query('search')) {
            $where .= " AND (t.description ILIKE :search OR t.reference ILIKE :search2 OR t.transaction_number ILIKE :search3)";
            $params['search'] = '%' . $request->query('search') . '%';
            $params['search2'] = '%' . $request->query('search') . '%';
            $params['search3'] = '%' . $request->query('search') . '%';
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM transactions t $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT t.*, u.first_name || ' ' || u.last_name as created_by_name,
                       c.name as contact_name, a.account_name, a.account_code
                FROM transactions t
                LEFT JOIN users u ON t.created_by = u.id
                LEFT JOIN contacts c ON t.contact_id = c.id
                LEFT JOIN chart_of_accounts a ON t.account_id = a.id
                $where ORDER BY t.transaction_date DESC, t.created_at DESC
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * POST /transactions
     */
    public function create(Request $request): void
    {
        $data = $request->validate([
            'transaction_type' => 'required|in:income,expense',
            'transaction_date' => 'required',
            'amount' => 'required|numeric',
            'description' => 'required',
            'payment_method' => 'required'
        ]);

        $companyId = $request->getCompanyId();
        $amount = abs((float) $data['amount']);

        if ($amount <= 0) {
            Response::error('Amount must be greater than zero', 422);
            return;
        }

        try {
            $this->db->beginTransaction();

            $txnNumber = $this->generateNumber('TXN', 'transactions', 'transaction_number', $companyId);

            $stmt = $this->db->prepare(
                "INSERT INTO transactions (company_id, transaction_number, transaction_type, transaction_date,
                    amount, description, category, payment_method, mobile_money_provider, mobile_money_number,
                    reference, contact_id, account_id, notes, created_by)
                 VALUES (:company_id, :txn_number, :type, :date, :amount, :description, :category,
                    :method, :momo_provider, :momo_number, :reference, :contact_id, :account_id, :notes, :uid)
                 RETURNING *"
            );
            $stmt->execute([
                'company_id' => $companyId,
                'txn_number' => $txnNumber,
                'type' => $data['transaction_type'],
                'date' => $data['transaction_date'],
                'amount' => $amount,
                'description' => $data['description'],
                'category' => $request->input('category'),
                'method' => $data['payment_method'],
                'momo_provider' => $request->input('mobile_money_provider'),
                'momo_number' => $request->input('mobile_money_number'),
                'reference' => $request->input('reference'),
                'contact_id' => $request->input('contact_id'),
                'account_id' => $request->input('account_id'),
                'notes' => $request->input('notes'),
                'uid' => $request->getUserId()
            ]);
            $txn = $stmt->fetch();

            // Create journal entry
            $this->createTransactionJournalEntry($companyId, $txn, $request->getUserId());

            // Re-fetch to include journal_entry_id
            $refetch = $this->db->prepare("SELECT * FROM transactions WHERE id = :id");
            $refetch->execute(['id' => $txn['id']]);
            $txn = $refetch->fetch();

            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'create', 'transaction', $txn['id'], null, $txn);
            Response::created($txn, 'Transaction recorded successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to record transaction: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /transactions/{id}
     */
    public function show(Request $request): void
    {
        $id = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT t.*, u.first_name || ' ' || u.last_name as created_by_name,
                    c.name as contact_name, a.account_name, a.account_code
             FROM transactions t
             LEFT JOIN users u ON t.created_by = u.id
             LEFT JOIN contacts c ON t.contact_id = c.id
             LEFT JOIN chart_of_accounts a ON t.account_id = a.id
             WHERE t.id = :id AND t.company_id = :cid"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $txn = $stmt->fetch();

        if (!$txn) {
            Response::notFound('Transaction not found');
            return;
        }

        Response::success($txn);
    }

    /**
     * DELETE /transactions/{id}
     */
    public function void(Request $request): void
    {
        $id = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM transactions WHERE id = :id AND company_id = :cid AND status != 'voided'");
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $txn = $stmt->fetch();

        if (!$txn) {
            Response::error('Transaction not found or already voided', 404);
            return;
        }

        try {
            $this->db->beginTransaction();

            $this->db->prepare("UPDATE transactions SET status = 'voided', updated_at = NOW() WHERE id = :id")
                ->execute(['id' => $id]);

            // Void the journal entry if exists
            if ($txn['journal_entry_id']) {
                $this->db->prepare("UPDATE journal_entries SET status = 'voided' WHERE id = :id")
                    ->execute(['id' => $txn['journal_entry_id']]);
            }

            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'void', 'transaction', $id, $txn, ['status' => 'voided']);
            Response::success(null, 'Transaction voided successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to void transaction: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /transactions/summary
     */
    public function summary(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $dateFrom = $request->query('date_from', date('Y-m-01'));
        $dateTo = $request->query('date_to', date('Y-m-d'));

        $stmt = $this->db->prepare(
            "SELECT 
                COALESCE(SUM(CASE WHEN transaction_type = 'income' AND status = 'completed' THEN amount ELSE 0 END), 0) as total_income,
                COALESCE(SUM(CASE WHEN transaction_type = 'expense' AND status = 'completed' THEN amount ELSE 0 END), 0) as total_expenses,
                COUNT(CASE WHEN transaction_type = 'income' AND status = 'completed' THEN 1 END) as income_count,
                COUNT(CASE WHEN transaction_type = 'expense' AND status = 'completed' THEN 1 END) as expense_count,
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'cash' AND transaction_type = 'income' THEN amount ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'cash' AND transaction_type = 'expense' THEN amount ELSE 0 END), 0) as cash_balance,
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'mobile_money' AND transaction_type = 'income' THEN amount ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'mobile_money' AND transaction_type = 'expense' THEN amount ELSE 0 END), 0) as momo_balance,
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'bank_transfer' AND transaction_type = 'income' THEN amount ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN status = 'completed' AND payment_method = 'bank_transfer' AND transaction_type = 'expense' THEN amount ELSE 0 END), 0) as bank_balance
             FROM transactions
             WHERE company_id = :cid AND transaction_date >= :from AND transaction_date <= :to"
        );
        $stmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $summary = $stmt->fetch();

        // Daily breakdown for chart
        $dailyStmt = $this->db->prepare(
            "SELECT transaction_date::date as date, transaction_type,
                    COALESCE(SUM(amount), 0) as total
             FROM transactions
             WHERE company_id = :cid AND transaction_date >= :from AND transaction_date <= :to AND status = 'completed'
             GROUP BY transaction_date::date, transaction_type
             ORDER BY date"
        );
        $dailyStmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $summary['daily'] = $dailyStmt->fetchAll();

        // Category breakdown
        $catStmt = $this->db->prepare(
            "SELECT COALESCE(category, 'Uncategorized') as category, transaction_type,
                    COALESCE(SUM(amount), 0) as total, COUNT(*) as count
             FROM transactions
             WHERE company_id = :cid AND transaction_date >= :from AND transaction_date <= :to AND status = 'completed'
             GROUP BY category, transaction_type
             ORDER BY total DESC"
        );
        $catStmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $summary['categories'] = $catStmt->fetchAll();

        Response::success($summary);
    }

    /**
     * Create journal entry for transaction
     */
    private function createTransactionJournalEntry(string $companyId, array $txn, string $userId): void
    {
        $entryNumber = $this->generateNumber('JE', 'journal_entries', 'entry_number', $companyId);

        // Determine accounts based on payment method
        $cashAccountCode = match ($txn['payment_method']) {
            'mobile_money' => '1800',
            'bank_transfer' => '1100',
            'card' => '1100',
            default => '1000'
        };

        // If an account_id is specified, use that; otherwise use default income/expense account
        if ($txn['account_id']) {
            $targetAccountId = $txn['account_id'];
        } else {
            $defaultCode = $txn['transaction_type'] === 'income' ? '4300' : '5900';
            $acctStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code");
            $acctStmt->execute(['cid' => $companyId, 'code' => $defaultCode]);
            $targetAccountId = $acctStmt->fetchColumn();
        }

        $cashAcctStmt = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code");
        $cashAcctStmt->execute(['cid' => $companyId, 'code' => $cashAccountCode]);
        $cashAccountId = $cashAcctStmt->fetchColumn();

        if (!$cashAccountId || !$targetAccountId) {
            return; // Skip journal entry if accounts not found
        }

        $jeStmt = $this->db->prepare(
            "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, source, source_id, total_debit, total_credit, status, created_by, posted_at)
             VALUES (:cid, :num, :date, :desc, 'transaction', :sid, :amount, :amount2, 'posted', :uid, NOW())
             RETURNING id"
        );
        $jeStmt->execute([
            'cid' => $companyId,
            'num' => $entryNumber,
            'date' => $txn['transaction_date'],
            'desc' => $txn['description'],
            'sid' => $txn['id'],
            'amount' => $txn['amount'],
            'amount2' => $txn['amount'],
            'uid' => $userId
        ]);
        $je = $jeStmt->fetch();

        $lineStmt = $this->db->prepare(
            "INSERT INTO journal_entry_lines (journal_entry_id, account_id, debit_amount, credit_amount, description)
             VALUES (:je_id, :account_id, :debit, :credit, :desc)"
        );

        if ($txn['transaction_type'] === 'income') {
            // Income: Debit Cash/Bank, Credit Revenue
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $cashAccountId, 'debit' => $txn['amount'], 'credit' => 0, 'desc' => $txn['description']]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $targetAccountId, 'debit' => 0, 'credit' => $txn['amount'], 'desc' => $txn['description']]);
        } else {
            // Expense: Debit Expense, Credit Cash/Bank
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $targetAccountId, 'debit' => $txn['amount'], 'credit' => 0, 'desc' => $txn['description']]);
            $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $cashAccountId, 'debit' => 0, 'credit' => $txn['amount'], 'desc' => $txn['description']]);
        }

        // Update transaction with journal entry ID
        $this->db->prepare("UPDATE transactions SET journal_entry_id = :je WHERE id = :id")
            ->execute(['je' => $je['id'], 'id' => $txn['id']]);
    }
}
