<?php

namespace CashBook\Core;
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../Util/UUID.php';

use CashBook\Util\UUID;

class Controller
{
    protected \PDO $db;

    public function __construct()
    {
        $database = new \Database();
        $this->db = $database->getConnection();
    }

    protected function generateId(): string
    {
        // Try PostgreSQL UUID function first (modern PostgreSQL 13+)
        try {
            $stmt = $this->db->query("SELECT gen_random_uuid() as id");
            $result = $stmt->fetch();
            if ($result && !empty($result['id'])) {
                return $result['id'];
            }
        } catch (\Exception $e) {
            // Fall back to PHP UUID generation (for cPanel hosts without extensions)
        }
        
        // Fall back to PHP UUID generation
        return UUID::v4Secure();
    }

    protected function generateNumber(string $prefix, string $table, string $column, string $companyId): string
    {
        $year = date('Y');
        $stmt = $this->db->prepare(
            "SELECT COUNT(*) + 1 as next_num FROM {$table} WHERE company_id = :company_id AND {$column} LIKE :prefix"
        );
        $likePrefix = $prefix . '-' . $year . '-%';
        $stmt->execute(['company_id' => $companyId, 'prefix' => $likePrefix]);
        $result = $stmt->fetch();
        $nextNum = str_pad($result['next_num'], 5, '0', STR_PAD_LEFT);
        return "{$prefix}-{$year}-{$nextNum}";
    }

    protected function paginate(Request $request): array
    {
        $page = max(1, (int) ($request->query('page') ?? 1));
        $perPage = min(100, max(1, (int) ($request->query('per_page') ?? 20)));
        $offset = ($page - 1) * $perPage;
        return ['page' => $page, 'per_page' => $perPage, 'offset' => $offset];
    }

    protected function auditLog(string $companyId, string $userId, string $action, string $entityType, ?string $entityId = null, $oldValues = null, $newValues = null): void
    {
        $stmt = $this->db->prepare(
            "INSERT INTO audit_log (company_id, user_id, action, entity_type, entity_id, old_values, new_values, ip_address) 
             VALUES (:company_id, :user_id, :action, :entity_type, :entity_id, :old_values, :new_values, :ip_address)"
        );
        $stmt->execute([
            'company_id' => $companyId,
            'user_id' => $userId,
            'action' => $action,
            'entity_type' => $entityType,
            'entity_id' => $entityId,
            'old_values' => $oldValues ? json_encode($oldValues) : null,
            'new_values' => $newValues ? json_encode($newValues) : null,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null
        ]);
    }
}
