<?php

namespace CashBook\Core;

class Request
{
    private string $method;
    private string $uri;
    private array $body;
    private array $headers;
    private array $query;
    private array $params = [];
    private ?array $user = null;
    private ?array $tenant = null;
    private ?string $businessRole = null;

    public function __construct(string $method, string $uri, array $body = [], array $headers = [], array $query = [])
    {
        $this->method = $method;
        $this->uri = $uri;
        $this->body = $body;
        $this->headers = $headers;
        $this->query = $query;
    }

    public function getMethod(): string
    {
        return $this->method;
    }

    public function getUri(): string
    {
        return $this->uri;
    }

    public function getBody(): array
    {
        return $this->body;
    }

    public function input(string $key, $default = null)
    {
        return $this->body[$key] ?? $default;
    }

    public function getHeaders(): array
    {
        return $this->headers;
    }

    public function getHeader(string $name): ?string
    {
        // Headers are case-insensitive
        foreach ($this->headers as $key => $value) {
            if (strtolower($key) === strtolower($name)) {
                return $value;
            }
        }
        return null;
    }

    public function getBearerToken(): ?string
    {
        $auth = $this->getHeader('Authorization');
        if ($auth && str_starts_with($auth, 'Bearer ')) {
            return substr($auth, 7);
        }
        return null;
    }

    public function getQuery(): array
    {
        return $this->query;
    }

    public function query(string $key, $default = null)
    {
        return $this->query[$key] ?? $default;
    }

    public function getParams(): array
    {
        return $this->params;
    }

    public function param(string $key, $default = null)
    {
        return $this->params[$key] ?? $default;
    }

    public function setParams(array $params): void
    {
        $this->params = $params;
    }

    public function getUser(): ?array
    {
        return $this->user;
    }

    public function setUser(array $user): void
    {
        $this->user = $user;
    }

    public function getUserId(): ?string
    {
        return $this->user['id'] ?? null;
    }

    public function getCompanyId(): ?string
    {
        return $this->user['company_id'] ?? null;
    }

    public function getTenantId(): ?string
    {
        return $this->user['tenant_id'] ?? null;
    }

    public function getTenant(): ?array
    {
        return $this->tenant;
    }

    public function setTenant(array $tenant): void
    {
        $this->tenant = $tenant;
    }

    public function getBusinessRole(): ?string
    {
        return $this->businessRole;
    }

    public function setBusinessRole(string $role): void
    {
        $this->businessRole = $role;
    }

    public function validate(array $rules): array
    {
        $errors = [];
        $validated = [];

        foreach ($rules as $field => $ruleString) {
            $fieldRules = explode('|', $ruleString);
            $value = $this->body[$field] ?? null;

            foreach ($fieldRules as $rule) {
                if ($rule === 'required' && (is_null($value) || $value === '')) {
                    $errors[$field][] = "$field is required";
                }
                if ($rule === 'email' && $value && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                    $errors[$field][] = "$field must be a valid email";
                }
                if ($rule === 'numeric' && $value && !is_numeric($value)) {
                    $errors[$field][] = "$field must be numeric";
                }
                if (str_starts_with($rule, 'min:')) {
                    $min = (int) substr($rule, 4);
                    if ($value && strlen($value) < $min) {
                        $errors[$field][] = "$field must be at least $min characters";
                    }
                }
                if (str_starts_with($rule, 'max:')) {
                    $max = (int) substr($rule, 4);
                    if ($value && strlen($value) > $max) {
                        $errors[$field][] = "$field must not exceed $max characters";
                    }
                }
                if (str_starts_with($rule, 'in:')) {
                    $options = explode(',', substr($rule, 3));
                    if ($value && !in_array($value, $options)) {
                        $errors[$field][] = "$field must be one of: " . implode(', ', $options);
                    }
                }
            }

            if (!isset($errors[$field]) && $value !== null) {
                $validated[$field] = $value;
            }
        }

        if (!empty($errors)) {
            Response::json(['success' => false, 'errors' => $errors], 422);
            exit;
        }

        return $validated;
    }
}
