<?php

namespace CashBook\Middleware;

use CashBook\Core\Request;
use CashBook\Core\Response;

require_once __DIR__ . '/../../config/database.php';

/**
 * Ensures the authenticated user belongs to an active tenant
 * and optionally verifies they have access to the requested business (company).
 */
class TenantMiddleware
{
    public function handle(Request $request): bool
    {
        $user = $request->getUser();

        if (!$user || empty($user['tenant_id'])) {
            Response::error('Tenant context required', 403);
            exit;
        }

        // Verify tenant is active
        $database = new \Database();
        $db = $database->getConnection();

        $stmt = $db->prepare("SELECT id, is_active, max_businesses, max_users FROM tenants WHERE id = :id");
        $stmt->execute(['id' => $user['tenant_id']]);
        $tenant = $stmt->fetch();

        if (!$tenant || !$tenant['is_active']) {
            Response::error('Tenant is inactive or not found', 403);
            exit;
        }

        // Store tenant info on the request
        $request->setTenant($tenant);

        // Verify user has access to the current company/business
        $companyId = $user['company_id'] ?? null;
        if ($companyId) {
            $stmt = $db->prepare(
                "SELECT ub.role FROM user_businesses ub 
                 WHERE ub.user_id = :user_id AND ub.company_id = :company_id"
            );
            $stmt->execute([
                'user_id' => $user['id'],
                'company_id' => $companyId
            ]);
            $businessAccess = $stmt->fetch();

            // Admins always have access to all businesses within their tenant
            if (!$businessAccess && $user['role'] !== 'admin' && $user['role'] !== 'super_admin') {
                Response::error('You do not have access to this business', 403);
                exit;
            }

            // Store the business-level role  
            if ($businessAccess) {
                $request->setBusinessRole($businessAccess['role']);
            }
        }

        return true;
    }
}
