'use client';

import React, { useState, useEffect } from 'react';
import { accountingAPI } from '@/lib/api';
import { formatCurrency, formatDate } from '@/lib/utils';
import { PageLoading, StatusBadge, Modal, EmptyState } from '@/components/ui';
import DatePicker from '@/components/ui/DatePicker';
import toast from 'react-hot-toast';
import { PlusIcon, BookOpenIcon } from '@heroicons/react/24/outline';

export default function AccountingPage() {
  const [tab, setTab] = useState<'coa' | 'journal' | 'trial' | 'ledger'>('coa');
  const [accounts, setAccounts] = useState<any[]>([]);
  const [entries, setEntries] = useState<any[]>([]);
  const [trialBalance, setTrialBalance] = useState<any[]>([]);
  const [trialBalanceMeta, setTrialBalanceMeta] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [showJEModal, setShowJEModal] = useState(false);

  // Account form
  const [accForm, setAccForm] = useState({
    account_code: '', account_name: '', account_type: 'asset', parent_id: '', description: '', is_header: false,
  });

  // Journal entry form
  const [jeForm, setJeForm] = useState({
    entry_date: new Date().toISOString().split('T')[0],
    description: '',
    reference: '',
    lines: [
      { account_id: '', description: '', debit: '', credit: '' },
      { account_id: '', description: '', debit: '', credit: '' },
    ],
  });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    setLoading(true);
    try {
      const [accRes, jeRes, tbRes] = await Promise.all([
        accountingAPI.getAccounts(),
        accountingAPI.getJournalEntries(),
        accountingAPI.getTrialBalance(),
      ]);
      setAccounts(accRes.data?.data || []);
      setEntries(jeRes.data?.data || []);
      const tbData = tbRes.data?.data;
      setTrialBalance(tbData?.accounts || []);
      setTrialBalanceMeta(tbData || null);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  const createAccount = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await accountingAPI.createAccount(accForm);
      toast.success('Account created');
      setShowModal(false);
      setAccForm({ account_code: '', account_name: '', account_type: 'asset', parent_id: '', description: '', is_header: false });
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error creating account');
    }
  };

  const createJournalEntry = async (e: React.FormEvent) => {
    e.preventDefault();
    const lines = jeForm.lines.filter((l) => l.account_id);
    const totalDebit = lines.reduce((s, l) => s + parseFloat(l.debit || '0'), 0);
    const totalCredit = lines.reduce((s, l) => s + parseFloat(l.credit || '0'), 0);

    if (Math.abs(totalDebit - totalCredit) > 0.01) {
      toast.error(`Debits (${totalDebit.toFixed(2)}) must equal Credits (${totalCredit.toFixed(2)})`);
      return;
    }

    try {
      await accountingAPI.createJournalEntry({ ...jeForm, lines });
      toast.success('Journal entry created');
      setShowJEModal(false);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const postEntry = async (id: string) => {
    try {
      await accountingAPI.postJournalEntry(id);
      toast.success('Entry posted');
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error posting');
    }
  };

  const addJELine = () => {
    setJeForm({
      ...jeForm,
      lines: [...jeForm.lines, { account_id: '', description: '', debit: '', credit: '' }],
    });
  };

  if (loading) return <PageLoading />;

  const tabs = [
    { key: 'coa', label: 'Chart of Accounts' },
    { key: 'journal', label: 'Journal Entries' },
    { key: 'trial', label: 'Trial Balance' },
    { key: 'ledger', label: 'General Ledger' },
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Accounting</h1>
        <div className="flex gap-2">
          <button className="btn-secondary" onClick={() => setShowModal(true)}>
            <PlusIcon className="h-4 w-4 mr-1" /> New Account
          </button>
          <button className="btn-primary" onClick={() => setShowJEModal(true)}>
            <PlusIcon className="h-4 w-4 mr-1" /> Journal Entry
          </button>
        </div>
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="flex gap-6">
          {tabs.map((t) => (
            <button
              key={t.key}
              className={`pb-3 text-sm font-medium border-b-2 transition-colors ${
                tab === t.key ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
              onClick={() => setTab(t.key as any)}
            >
              {t.label}
            </button>
          ))}
        </nav>
      </div>

      {/* Chart of Accounts */}
      {tab === 'coa' && (
        <div className="card">
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Code</th>
                  <th>Account Name</th>
                  <th>Type</th>
                  <th className="text-right">Balance</th>
                </tr>
              </thead>
              <tbody>
                {accounts.length === 0 ? (
                  <tr><td colSpan={4}><EmptyState title="No accounts yet" icon={<BookOpenIcon />} /></td></tr>
                ) : (
                  accounts.map((acc: any) => (
                    <tr key={acc.id} className={acc.is_header ? 'font-semibold bg-gray-50' : ''}>
                      <td className="font-mono text-xs">{acc.account_code}</td>
                      <td>{acc.account_name}</td>
                      <td className="capitalize">{acc.account_type?.replace('_', ' ')}</td>
                      <td className="text-right">{formatCurrency(acc.current_balance || 0)}</td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Journal Entries */}
      {tab === 'journal' && (
        <div className="card">
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Number</th>
                  <th>Date</th>
                  <th>Description</th>
                  <th>Reference</th>
                  <th>Status</th>
                  <th className="text-right">Amount</th>
                  <th></th>
                </tr>
              </thead>
              <tbody>
                {entries.length === 0 ? (
                  <tr><td colSpan={7}><EmptyState title="No journal entries" /></td></tr>
                ) : (
                  entries.map((je: any) => (
                    <tr key={je.id}>
                      <td className="font-mono text-xs">{je.entry_number}</td>
                      <td>{formatDate(je.entry_date)}</td>
                      <td>{je.description}</td>
                      <td>{je.reference || '-'}</td>
                      <td><StatusBadge status={je.status} /></td>
                      <td className="text-right">{formatCurrency(je.total_debit || 0)}</td>
                      <td>
                        {je.status === 'draft' && (
                          <button className="btn-primary btn-sm" onClick={() => postEntry(je.id)}>Post</button>
                        )}
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Trial Balance */}
      {tab === 'trial' && (
        <div className="card">
          {trialBalanceMeta && (
            <div className="px-5 py-3 border-b border-gray-200 flex items-center justify-between">
              <div className="text-sm text-gray-600">
                As of <span className="font-medium text-gray-900">{formatDate(trialBalanceMeta.as_of)}</span>
              </div>
              <div className={`text-xs font-medium px-2.5 py-1 rounded-full ${trialBalanceMeta.is_balanced ? 'bg-emerald-50 text-emerald-700' : 'bg-red-50 text-red-700'}`}>
                {trialBalanceMeta.is_balanced ? '✓ Balanced' : '✗ Not Balanced'}
              </div>
            </div>
          )}
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Code</th>
                  <th>Account</th>
                  <th>Type</th>
                  <th className="text-right">Debit</th>
                  <th className="text-right">Credit</th>
                  <th className="text-right">Balance</th>
                </tr>
              </thead>
              <tbody>
                {trialBalance.length === 0 ? (
                  <tr><td colSpan={6}><EmptyState title="No trial balance data" description="Post journal entries to see trial balance" icon={<BookOpenIcon className="h-12 w-12" />} /></td></tr>
                ) : (
                  trialBalance.map((tb: any) => (
                    <tr key={tb.account_code}>
                      <td className="font-mono text-xs">{tb.account_code}</td>
                      <td>{tb.account_name}</td>
                      <td className="capitalize text-xs text-gray-500">{tb.account_type?.replace('_', ' ')}</td>
                      <td className="text-right">{parseFloat(tb.total_debit) > 0 ? formatCurrency(tb.total_debit) : '-'}</td>
                      <td className="text-right">{parseFloat(tb.total_credit) > 0 ? formatCurrency(tb.total_credit) : '-'}</td>
                      <td className="text-right font-medium">{formatCurrency(tb.balance)}</td>
                    </tr>
                  ))
                )}
              </tbody>
              <tfoot>
                <tr className="font-bold bg-gray-50">
                  <td colSpan={3}>Total</td>
                  <td className="text-right">
                    {formatCurrency(trialBalance.reduce((s: number, t: any) => s + parseFloat(t.total_debit || 0), 0))}
                  </td>
                  <td className="text-right">
                    {formatCurrency(trialBalance.reduce((s: number, t: any) => s + parseFloat(t.total_credit || 0), 0))}
                  </td>
                  <td></td>
                </tr>
              </tfoot>
            </table>
          </div>
        </div>
      )}

      {/* General Ledger placeholder */}
      {tab === 'ledger' && (
        <div className="card card-body">
          <p className="text-sm text-gray-500">Select an account from the Chart of Accounts to view its general ledger.</p>
        </div>
      )}

      {/* New Account Modal */}
      <Modal open={showModal} onClose={() => setShowModal(false)} title="New Account">
        <form onSubmit={createAccount} className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Account Code *</label>
              <input className="input" value={accForm.account_code} onChange={(e) => setAccForm({ ...accForm, account_code: e.target.value })} required />
            </div>
            <div>
              <label className="label">Type *</label>
              <select className="select" value={accForm.account_type} onChange={(e) => setAccForm({ ...accForm, account_type: e.target.value })}>
                <option value="asset">Asset</option>
                <option value="liability">Liability</option>
                <option value="equity">Equity</option>
                <option value="revenue">Revenue</option>
                <option value="expense">Expense</option>
              </select>
            </div>
          </div>
          <div>
            <label className="label">Account Name *</label>
            <input className="input" value={accForm.account_name} onChange={(e) => setAccForm({ ...accForm, account_name: e.target.value })} required />
          </div>
          <div>
            <label className="label">Description</label>
            <input className="input" value={accForm.description} onChange={(e) => setAccForm({ ...accForm, description: e.target.value })} />
          </div>
          <div className="flex items-center gap-2">
            <input type="checkbox" id="is_header" checked={accForm.is_header} onChange={(e) => setAccForm({ ...accForm, is_header: e.target.checked })} />
            <label htmlFor="is_header" className="text-sm text-gray-700">This is a header/group account</label>
          </div>
          <div className="flex justify-end gap-3">
            <button type="button" className="btn-secondary" onClick={() => setShowModal(false)}>Cancel</button>
            <button type="submit" className="btn-primary">Create Account</button>
          </div>
        </form>
      </Modal>

      {/* Journal Entry Modal */}
      <Modal open={showJEModal} onClose={() => setShowJEModal(false)} title="New Journal Entry" size="xl">
        <form onSubmit={createJournalEntry} className="space-y-4">
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">Date *</label>
              <DatePicker value={jeForm.entry_date} onChange={(v) => setJeForm({ ...jeForm, entry_date: v })} required />
            </div>
            <div>
              <label className="label">Reference</label>
              <input className="input" value={jeForm.reference} onChange={(e) => setJeForm({ ...jeForm, reference: e.target.value })} />
            </div>
            <div>
              <label className="label">Description *</label>
              <input className="input" value={jeForm.description} onChange={(e) => setJeForm({ ...jeForm, description: e.target.value })} required />
            </div>
          </div>

          <div>
            <div className="flex items-center justify-between mb-2">
              <h4 className="text-sm font-medium text-gray-700">Lines</h4>
              <button type="button" className="btn-secondary btn-sm" onClick={addJELine}>+ Add Line</button>
            </div>
            <table className="table">
              <thead>
                <tr>
                  <th>Account</th>
                  <th>Description</th>
                  <th>Debit (GHS)</th>
                  <th>Credit (GHS)</th>
                </tr>
              </thead>
              <tbody>
                {jeForm.lines.map((line, i) => (
                  <tr key={i}>
                    <td>
                      <select
                        className="select text-xs"
                        value={line.account_id}
                        onChange={(e) => {
                          const lines = [...jeForm.lines];
                          lines[i].account_id = e.target.value;
                          setJeForm({ ...jeForm, lines });
                        }}
                      >
                        <option value="">Select account</option>
                        {accounts.filter((a: any) => !a.is_header).map((a: any) => (
                          <option key={a.id} value={a.id}>{a.account_code} - {a.account_name}</option>
                        ))}
                      </select>
                    </td>
                    <td>
                      <input
                        className="input text-xs"
                        value={line.description}
                        onChange={(e) => {
                          const lines = [...jeForm.lines];
                          lines[i].description = e.target.value;
                          setJeForm({ ...jeForm, lines });
                        }}
                      />
                    </td>
                    <td>
                      <input
                        type="number"
                        step="0.01"
                        className="input text-xs text-right"
                        value={line.debit}
                        onChange={(e) => {
                          const lines = [...jeForm.lines];
                          lines[i].debit = e.target.value;
                          if (e.target.value) lines[i].credit = '';
                          setJeForm({ ...jeForm, lines });
                        }}
                      />
                    </td>
                    <td>
                      <input
                        type="number"
                        step="0.01"
                        className="input text-xs text-right"
                        value={line.credit}
                        onChange={(e) => {
                          const lines = [...jeForm.lines];
                          lines[i].credit = e.target.value;
                          if (e.target.value) lines[i].debit = '';
                          setJeForm({ ...jeForm, lines });
                        }}
                      />
                    </td>
                  </tr>
                ))}
              </tbody>
              <tfoot>
                <tr className="font-bold">
                  <td colSpan={2} className="text-right">Totals</td>
                  <td className="text-right">{jeForm.lines.reduce((s, l) => s + parseFloat(l.debit || '0'), 0).toFixed(2)}</td>
                  <td className="text-right">{jeForm.lines.reduce((s, l) => s + parseFloat(l.credit || '0'), 0).toFixed(2)}</td>
                </tr>
              </tfoot>
            </table>
          </div>

          <div className="flex justify-end gap-3">
            <button type="button" className="btn-secondary" onClick={() => setShowJEModal(false)}>Cancel</button>
            <button type="submit" className="btn-primary">Create Entry</button>
          </div>
        </form>
      </Modal>
    </div>
  );
}
