'use client';

import React, { useState, useEffect } from 'react';
import { businessAPI, adminAPI } from '@/lib/api';
import { useAuth } from '@/context/AuthContext';
import { Modal, ConfirmDialog, PageLoading, StatusBadge } from '@/components/ui';
import toast from 'react-hot-toast';
import { ghanaRegions } from '@/lib/utils';
import {
  BuildingStorefrontIcon,
  PlusIcon,
  PencilSquareIcon,
  UserPlusIcon,
  TrashIcon,
  UserMinusIcon,
  MagnifyingGlassIcon,
  XMarkIcon,
} from '@heroicons/react/24/outline';

export default function BusinessesPage() {
  const { user, isAdmin, switchBusiness } = useAuth();
  const [businesses, setBusinesses] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showAssignModal, setShowAssignModal] = useState(false);
  const [selectedBiz, setSelectedBiz] = useState<any>(null);
  const [confirmDeactivate, setConfirmDeactivate] = useState<string | null>(null);
  const [users, setUsers] = useState<any[]>([]);
  const [form, setForm] = useState<any>({});
  const [assignForm, setAssignForm] = useState<any>({ user_id: '', role: 'staff' });
  const [saving, setSaving] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');

  const loadBusinesses = async () => {
    setLoading(true);
    try {
      const res = await businessAPI.list();
      setBusinesses(res.data?.data || []);
    } catch {
      toast.error('Failed to load businesses');
    }
    setLoading(false);
  };

  const loadUsers = async () => {
    try {
      const res = await adminAPI.listUsers();
      setUsers(res.data?.data || []);
    } catch {
      // silently fail
    }
  };

  useEffect(() => {
    loadBusinesses();
    if (isAdmin) loadUsers();
  }, [isAdmin]);

  const openCreate = () => {
    setForm({
      name: '',
      trading_name: '',
      tin_number: '',
      phone: '',
      email: '',
      address: '',
      city: 'Accra',
      region: 'Greater Accra',
      currency: 'GHS',
      industry: '',
    });
    setShowCreateModal(true);
  };

  const openEdit = async (biz: any) => {
    try {
      const res = await businessAPI.get(biz.id);
      const data = res.data?.data || biz;
      setSelectedBiz(data);
      setForm({
        name: data.name || '',
        trading_name: data.trading_name || '',
        tin_number: data.tin_number || '',
        phone: data.phone || '',
        email: data.email || '',
        address: data.address || '',
        city: data.city || 'Accra',
        region: data.region || '',
        currency: data.currency || 'GHS',
        industry: data.industry || '',
        is_active: data.is_active !== false,
      });
      setShowEditModal(true);
    } catch {
      toast.error('Failed to load business details');
    }
  };

  const handleCreate = async () => {
    if (!form.name) { toast.error('Business name is required'); return; }
    setSaving(true);
    try {
      await businessAPI.create(form);
      toast.success('Business created!');
      setShowCreateModal(false);
      loadBusinesses();
    } catch (e: any) {
      toast.error(e?.response?.data?.message || 'Failed to create business');
    }
    setSaving(false);
  };

  const handleUpdate = async () => {
    if (!selectedBiz) return;
    setSaving(true);
    try {
      await businessAPI.update(selectedBiz.id, form);
      toast.success('Business updated!');
      setShowEditModal(false);
      loadBusinesses();
    } catch (e: any) {
      toast.error(e?.response?.data?.message || 'Failed to update business');
    }
    setSaving(false);
  };

  const handleDeactivate = async () => {
    if (!confirmDeactivate) return;
    try {
      await businessAPI.deactivate(confirmDeactivate);
      toast.success('Business deactivated');
      setConfirmDeactivate(null);
      loadBusinesses();
    } catch (e: any) {
      toast.error(e?.response?.data?.message || 'Failed to deactivate');
    }
  };

  const openAssign = (biz: any) => {
    setSelectedBiz(biz);
    setAssignForm({ user_id: '', role: 'staff' });
    setShowAssignModal(true);
  };

  const handleAssignUser = async () => {
    if (!selectedBiz || !assignForm.user_id) { toast.error('Select a user'); return; }
    setSaving(true);
    try {
      await businessAPI.assignUser(selectedBiz.id, assignForm);
      toast.success('User assigned!');
      setShowAssignModal(false);
      loadBusinesses();
    } catch (e: any) {
      toast.error(e?.response?.data?.message || 'Failed to assign user');
    }
    setSaving(false);
  };

  const handleRemoveUser = async (bizId: string, userId: string) => {
    if (!confirm('Remove this user from the business?')) return;
    try {
      await businessAPI.removeUser(bizId, { user_id: userId });
      toast.success('User removed');
      // Refresh the selected business if we're in edit mode
      if (selectedBiz?.id === bizId) {
        const res = await businessAPI.get(bizId);
        setSelectedBiz(res.data?.data);
      }
    } catch (e: any) {
      toast.error(e?.response?.data?.message || 'Failed to remove user');
    }
  };

  const handleSwitch = async (companyId: string) => {
    try {
      await switchBusiness(companyId);
    } catch {
      toast.error('Failed to switch');
    }
  };

  if (loading) return <PageLoading />;

  const roles = ['admin', 'manager', 'accountant', 'cashier', 'staff', 'viewer'];

  const filteredBusinesses = businesses.filter((biz) => {
    if (!searchQuery.trim()) return true;
    const q = searchQuery.toLowerCase();
    return (
      biz.name?.toLowerCase().includes(q) ||
      biz.trading_name?.toLowerCase().includes(q) ||
      biz.city?.toLowerCase().includes(q) ||
      biz.region?.toLowerCase().includes(q) ||
      biz.email?.toLowerCase().includes(q) ||
      biz.phone?.toLowerCase().includes(q) ||
      biz.industry?.toLowerCase().includes(q)
    );
  });

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-neutral-900">Businesses</h1>
          <p className="text-sm text-neutral-500 mt-1">
            Manage businesses within your organization. Each business has its own accounts, invoices, and data.
          </p>
        </div>
        {isAdmin && (
          <button onClick={openCreate} className="btn btn-primary flex items-center gap-2">
            <PlusIcon className="h-5 w-5" />
            New Business
          </button>
        )}
      </div>

      {/* Search Bar */}
      <div className="relative">
        <MagnifyingGlassIcon className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-neutral-400" />
        <input
          type="text"
          placeholder="Search businesses by name, city, region, email..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="input pl-9 pr-9 w-full"
        />
        {searchQuery && (
          <button
            onClick={() => setSearchQuery('')}
            className="absolute right-3 top-1/2 -translate-y-1/2 text-neutral-400 hover:text-neutral-600"
          >
            <XMarkIcon className="h-4 w-4" />
          </button>
        )}
      </div>

      {/* Business Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {filteredBusinesses.map((biz) => (
          <div
            key={biz.id}
            className={`bg-white rounded-lg border p-5 hover:shadow-md transition-shadow ${
              biz.id === user?.company?.id ? 'border-green-300 ring-1 ring-green-200' : 'border-neutral-200'
            }`}
          >
            <div className="flex items-start justify-between">
              <div className="flex items-center gap-3">
                <div className="h-10 w-10 rounded-lg bg-neutral-100 flex items-center justify-center">
                  <BuildingStorefrontIcon className="h-5 w-5 text-neutral-600" />
                </div>
                <div>
                  <h3 className="text-sm font-semibold text-neutral-900">{biz.name}</h3>
                  {biz.trading_name && (
                    <p className="text-xs text-neutral-500">t/a {biz.trading_name}</p>
                  )}
                </div>
              </div>
              {biz.id === user?.company?.id && (
                <span className="text-xs bg-green-100 text-green-700 px-2 py-0.5 rounded-full font-medium">
                  Active
                </span>
              )}
            </div>

            <div className="mt-4 space-y-1 text-xs text-neutral-500">
              {biz.city && <p>{biz.city}{biz.region ? `, ${biz.region}` : ''}</p>}
              {biz.email && <p>{biz.email}</p>}
              {biz.phone && <p>{biz.phone}</p>}
              <p className="text-neutral-400">
                {biz.user_count || 0} user{(biz.user_count || 0) !== 1 ? 's' : ''}
              </p>
            </div>

            <div className="mt-4 pt-3 border-t border-neutral-100 flex items-center gap-2 flex-wrap">
              {biz.id !== user?.company?.id && (
                <button
                  onClick={() => handleSwitch(biz.id)}
                  className="text-xs px-3 py-1.5 rounded-md bg-neutral-900 text-white hover:bg-neutral-800 transition-colors"
                >
                  Switch to this
                </button>
              )}
              {isAdmin && (
                <>
                  <button
                    onClick={() => openEdit(biz)}
                    className="text-xs px-2 py-1.5 rounded-md text-neutral-600 hover:bg-neutral-100 transition-colors"
                    title="Edit"
                  >
                    <PencilSquareIcon className="h-4 w-4" />
                  </button>
                  <button
                    onClick={() => openAssign(biz)}
                    className="text-xs px-2 py-1.5 rounded-md text-neutral-600 hover:bg-neutral-100 transition-colors"
                    title="Assign User"
                  >
                    <UserPlusIcon className="h-4 w-4" />
                  </button>
                  {biz.id !== user?.company?.id && (
                    <button
                      onClick={() => setConfirmDeactivate(biz.id)}
                      className="text-xs px-2 py-1.5 rounded-md text-red-500 hover:bg-red-50 transition-colors"
                      title="Deactivate"
                    >
                      <TrashIcon className="h-4 w-4" />
                    </button>
                  )}
                </>
              )}
            </div>

            {/* Assigned users preview */}
            {selectedBiz?.id === biz.id && selectedBiz?.assigned_users?.length > 0 && (
              <div className="mt-3 pt-3 border-t border-neutral-100">
                <p className="text-xs font-medium text-neutral-700 mb-2">Assigned Users:</p>
                {selectedBiz.assigned_users.map((u: any) => (
                  <div key={u.id} className="flex items-center justify-between py-1">
                    <span className="text-xs text-neutral-600">
                      {u.first_name} {u.last_name} – <span className="capitalize text-neutral-400">{u.business_role}</span>
                    </span>
                    {isAdmin && u.id !== user?.id && (
                      <button
                        onClick={() => handleRemoveUser(biz.id, u.id)}
                        className="text-red-400 hover:text-red-600"
                      >
                        <UserMinusIcon className="h-3.5 w-3.5" />
                      </button>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        ))}
      </div>

      {filteredBusinesses.length === 0 && (
        <div className="text-center py-16">
          <BuildingStorefrontIcon className="h-12 w-12 text-neutral-300 mx-auto" />
          {searchQuery ? (
            <>
              <h3 className="mt-4 text-sm font-medium text-neutral-900">No matching businesses</h3>
              <p className="text-xs text-neutral-500 mt-1">
                No businesses match &ldquo;{searchQuery}&rdquo;. Try a different search term.
              </p>
              <button onClick={() => setSearchQuery('')} className="btn btn-outline mt-4 text-sm">
                Clear Search
              </button>
            </>
          ) : (
            <>
              <h3 className="mt-4 text-sm font-medium text-neutral-900">No businesses</h3>
              <p className="text-xs text-neutral-500 mt-1">Get started by creating your first business.</p>
              {isAdmin && (
                <button onClick={openCreate} className="btn btn-primary mt-4 text-sm">
                  Create Business
                </button>
              )}
            </>
          )}
        </div>
      )}

      {/* Create Business Modal */}
      {showCreateModal && (
        <Modal open={showCreateModal} title="New Business" onClose={() => setShowCreateModal(false)} size="lg">
          <div className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Business Name *</label>
                <input
                  type="text"
                  className="input"
                  value={form.name}
                  onChange={(e) => setForm({ ...form, name: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Trading Name</label>
                <input
                  type="text"
                  className="input"
                  value={form.trading_name}
                  onChange={(e) => setForm({ ...form, trading_name: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">TIN Number</label>
                <input
                  type="text"
                  className="input"
                  value={form.tin_number}
                  onChange={(e) => setForm({ ...form, tin_number: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Phone</label>
                <input
                  type="text"
                  className="input"
                  value={form.phone}
                  onChange={(e) => setForm({ ...form, phone: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Email</label>
                <input
                  type="email"
                  className="input"
                  value={form.email}
                  onChange={(e) => setForm({ ...form, email: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Currency</label>
                <select className="input" value={form.currency} onChange={(e) => setForm({ ...form, currency: e.target.value })}>
                  <option value="GHS">GHS - Ghana Cedi</option>
                  <option value="USD">USD - US Dollar</option>
                  <option value="GBP">GBP - British Pound</option>
                  <option value="EUR">EUR - Euro</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">City</label>
                <input
                  type="text"
                  className="input"
                  value={form.city}
                  onChange={(e) => setForm({ ...form, city: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Region</label>
                <select className="input" value={form.region} onChange={(e) => setForm({ ...form, region: e.target.value })}>
                  <option value="">Select region</option>
                  {(ghanaRegions || []).map((r: string) => (
                    <option key={r} value={r}>{r}</option>
                  ))}
                </select>
              </div>
              <div className="md:col-span-2">
                <label className="block text-sm font-medium text-neutral-700 mb-1">Address</label>
                <textarea
                  className="input"
                  rows={2}
                  value={form.address}
                  onChange={(e) => setForm({ ...form, address: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Industry</label>
                <input
                  type="text"
                  className="input"
                  placeholder="e.g. Retail, Manufacturing"
                  value={form.industry}
                  onChange={(e) => setForm({ ...form, industry: e.target.value })}
                />
              </div>
            </div>
            <div className="flex justify-end gap-3 pt-2">
              <button onClick={() => setShowCreateModal(false)} className="btn btn-outline">Cancel</button>
              <button onClick={handleCreate} disabled={saving} className="btn btn-primary">
                {saving ? 'Creating...' : 'Create Business'}
              </button>
            </div>
          </div>
        </Modal>
      )}

      {/* Edit Business Modal */}
      {showEditModal && selectedBiz && (
        <Modal open={showEditModal} title={`Edit: ${selectedBiz.name}`} onClose={() => setShowEditModal(false)} size="lg">
          <div className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Business Name *</label>
                <input type="text" className="input" value={form.name} onChange={(e) => setForm({ ...form, name: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Trading Name</label>
                <input type="text" className="input" value={form.trading_name} onChange={(e) => setForm({ ...form, trading_name: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">TIN Number</label>
                <input type="text" className="input" value={form.tin_number} onChange={(e) => setForm({ ...form, tin_number: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Phone</label>
                <input type="text" className="input" value={form.phone} onChange={(e) => setForm({ ...form, phone: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Email</label>
                <input type="email" className="input" value={form.email} onChange={(e) => setForm({ ...form, email: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Currency</label>
                <select className="input" value={form.currency} onChange={(e) => setForm({ ...form, currency: e.target.value })}>
                  <option value="GHS">GHS</option><option value="USD">USD</option><option value="GBP">GBP</option><option value="EUR">EUR</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">City</label>
                <input type="text" className="input" value={form.city} onChange={(e) => setForm({ ...form, city: e.target.value })} />
              </div>
              <div>
                <label className="block text-sm font-medium text-neutral-700 mb-1">Region</label>
                <select className="input" value={form.region} onChange={(e) => setForm({ ...form, region: e.target.value })}>
                  <option value="">Select region</option>
                  {(ghanaRegions || []).map((r: string) => (<option key={r} value={r}>{r}</option>))}
                </select>
              </div>
              <div className="md:col-span-2">
                <label className="block text-sm font-medium text-neutral-700 mb-1">Address</label>
                <textarea className="input" rows={2} value={form.address} onChange={(e) => setForm({ ...form, address: e.target.value })} />
              </div>
            </div>

            {/* Assigned Users Section */}
            {selectedBiz.assigned_users && selectedBiz.assigned_users.length > 0 && (
              <div>
                <h4 className="text-sm font-semibold text-neutral-800 mb-2">Assigned Users</h4>
                <div className="bg-neutral-50 rounded-md divide-y divide-neutral-200">
                  {selectedBiz.assigned_users.map((u: any) => (
                    <div key={u.id} className="flex items-center justify-between px-3 py-2">
                      <div>
                        <span className="text-sm text-neutral-800">{u.first_name} {u.last_name}</span>
                        <span className="text-xs text-neutral-400 ml-2">{u.email}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <span className="text-xs capitalize bg-neutral-200 px-2 py-0.5 rounded">{u.business_role}</span>
                        {u.id !== user?.id && isAdmin && (
                          <button
                            onClick={() => handleRemoveUser(selectedBiz.id, u.id)}
                            className="text-red-400 hover:text-red-600"
                          >
                            <UserMinusIcon className="h-4 w-4" />
                          </button>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            <div className="flex justify-end gap-3 pt-2">
              <button onClick={() => setShowEditModal(false)} className="btn btn-outline">Cancel</button>
              <button onClick={handleUpdate} disabled={saving} className="btn btn-primary">
                {saving ? 'Saving...' : 'Save Changes'}
              </button>
            </div>
          </div>
        </Modal>
      )}

      {/* Assign User Modal */}
      {showAssignModal && selectedBiz && (
        <Modal open={showAssignModal} title={`Assign User to ${selectedBiz.name}`} onClose={() => setShowAssignModal(false)}>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-neutral-700 mb-1">User</label>
              <select
                className="input"
                value={assignForm.user_id}
                onChange={(e) => setAssignForm({ ...assignForm, user_id: e.target.value })}
              >
                <option value="">Select a user</option>
                {users.map((u) => (
                  <option key={u.id} value={u.id}>
                    {u.first_name} {u.last_name} ({u.email})
                  </option>
                ))}
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-neutral-700 mb-1">Role in this Business</label>
              <select
                className="input"
                value={assignForm.role}
                onChange={(e) => setAssignForm({ ...assignForm, role: e.target.value })}
              >
                {roles.map((r) => (
                  <option key={r} value={r}>{r.charAt(0).toUpperCase() + r.slice(1)}</option>
                ))}
              </select>
            </div>
            <div className="flex justify-end gap-3 pt-2">
              <button onClick={() => setShowAssignModal(false)} className="btn btn-outline">Cancel</button>
              <button onClick={handleAssignUser} disabled={saving} className="btn btn-primary">
                {saving ? 'Assigning...' : 'Assign User'}
              </button>
            </div>
          </div>
        </Modal>
      )}

      {/* Confirm Deactivate */}
      {confirmDeactivate && (
        <ConfirmDialog
          open={!!confirmDeactivate}
          title="Deactivate Business"
          message="Are you sure you want to deactivate this business? Users will no longer be able to access its data."
          confirmLabel="Deactivate"
          onConfirm={handleDeactivate}
          onCancel={() => setConfirmDeactivate(null)}
          variant="danger"
        />
      )}
    </div>
  );
}
