'use client';

import React, { useState, useEffect } from 'react';
import { contactAPI } from '@/lib/api';
import { formatCurrency, formatDate, ghanaRegions } from '@/lib/utils';
import { PageLoading, Modal, EmptyState } from '@/components/ui';
import toast from 'react-hot-toast';
import { PlusIcon, UsersIcon, PencilSquareIcon } from '@heroicons/react/24/outline';

const emptyForm = {
  name: '', contact_type: 'customer', email: '', phone: '',
  address: '', city: '', region: '', tin_number: '', notes: '',
  trading_name: '', vat_number: '', mobile: '', digital_address: '',
  payment_terms: '30', credit_limit: '0',
};

export default function ContactsPage() {
  const [contacts, setContacts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [filter, setFilter] = useState('all');
  const [search, setSearch] = useState('');
  const [editingContact, setEditingContact] = useState<any>(null);
  const [form, setForm] = useState(emptyForm);

  useEffect(() => { loadData(); }, []);

  const loadData = async () => {
    try {
      const { data } = await contactAPI.list({ limit: 500 });
      setContacts(data?.data || []);
    } catch (err) { console.error(err); }
    setLoading(false);
  };

  const openEdit = (contact: any) => {
    setEditingContact(contact);
    setForm({
      name: contact.name || '', contact_type: contact.contact_type || 'customer',
      email: contact.email || '', phone: contact.phone || '',
      address: contact.address || '', city: contact.city || '',
      region: contact.region || '', tin_number: contact.tin_number || '',
      notes: contact.notes || '', trading_name: contact.trading_name || '',
      vat_number: contact.vat_number || '', mobile: contact.mobile || '',
      digital_address: contact.digital_address || '',
      payment_terms: contact.payment_terms?.toString() || '30',
      credit_limit: contact.credit_limit?.toString() || '0',
    });
    setShowModal(true);
  };

  const closeModal = () => {
    setShowModal(false);
    setEditingContact(null);
    setForm(emptyForm);
  };

  const submitContact = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      if (editingContact) {
        await contactAPI.update(editingContact.id, form);
        toast.success('Contact updated');
      } else {
        await contactAPI.create(form);
        toast.success('Contact created');
      }
      closeModal();
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const filtered = contacts.filter((c: any) => {
    const matchType = filter === 'all' || c.contact_type === filter;
    const matchSearch = c.name?.toLowerCase().includes(search.toLowerCase()) ||
      c.email?.toLowerCase().includes(search.toLowerCase()) ||
      c.phone?.includes(search);
    return matchType && matchSearch;
  });

  if (loading) return <PageLoading />;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Contacts</h1>
        <button className="btn-primary" onClick={() => { setEditingContact(null); setForm(emptyForm); setShowModal(true); }}>
          <PlusIcon className="h-4 w-4 mr-1" /> Add Contact
        </button>
      </div>

      <div className="flex items-center gap-4">
        <input className="input max-w-sm" placeholder="Search contacts..." value={search} onChange={(e) => setSearch(e.target.value)} />
        <div className="flex gap-2">
          {['all', 'customer', 'supplier', 'both'].map((f) => (
            <button key={f}
              className={`px-3 py-1.5 text-xs font-medium rounded-lg capitalize ${
                filter === f ? 'bg-neutral-900 text-white' : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
              }`}
              onClick={() => setFilter(f)}
            >{f === 'all' ? 'All' : f + 's'}</button>
          ))}
        </div>
      </div>

      <div className="card">
        <div className="table-container">
          <table className="table">
            <thead>
              <tr>
                <th>Name</th>
                <th>Type</th>
                <th>Email</th>
                <th>Phone</th>
                <th>Region</th>
                <th>TIN</th>
                <th className="text-right">Balance</th>
                <th></th>
              </tr>
            </thead>
            <tbody>
              {filtered.length === 0 ? (
                <tr><td colSpan={8}><EmptyState title="No contacts found" icon={<UsersIcon className="h-12 w-12" />} /></td></tr>
              ) : (
                filtered.map((c: any) => (
                  <tr key={c.id}>
                    <td className="font-medium">{c.name}</td>
                    <td className="capitalize">{c.contact_type}</td>
                    <td>{c.email || '-'}</td>
                    <td>{c.phone || '-'}</td>
                    <td>{c.region || '-'}</td>
                    <td className="text-xs">{c.tin_number || '-'}</td>
                    <td className="text-right">{formatCurrency(c.outstanding_balance || 0)}</td>
                    <td>
                      <button onClick={() => openEdit(c)} className="p-1 text-neutral-400 hover:text-neutral-700 transition-colors" title="Edit contact">
                        <PencilSquareIcon className="h-4 w-4" />
                      </button>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      <Modal open={showModal} onClose={closeModal} title={editingContact ? 'Edit Contact' : 'Add Contact'} size="lg">
        <form onSubmit={submitContact} className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Name *</label>
              <input className="input" value={form.name} onChange={(e) => setForm({ ...form, name: e.target.value })} required />
            </div>
            <div>
              <label className="label">Type *</label>
              <select className="select" value={form.contact_type} onChange={(e) => setForm({ ...form, contact_type: e.target.value })}>
                <option value="customer">Customer</option>
                <option value="supplier">Supplier</option>
                <option value="both">Both</option>
              </select>
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Trading Name</label>
              <input className="input" value={form.trading_name} onChange={(e) => setForm({ ...form, trading_name: e.target.value })} />
            </div>
            <div>
              <label className="label">TIN Number</label>
              <input className="input" value={form.tin_number} onChange={(e) => setForm({ ...form, tin_number: e.target.value })} />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Email</label>
              <input type="email" className="input" value={form.email} onChange={(e) => setForm({ ...form, email: e.target.value })} />
            </div>
            <div>
              <label className="label">Phone</label>
              <input className="input" value={form.phone} onChange={(e) => setForm({ ...form, phone: e.target.value })} placeholder="+233" />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Mobile</label>
              <input className="input" value={form.mobile} onChange={(e) => setForm({ ...form, mobile: e.target.value })} placeholder="+233" />
            </div>
            <div>
              <label className="label">VAT Number</label>
              <input className="input" value={form.vat_number} onChange={(e) => setForm({ ...form, vat_number: e.target.value })} />
            </div>
          </div>
          <div>
            <label className="label">Address</label>
            <input className="input" value={form.address} onChange={(e) => setForm({ ...form, address: e.target.value })} />
          </div>
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">City</label>
              <input className="input" value={form.city} onChange={(e) => setForm({ ...form, city: e.target.value })} />
            </div>
            <div>
              <label className="label">Region</label>
              <select className="select" value={form.region} onChange={(e) => setForm({ ...form, region: e.target.value })}>
                <option value="">Select region</option>
                {ghanaRegions.map((r) => <option key={r} value={r}>{r}</option>)}
              </select>
            </div>
            <div>
              <label className="label">Digital Address</label>
              <input className="input" value={form.digital_address} onChange={(e) => setForm({ ...form, digital_address: e.target.value })} placeholder="GA-XXX-XXXX" />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Payment Terms (days)</label>
              <input type="number" className="input" value={form.payment_terms} onChange={(e) => setForm({ ...form, payment_terms: e.target.value })} />
            </div>
            <div>
              <label className="label">Credit Limit (GHS)</label>
              <input type="number" step="0.01" className="input" value={form.credit_limit} onChange={(e) => setForm({ ...form, credit_limit: e.target.value })} />
            </div>
          </div>
          <div>
            <label className="label">Notes</label>
            <textarea className="input" rows={2} value={form.notes} onChange={(e) => setForm({ ...form, notes: e.target.value })} />
          </div>
          <div className="flex justify-end gap-3 pt-2">
            <button type="button" className="btn-secondary" onClick={closeModal}>Cancel</button>
            <button type="submit" className="btn-primary">{editingContact ? 'Update Contact' : 'Create Contact'}</button>
          </div>
        </form>
      </Modal>
    </div>
  );
}
