'use client';

import React, { useState, useEffect } from 'react';
import { inventoryAPI } from '@/lib/api';
import { formatCurrency, formatDate } from '@/lib/utils';
import { PageLoading, StatusBadge, Modal, EmptyState } from '@/components/ui';
import toast from 'react-hot-toast';
import { PlusIcon, CubeIcon, ExclamationTriangleIcon, PencilSquareIcon, ArrowUpTrayIcon, ArrowDownTrayIcon, TrashIcon } from '@heroicons/react/24/outline';
import * as XLSX from 'xlsx';

export default function InventoryPage() {
  const [tab, setTab] = useState<'products' | 'lowstock' | 'categories'>('products');
  const [products, setProducts] = useState<any[]>([]);
  const [categories, setCategories] = useState<any[]>([]);
  const [lowStock, setLowStock] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [search, setSearch] = useState('');
  const [editingProduct, setEditingProduct] = useState<any>(null);
  const [showImportModal, setShowImportModal] = useState(false);
  const [importData, setImportData] = useState<any[]>([]);
  const [importing, setImporting] = useState(false);
  const fileInputRef = React.useRef<HTMLInputElement>(null);
  const [showCatModal, setShowCatModal] = useState(false);
  const [editingCategory, setEditingCategory] = useState<any>(null);
  const [catForm, setCatForm] = useState({ name: '', description: '' });

  const emptyForm = {
    name: '', sku: '', barcode: '', description: '', category_id: '',
    cost_price: '', selling_price: '', quantity_on_hand: '0', reorder_level: '10',
    unit_of_measure: 'piece', is_service: false, margin_percent: '',
  };
  const [form, setForm] = useState(emptyForm);

  useEffect(() => { loadData(); }, []);

  const loadData = async () => {
    try {
      const [prodRes, catRes, lowRes] = await Promise.all([
        inventoryAPI.listProducts({ limit: 500 }),
        inventoryAPI.getCategories(),
        inventoryAPI.lowStock(),
      ]);
      setProducts(prodRes.data?.data || []);
      setCategories(catRes.data?.data || []);
      setLowStock(lowRes.data?.data || []);
    } catch (err) { console.error(err); }
    setLoading(false);
  };

  // Auto-calculate selling price when cost or margin changes
  const handleCostChange = (cost: string) => {
    const newForm = { ...form, cost_price: cost };
    if (form.margin_percent && cost) {
      newForm.selling_price = (parseFloat(cost) * (1 + parseFloat(form.margin_percent) / 100)).toFixed(2);
    }
    setForm(newForm);
  };

  const handleMarginChange = (margin: string) => {
    const newForm = { ...form, margin_percent: margin };
    if (margin && form.cost_price) {
      newForm.selling_price = (parseFloat(form.cost_price) * (1 + parseFloat(margin) / 100)).toFixed(2);
    }
    setForm(newForm);
  };

  const openEdit = (product: any) => {
    setEditingProduct(product);
    const cost = parseFloat(product.cost_price) || 0;
    const sell = parseFloat(product.selling_price) || 0;
    const margin = cost > 0 ? (((sell - cost) / cost) * 100).toFixed(1) : '';
    setForm({
      name: product.name || '', sku: product.sku || '', barcode: product.barcode || '',
      description: product.description || '', category_id: product.category_id?.toString() || '',
      cost_price: product.cost_price?.toString() || '', selling_price: product.selling_price?.toString() || '',
      quantity_on_hand: (product.quantity_on_hand ?? product.stock_quantity)?.toString() || '0',
      reorder_level: product.reorder_level?.toString() || '10',
      unit_of_measure: product.unit_of_measure || product.unit || 'piece',
      is_service: !!product.is_service, margin_percent: margin,
    });
    setShowModal(true);
  };

  const closeModal = () => {
    setShowModal(false);
    setEditingProduct(null);
    setForm(emptyForm);
  };

  // ── Excel Import ─────────────────────────
  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    const reader = new FileReader();
    reader.onload = (evt) => {
      try {
        const data = new Uint8Array(evt.target?.result as ArrayBuffer);
        const workbook = XLSX.read(data, { type: 'array' });
        const sheet = workbook.Sheets[workbook.SheetNames[0]];
        const rows: any[] = XLSX.utils.sheet_to_json(sheet, { defval: '' });

        // Normalize column headers (case-insensitive, underscored)
        const normalized = rows.map((row) => {
          const clean: any = {};
          Object.entries(row).forEach(([key, val]) => {
            const k = key.toLowerCase().trim().replace(/\s+/g, '_');
            // Map common column name variations
            const map: Record<string, string> = {
              'product_name': 'name', 'product': 'name', 'item': 'name', 'item_name': 'name',
              'cost': 'cost_price', 'buying_price': 'cost_price', 'purchase_price': 'cost_price',
              'price': 'selling_price', 'sell_price': 'selling_price', 'retail_price': 'selling_price',
              'stock': 'quantity_on_hand', 'qty': 'quantity_on_hand', 'quantity': 'quantity_on_hand',
              'stock_quantity': 'quantity_on_hand', 'opening_stock': 'quantity_on_hand',
              'unit': 'unit_of_measure', 'uom': 'unit_of_measure',
              'category_name': 'category', 'product_category': 'category',
              'reorder': 'reorder_level', 'min_stock': 'reorder_level',
            };
            clean[map[k] || k] = val;
          });
          return clean;
        }).filter((r) => r.name);

        if (normalized.length === 0) {
          toast.error('No valid product rows found. Ensure column "Name" exists.');
          return;
        }

        setImportData(normalized);
        setShowImportModal(true);
      } catch {
        toast.error('Failed to parse Excel file');
      }
    };
    reader.readAsArrayBuffer(file);
    e.target.value = '';
  };

  const submitImport = async () => {
    setImporting(true);
    try {
      const res = await inventoryAPI.importProducts(importData);
      const d = res.data?.data;
      toast.success(`${d?.imported || 0} product(s) imported`);
      if (d?.errors?.length) {
        d.errors.slice(0, 5).forEach((err: string) => toast.error(err, { duration: 6000 }));
      }
      setShowImportModal(false);
      setImportData([]);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Import failed');
    }
    setImporting(false);
  };

  const downloadTemplate = () => {
    const template = [
      { Name: 'Example Product', SKU: 'EXP-001', Category: 'General', Cost_Price: 10.00,
        Selling_Price: 15.00, Quantity: 100, Reorder_Level: 10, Unit: 'piece', Barcode: '', Description: '' }
    ];
    const ws = XLSX.utils.json_to_sheet(template);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Products');
    XLSX.writeFile(wb, 'inventory_import_template.xlsx');
  };

  // ── Category CRUD ─────────────────────────
  const openCatEdit = (cat: any) => {
    setEditingCategory(cat);
    setCatForm({ name: cat.name || '', description: cat.description || '' });
    setShowCatModal(true);
  };

  const closeCatModal = () => {
    setShowCatModal(false);
    setEditingCategory(null);
    setCatForm({ name: '', description: '' });
  };

  const submitCategory = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      if (editingCategory) {
        await inventoryAPI.updateCategory(editingCategory.id, catForm);
        toast.success('Category updated');
      } else {
        await inventoryAPI.createCategory(catForm);
        toast.success('Category created');
      }
      closeCatModal();
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const deleteCategory = async (cat: any) => {
    if (!confirm(`Delete category "${cat.name}"? This cannot be undone.`)) return;
    try {
      await inventoryAPI.deleteCategory(cat.id);
      toast.success('Category deleted');
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Cannot delete — products may be assigned');
    }
  };

  const submitProduct = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const { margin_percent, ...payload } = form;
      if (editingProduct) {
        await inventoryAPI.updateProduct(editingProduct.id, payload);
        toast.success('Product updated');
      } else {
        await inventoryAPI.createProduct(payload);
        toast.success('Product created');
      }
      closeModal();
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const filtered = products.filter((p: any) =>
    p.name?.toLowerCase().includes(search.toLowerCase()) ||
    p.sku?.toLowerCase().includes(search.toLowerCase())
  );

  if (loading) return <PageLoading />;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Inventory</h1>
        <div className="flex items-center gap-2">
          <button className="btn-secondary text-sm" onClick={downloadTemplate}>
            <ArrowDownTrayIcon className="h-4 w-4 mr-1" /> Template
          </button>
          <button className="btn-secondary text-sm" onClick={() => fileInputRef.current?.click()}>
            <ArrowUpTrayIcon className="h-4 w-4 mr-1" /> Upload Excel
          </button>
          <input ref={fileInputRef} type="file" accept=".xlsx,.xls,.csv" className="hidden" onChange={handleFileSelect} />
          <button className="btn-primary" onClick={() => { setEditingProduct(null); setForm(emptyForm); setShowModal(true); }}>
            <PlusIcon className="h-4 w-4 mr-1" /> Add Product
          </button>
        </div>
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="flex gap-6">
          {[
            { key: 'products', label: 'All Products' },
            { key: 'lowstock', label: `Low Stock (${lowStock.length})` },
            { key: 'categories', label: 'Categories' },
          ].map((t) => (
            <button key={t.key}
              className={`pb-3 text-sm font-medium border-b-2 transition-colors ${
                tab === t.key ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-gray-500'
              }`}
              onClick={() => setTab(t.key as any)}
            >{t.label}</button>
          ))}
        </nav>
      </div>

      {tab === 'products' && (
        <>
          <input
            className="input max-w-md"
            placeholder="Search products..."
            value={search}
            onChange={(e) => setSearch(e.target.value)}
          />
          <div className="card">
            <div className="table-container">
              <table className="table">
                <thead>
                  <tr>
                    <th>SKU</th>
                    <th>Name</th>
                    <th>Category</th>
                    <th className="text-right">Cost</th>
                    <th className="text-right">Selling Price</th>
                    <th className="text-right">Stock</th>
                    <th>Status</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody>
                  {filtered.length === 0 ? (
                    <tr><td colSpan={8}><EmptyState title="No products" icon={<CubeIcon className="h-12 w-12" />} /></td></tr>
                  ) : (
                    filtered.map((p: any) => (
                      <tr key={p.id}>
                        <td className="font-mono text-xs">{p.sku}</td>
                        <td className="font-medium">{p.name}</td>
                        <td>{p.category_name || '-'}</td>
                        <td className="text-right">{formatCurrency(p.cost_price)}</td>
                        <td className="text-right font-medium">{formatCurrency(p.selling_price)}</td>
                        <td className={`text-right font-bold ${parseInt(p.quantity_on_hand ?? p.stock_quantity) <= parseInt(p.reorder_level) ? 'text-red-600' : 'text-green-600'}`}>
                          {p.quantity_on_hand ?? p.stock_quantity}
                        </td>
                        <td><StatusBadge status={p.is_active ? 'active' : 'inactive'} /></td>
                        <td>
                          <button onClick={() => openEdit(p)} className="p-1 text-neutral-400 hover:text-neutral-700 transition-colors" title="Edit product">
                            <PencilSquareIcon className="h-4 w-4" />
                          </button>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </div>
        </>
      )}

      {tab === 'lowstock' && (
        <div className="card">
          {lowStock.length === 0 ? (
            <div className="card-body">
              <EmptyState title="All stock levels are healthy" icon={<CubeIcon className="h-12 w-12" />} />
            </div>
          ) : (
            <div className="table-container">
              <table className="table">
                <thead>
                  <tr>
                    <th></th>
                    <th>Product</th>
                    <th className="text-right">Current Stock</th>
                    <th className="text-right">Reorder Level</th>
                    <th className="text-right">Deficit</th>
                  </tr>
                </thead>
                <tbody>
                  {lowStock.map((p: any) => (
                    <tr key={p.id}>
                      <td><ExclamationTriangleIcon className="h-5 w-5 text-red-500" /></td>
                      <td className="font-medium">{p.name}</td>
                      <td className="text-right text-red-600 font-bold">{p.stock_quantity}</td>
                      <td className="text-right">{p.reorder_level}</td>
                      <td className="text-right font-medium text-red-600">
                        {parseInt(p.reorder_level) - parseInt(p.stock_quantity)}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </div>
      )}

      {tab === 'categories' && (
        <div>
          <div className="flex justify-end mb-4">
            <button className="btn-primary" onClick={() => { setEditingCategory(null); setCatForm({ name: '', description: '' }); setShowCatModal(true); }}>
              <PlusIcon className="h-4 w-4 mr-1" /> Add Category
            </button>
          </div>
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
            {categories.map((cat: any) => (
              <div key={cat.id} className="card p-4 flex items-start justify-between">
                <div>
                  <h3 className="font-medium text-gray-900">{cat.name}</h3>
                  <p className="text-sm text-gray-500 mt-1">{cat.description || 'No description'}</p>
                </div>
                <div className="flex gap-1 shrink-0 ml-2">
                  <button onClick={() => openCatEdit(cat)} className="p-1 text-neutral-400 hover:text-neutral-700 transition-colors" title="Edit">
                    <PencilSquareIcon className="h-4 w-4" />
                  </button>
                  <button onClick={() => deleteCategory(cat)} className="p-1 text-neutral-400 hover:text-red-600 transition-colors" title="Delete">
                    <TrashIcon className="h-4 w-4" />
                  </button>
                </div>
              </div>
            ))}
            {categories.length === 0 && (
              <EmptyState title="No categories" description="Create product categories to organize your inventory" />
            )}
          </div>

          {/* Category Modal */}
          <Modal open={showCatModal} onClose={closeCatModal} title={editingCategory ? 'Edit Category' : 'Add Category'}>
            <form onSubmit={submitCategory} className="space-y-4">
              <div>
                <label className="label">Category Name *</label>
                <input className="input" value={catForm.name} onChange={(e) => setCatForm({ ...catForm, name: e.target.value })} required />
              </div>
              <div>
                <label className="label">Description</label>
                <textarea className="input" rows={2} value={catForm.description} onChange={(e) => setCatForm({ ...catForm, description: e.target.value })} />
              </div>
              <div className="flex justify-end gap-3 pt-2">
                <button type="button" className="btn-secondary" onClick={closeCatModal}>Cancel</button>
                <button type="submit" className="btn-primary">{editingCategory ? 'Update Category' : 'Create Category'}</button>
              </div>
            </form>
          </Modal>
        </div>
      )}

      {/* Add Product Modal */}
      <Modal open={showModal} onClose={closeModal} title={editingProduct ? 'Edit Product' : 'Add Product'} size="lg">
        <form onSubmit={submitProduct} className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Product Name *</label>
              <input className="input" value={form.name} onChange={(e) => setForm({ ...form, name: e.target.value })} required />
            </div>
            <div>
              <label className="label">SKU *</label>
              <input className="input" value={form.sku} onChange={(e) => setForm({ ...form, sku: e.target.value })} required />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Barcode</label>
              <input className="input" value={form.barcode} onChange={(e) => setForm({ ...form, barcode: e.target.value })} />
            </div>
            <div>
              <label className="label">Category</label>
              <select className="select" value={form.category_id} onChange={(e) => setForm({ ...form, category_id: e.target.value })}>
                <option value="">None</option>
                {categories.map((c: any) => <option key={c.id} value={c.id}>{c.name}</option>)}
              </select>
            </div>
          </div>
          <div className="grid grid-cols-4 gap-4">
            <div>
              <label className="label">Cost Price (GHS) *</label>
              <input type="number" step="0.01" className="input" value={form.cost_price} onChange={(e) => handleCostChange(e.target.value)} required />
            </div>
            <div>
              <label className="label">Margin %</label>
              <input type="number" step="0.1" min="0" className="input" placeholder="e.g. 30" value={form.margin_percent} onChange={(e) => handleMarginChange(e.target.value)} />
            </div>
            <div>
              <label className="label">Selling Price (GHS) *</label>
              <input type="number" step="0.01" className="input" value={form.selling_price} onChange={(e) => setForm({ ...form, selling_price: e.target.value })} required />
            </div>
            <div>
              <label className="label">{editingProduct ? 'Stock' : 'Opening Stock'}</label>
              <input type="number" className="input" value={form.quantity_on_hand} onChange={(e) => setForm({ ...form, quantity_on_hand: e.target.value })} />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Reorder Level</label>
              <input type="number" className="input" value={form.reorder_level} onChange={(e) => setForm({ ...form, reorder_level: e.target.value })} />
            </div>
            <div>
              <label className="label">Unit</label>
              <select className="select" value={form.unit_of_measure} onChange={(e) => setForm({ ...form, unit_of_measure: e.target.value })}>
                <option value="piece">Piece</option>
                <option value="unit">Unit</option>
                <option value="kg">Kilogram</option>
                <option value="litre">Litre</option>
                <option value="plate">Plate</option>
                <option value="box">Box</option>
                <option value="pack">Pack</option>
                <option value="dozen">Dozen</option>
              </select>
            </div>
          </div>
          <div>
            <label className="label">Description</label>
            <textarea className="input" rows={2} value={form.description} onChange={(e) => setForm({ ...form, description: e.target.value })} />
          </div>
          <div className="flex items-center gap-2">
            <input type="checkbox" checked={form.is_service} onChange={(e) => setForm({ ...form, is_service: e.target.checked })} id="is_service" />
            <label htmlFor="is_service" className="text-sm text-gray-700">This is a service (no stock tracking)</label>
          </div>
          <div className="flex justify-end gap-3 pt-2">
            <button type="button" className="btn-secondary" onClick={closeModal}>Cancel</button>
            <button type="submit" className="btn-primary">{editingProduct ? 'Update Product' : 'Create Product'}</button>
          </div>
        </form>
      </Modal>

      {/* Import Preview Modal */}
      <Modal open={showImportModal} onClose={() => { setShowImportModal(false); setImportData([]); }} title={`Import Preview — ${importData.length} product(s)`} size="lg">
        <div className="space-y-4">
          <p className="text-sm text-gray-500">
            Review the data below. Products with matching SKUs will be updated. New categories will be auto-created.
          </p>
          <div className="max-h-80 overflow-auto border border-gray-200 rounded-lg">
            <table className="table text-xs">
              <thead className="sticky top-0 bg-white">
                <tr>
                  <th className="py-1 px-2">#</th>
                  <th className="py-1 px-2">Name</th>
                  <th className="py-1 px-2">SKU</th>
                  <th className="py-1 px-2">Category</th>
                  <th className="py-1 px-2 text-right">Cost</th>
                  <th className="py-1 px-2 text-right">Selling</th>
                  <th className="py-1 px-2 text-right">Stock</th>
                  <th className="py-1 px-2">Unit</th>
                </tr>
              </thead>
              <tbody>
                {importData.map((row, i) => (
                  <tr key={i} className={i % 2 === 0 ? 'bg-gray-50' : ''}>
                    <td className="py-1 px-2 text-gray-400">{i + 1}</td>
                    <td className="py-1 px-2 font-medium">{row.name}</td>
                    <td className="py-1 px-2 font-mono">{row.sku || '(auto)'}</td>
                    <td className="py-1 px-2">{row.category || '-'}</td>
                    <td className="py-1 px-2 text-right">{formatCurrency(row.cost_price || 0)}</td>
                    <td className="py-1 px-2 text-right">{formatCurrency(row.selling_price || 0)}</td>
                    <td className="py-1 px-2 text-right">{row.quantity_on_hand || 0}</td>
                    <td className="py-1 px-2">{row.unit_of_measure || 'piece'}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          <div className="flex justify-end gap-3">
            <button type="button" className="btn-secondary" onClick={() => { setShowImportModal(false); setImportData([]); }}>Cancel</button>
            <button
              type="button"
              className="btn-primary"
              disabled={importing}
              onClick={submitImport}
            >
              {importing ? 'Importing...' : `Import ${importData.length} Product(s)`}
            </button>
          </div>
        </div>
      </Modal>
    </div>
  );
}
