'use client';

import React, { useState, useEffect } from 'react';
import { invoiceAPI, contactAPI, inventoryAPI } from '@/lib/api';
import { formatCurrency, formatDate } from '@/lib/utils';
import { PageLoading, StatusBadge, Modal, EmptyState } from '@/components/ui';
import DatePicker from '@/components/ui/DatePicker';
import FileAttachments from '@/components/ui/FileAttachments';
import toast from 'react-hot-toast';
import {
  PlusIcon,
  DocumentTextIcon,
  PencilSquareIcon,
  PrinterIcon,
  EyeIcon,
  XMarkIcon,
  MagnifyingGlassIcon,
  ArrowLeftIcon,
} from '@heroicons/react/24/outline';

interface InvoiceItem {
  product_id: string;
  description: string;
  quantity: string;
  unit_price: string;
  tax_rate_id: string;
}

const emptyItem = (): InvoiceItem => ({
  product_id: '',
  description: '',
  quantity: '1',
  unit_price: '',
  tax_rate_id: '',
});

export default function InvoicesPage() {
  const [invoices, setInvoices] = useState<any[]>([]);
  const [contacts, setContacts] = useState<any[]>([]);
  const [products, setProducts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [filter, setFilter] = useState('all');

  // Detail / View
  const [selectedInvoice, setSelectedInvoice] = useState<any>(null);
  const [invoiceDetail, setInvoiceDetail] = useState<any>(null);
  const [detailLoading, setDetailLoading] = useState(false);
  const [viewMode, setViewMode] = useState<'list' | 'detail'>('list');

  // Edit modal
  const [showEditModal, setShowEditModal] = useState(false);
  const [editForm, setEditForm] = useState<any>(null);

  // Create form
  const [form, setForm] = useState({
    invoice_type: 'sales',
    contact_id: '',
    invoice_date: new Date().toISOString().split('T')[0],
    due_date: '',
    notes: '',
    tax_inclusive: true,
    items: [emptyItem()],
  });

  // Product search for line items
  const [activeItemIdx, setActiveItemIdx] = useState<number | null>(null);
  const [productSearch, setProductSearch] = useState('');

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [invRes, contRes, prodRes] = await Promise.all([
        invoiceAPI.list(),
        contactAPI.list(),
        inventoryAPI.listProducts({ limit: 500 }),
      ]);
      setInvoices(invRes.data?.data || []);
      setContacts(contRes.data?.data || []);
      setProducts(prodRes.data?.data || []);
    } catch (err) { console.error(err); }
    setLoading(false);
  };

  const loadInvoiceDetail = async (id: string) => {
    setDetailLoading(true);
    try {
      const res = await invoiceAPI.get(id);
      setInvoiceDetail(res.data?.data || null);
    } catch (err) {
      console.error(err);
      toast.error('Failed to load invoice details');
    }
    setDetailLoading(false);
  };

  // ── Create invoice ──
  const createInvoice = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await invoiceAPI.create(form);
      toast.success('Invoice created');
      setShowModal(false);
      setForm({
        invoice_type: 'sales', contact_id: '', invoice_date: new Date().toISOString().split('T')[0],
        due_date: '', notes: '', tax_inclusive: true, items: [emptyItem()],
      });
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error creating invoice');
    }
  };

  // ── Update invoice ──
  const updateInvoice = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!editForm) return;
    try {
      await invoiceAPI.update(editForm.id, {
        contact_id: editForm.contact_id,
        invoice_date: editForm.invoice_date,
        due_date: editForm.due_date,
        notes: editForm.notes,
        items: editForm.items,
      });
      toast.success('Invoice updated');
      setShowEditModal(false);
      setEditForm(null);
      loadData();
      if (invoiceDetail?.id === editForm.id) {
        loadInvoiceDetail(editForm.id);
      }
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error updating invoice');
    }
  };

  // ── Void invoice ──
  const voidInvoice = async (id: string) => {
    if (!confirm('Are you sure you want to void this invoice?')) return;
    try {
      await invoiceAPI.void(id);
      toast.success('Invoice voided');
      loadData();
      setViewMode('list');
      setInvoiceDetail(null);
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error voiding invoice');
    }
  };

  // ── Line item helpers (shared between create & edit) ──
  const addItem = (formState: any, setFormState: (v: any) => void) => {
    setFormState({ ...formState, items: [...formState.items, emptyItem()] });
  };

  const updateItem = (formState: any, setFormState: (v: any) => void, i: number, field: string, value: string) => {
    const items = [...formState.items];
    (items[i] as any)[field] = value;
    setFormState({ ...formState, items });
  };

  const removeItem = (formState: any, setFormState: (v: any) => void, i: number) => {
    if (formState.items.length <= 1) return;
    setFormState({ ...formState, items: formState.items.filter((_: any, idx: number) => idx !== i) });
  };

  const selectProduct = (formState: any, setFormState: (v: any) => void, itemIdx: number, product: any) => {
    const items = [...formState.items];
    items[itemIdx] = {
      ...items[itemIdx],
      product_id: product.id,
      description: product.name,
      unit_price: product.selling_price?.toString() || '',
    };
    setFormState({ ...formState, items });
    setActiveItemIdx(null);
    setProductSearch('');
  };

  const clearProductSelection = (formState: any, setFormState: (v: any) => void, itemIdx: number) => {
    const items = [...formState.items];
    items[itemIdx] = { ...items[itemIdx], product_id: '' };
    setFormState({ ...formState, items });
  };

  // ── Open edit modal ──
  const openEditModal = () => {
    if (!invoiceDetail) return;
    setEditForm({
      id: invoiceDetail.id,
      contact_id: invoiceDetail.contact_id,
      invoice_date: invoiceDetail.invoice_date,
      due_date: invoiceDetail.due_date || '',
      notes: invoiceDetail.notes || '',
      items: (invoiceDetail.items || []).map((it: any) => ({
        product_id: it.product_id || '',
        description: it.description || it.product_name || '',
        quantity: it.quantity?.toString() || '1',
        unit_price: it.unit_price?.toString() || '',
        tax_rate_id: it.tax_rate_id || '',
      })),
    });
    setShowEditModal(true);
  };

  // ── View & Print ──
  const handleViewInvoice = async (inv: any) => {
    setSelectedInvoice(inv);
    setViewMode('detail');
    await loadInvoiceDetail(inv.id);
  };

  const handlePrint = () => {
    if (!invoiceDetail) return;
    const pw = window.open('', '_blank', 'width=800,height=900');
    if (!pw) return;

    const items = invoiceDetail.items || [];
    const payments = invoiceDetail.payments || [];

    const itemRows = items.map((it: any) =>
      `<tr>
        <td style="padding:6px 10px;border-bottom:1px solid #eee">${it.description || it.product_name || '-'}</td>
        <td style="padding:6px 10px;border-bottom:1px solid #eee;text-align:center">${it.quantity}</td>
        <td style="padding:6px 10px;border-bottom:1px solid #eee;text-align:right">${formatCurrency(it.unit_price)}</td>
        <td style="padding:6px 10px;border-bottom:1px solid #eee;text-align:right">${formatCurrency(it.line_total)}</td>
      </tr>`
    ).join('');

    const paymentRows = payments.length > 0
      ? payments.map((p: any) =>
          `<tr><td style="padding:4px 0">${formatDate(p.payment_date)}</td><td style="padding:4px 0">${p.payment_method || '-'}</td><td style="padding:4px 0;text-align:right">${formatCurrency(p.amount)}</td></tr>`
        ).join('')
      : '';

    const html = `<!DOCTYPE html><html><head><meta charset="utf-8"><title>Invoice ${invoiceDetail.invoice_number}</title>
    <style>
      *{margin:0;padding:0;box-sizing:border-box}
      body{font-family:'Segoe UI',Arial,sans-serif;color:#333;padding:40px;max-width:800px;margin:0 auto;font-size:13px;line-height:1.5}
      .header{display:flex;justify-content:space-between;margin-bottom:30px}
      .company h1{font-size:22px;font-weight:800;color:#111}
      .company p{font-size:11px;color:#666;margin-top:2px}
      .inv-title{font-size:28px;font-weight:800;color:#111;text-align:right}
      .inv-num{font-size:12px;color:#666;text-align:right;margin-top:4px}
      .meta{display:flex;justify-content:space-between;margin-bottom:24px;padding:16px;background:#f8f8f8;border-radius:6px}
      .meta-col{flex:1}
      .meta-col h4{font-size:10px;text-transform:uppercase;color:#999;letter-spacing:0.5px;margin-bottom:4px}
      .meta-col p{font-size:12px;color:#333}
      table{width:100%;border-collapse:collapse;margin-bottom:20px}
      th{background:#111;color:#fff;padding:8px 10px;font-size:11px;text-transform:uppercase;text-align:left}
      th:nth-child(2){text-align:center}
      th:nth-child(3),th:nth-child(4){text-align:right}
      .totals{width:300px;margin-left:auto}
      .totals .row{display:flex;justify-content:space-between;padding:4px 0;font-size:12px}
      .totals .row.total{font-size:16px;font-weight:800;border-top:2px solid #111;padding-top:8px;margin-top:4px}
      .totals .row.tax{color:#777;font-size:11px}
      .notes{margin-top:20px;padding:12px;background:#fafafa;border-radius:6px;font-size:11px;color:#666}
      .payments{margin-top:20px}
      .payments h3{font-size:12px;font-weight:700;margin-bottom:8px}
      .payments table th{background:#555}
      .footer{text-align:center;margin-top:40px;font-size:10px;color:#999}
      @media print{@page{size:A4;margin:15mm}body{padding:0}}
    </style></head><body>
    <div class="header">
      <div class="company">
        <h1>${invoiceDetail.company_name || 'Company'}</h1>
        <p>${invoiceDetail.company_address || ''}</p>
        ${invoiceDetail.company_phone ? `<p>Tel: ${invoiceDetail.company_phone}</p>` : ''}
        ${invoiceDetail.company_email ? `<p>${invoiceDetail.company_email}</p>` : ''}
        ${invoiceDetail.company_tin ? `<p>TIN: ${invoiceDetail.company_tin}</p>` : ''}
        ${invoiceDetail.company_vat ? `<p>VAT: ${invoiceDetail.company_vat}</p>` : ''}
      </div>
      <div>
        <div class="inv-title">${invoiceDetail.invoice_type === 'sales' ? 'INVOICE' : 'BILL'}</div>
        <div class="inv-num">${invoiceDetail.invoice_number}</div>
        <div class="inv-num" style="margin-top:8px"><strong>Status:</strong> ${invoiceDetail.status?.toUpperCase()}</div>
      </div>
    </div>

    <div class="meta">
      <div class="meta-col">
        <h4>Bill To</h4>
        <p><strong>${invoiceDetail.contact_name || '-'}</strong></p>
        <p>${invoiceDetail.contact_address || ''}</p>
        ${invoiceDetail.contact_email ? `<p>${invoiceDetail.contact_email}</p>` : ''}
        ${invoiceDetail.contact_phone ? `<p>${invoiceDetail.contact_phone}</p>` : ''}
        ${invoiceDetail.contact_tin ? `<p>TIN: ${invoiceDetail.contact_tin}</p>` : ''}
      </div>
      <div class="meta-col" style="text-align:right">
        <h4>Invoice Date</h4>
        <p>${formatDate(invoiceDetail.invoice_date)}</p>
        <h4 style="margin-top:8px">Due Date</h4>
        <p>${invoiceDetail.due_date ? formatDate(invoiceDetail.due_date) : '-'}</p>
        ${invoiceDetail.reference ? `<h4 style="margin-top:8px">Reference</h4><p>${invoiceDetail.reference}</p>` : ''}
      </div>
    </div>

    <table>
      <thead><tr><th>Description</th><th>Qty</th><th>Unit Price</th><th>Amount</th></tr></thead>
      <tbody>${itemRows}</tbody>
    </table>

    <div class="totals">
      <div class="row"><span>Subtotal</span><span>${formatCurrency(invoiceDetail.subtotal)}</span></div>
      ${parseFloat(invoiceDetail.nhil_amount) > 0 ? `<div class="row tax"><span>NHIL (2.5%)</span><span>${formatCurrency(invoiceDetail.nhil_amount)}</span></div>` : ''}
      ${parseFloat(invoiceDetail.getfund_amount) > 0 ? `<div class="row tax"><span>GETFund (2.5%)</span><span>${formatCurrency(invoiceDetail.getfund_amount)}</span></div>` : ''}
      ${parseFloat(invoiceDetail.vat_amount) > 0 ? `<div class="row tax"><span>VAT (15%)</span><span>${formatCurrency(invoiceDetail.vat_amount)}</span></div>` : ''}
      ${parseFloat(invoiceDetail.discount_amount) > 0 ? `<div class="row"><span>Discount</span><span>-${formatCurrency(invoiceDetail.discount_amount)}</span></div>` : ''}
      <div class="row total"><span>Total</span><span>${formatCurrency(invoiceDetail.total_amount)}</span></div>
      ${parseFloat(invoiceDetail.amount_paid) > 0 ? `<div class="row"><span>Amount Paid</span><span>${formatCurrency(invoiceDetail.amount_paid)}</span></div>` : ''}
      <div class="row" style="font-weight:700;color:${parseFloat(invoiceDetail.balance_due) > 0 ? '#dc2626' : '#059669'}"><span>Balance Due</span><span>${formatCurrency(invoiceDetail.balance_due)}</span></div>
    </div>

    ${payments.length > 0 ? `
    <div class="payments">
      <h3>Payment History</h3>
      <table>
        <thead><tr><th>Date</th><th>Method</th><th style="text-align:right">Amount</th></tr></thead>
        <tbody>${paymentRows}</tbody>
      </table>
    </div>` : ''}

    ${invoiceDetail.notes ? `<div class="notes"><strong>Notes:</strong> ${invoiceDetail.notes}</div>` : ''}

    <div class="footer">
      <p>Thank you for your business!</p>
      <p style="margin-top:4px">Powered by CashBook ERP</p>
    </div>
    <script>window.onload=function(){window.print();}<\/script>
    </body></html>`;

    pw.document.write(html);
    pw.document.close();
  };

  // ── Filtered list ──
  const filteredInvoices = filter === 'all'
    ? invoices
    : invoices.filter((inv) => inv.status === filter);

  // ── Product dropdown filter ──
  const filteredProducts = products.filter((p: any) =>
    p.name?.toLowerCase().includes(productSearch.toLowerCase()) ||
    p.sku?.toLowerCase().includes(productSearch.toLowerCase())
  );

  if (loading) return <PageLoading />;

  // ── LINE ITEMS TABLE (shared) ──
  const renderLineItems = (formState: any, setFormState: (v: any) => void) => (
    <div>
      <div className="flex justify-between items-center mb-2">
        <h4 className="text-sm font-medium">Items</h4>
        <button type="button" className="btn-secondary btn-sm" onClick={() => addItem(formState, setFormState)}>+ Add Item</button>
      </div>
      <div className="space-y-3">
        {formState.items.map((item: InvoiceItem, i: number) => (
          <div key={i} className="border border-gray-200 rounded-lg p-3 bg-gray-50 relative">
            <div className="grid grid-cols-12 gap-3">
              {/* Product / Description - spans 5 cols */}
              <div className="col-span-5 relative">
                <label className="text-xs text-gray-500 mb-1 block">
                  {item.product_id ? '📦 Product' : 'Description'}
                </label>
                <div className="flex gap-1">
                  <input
                    className="input text-xs flex-1"
                    placeholder="Type or search product..."
                    value={item.description}
                    onChange={(e) => {
                      updateItem(formState, setFormState, i, 'description', e.target.value);
                      if (!item.product_id) {
                        setActiveItemIdx(i);
                        setProductSearch(e.target.value);
                      }
                    }}
                    onFocus={() => {
                      if (!item.product_id) {
                        setActiveItemIdx(i);
                        setProductSearch(item.description);
                      }
                    }}
                    required
                  />
                  {item.product_id ? (
                    <button
                      type="button"
                      className="text-red-400 hover:text-red-600 px-1"
                      onClick={() => clearProductSelection(formState, setFormState, i)}
                      title="Clear product selection"
                    >
                      <XMarkIcon className="h-4 w-4" />
                    </button>
                  ) : (
                    <button
                      type="button"
                      className="text-gray-400 hover:text-gray-600 px-1"
                      onClick={() => {
                        setActiveItemIdx(activeItemIdx === i ? null : i);
                        setProductSearch('');
                      }}
                      title="Browse products"
                    >
                      <MagnifyingGlassIcon className="h-4 w-4" />
                    </button>
                  )}
                </div>

                {/* Product dropdown */}
                {activeItemIdx === i && !item.product_id && (
                  <div className="absolute z-20 top-full left-0 right-0 mt-1 bg-white border border-gray-200 rounded-lg shadow-lg max-h-48 overflow-y-auto">
                    <div className="sticky top-0 bg-white px-2 py-1.5 border-b border-gray-100">
                      <input
                        className="input text-xs w-full"
                        placeholder="Search inventory..."
                        value={productSearch}
                        onChange={(e) => setProductSearch(e.target.value)}
                        autoFocus
                      />
                    </div>
                    {filteredProducts.length === 0 ? (
                      <div className="px-3 py-2 text-xs text-gray-400">No products found — type a custom description</div>
                    ) : (
                      filteredProducts.slice(0, 20).map((p: any) => (
                        <button
                          key={p.id}
                          type="button"
                          className="w-full text-left px-3 py-2 hover:bg-gray-50 flex items-center justify-between text-xs border-b border-gray-50"
                          onClick={() => selectProduct(formState, setFormState, i, p)}
                        >
                          <div>
                            <p className="font-medium text-gray-900">{p.name}</p>
                            <p className="text-gray-400">{p.sku || 'No SKU'}</p>
                          </div>
                          <span className="font-semibold text-gray-700">{formatCurrency(p.selling_price)}</span>
                        </button>
                      ))
                    )}
                    <button
                      type="button"
                      className="w-full text-center px-3 py-2 text-xs text-gray-500 hover:bg-gray-50 border-t border-gray-100"
                      onClick={() => { setActiveItemIdx(null); setProductSearch(''); }}
                    >
                      Close
                    </button>
                  </div>
                )}
              </div>

              {/* Quantity */}
              <div className="col-span-2">
                <label className="text-xs text-gray-500 mb-1 block">Qty</label>
                <input
                  type="number"
                  min="1"
                  className="input text-xs w-full"
                  value={item.quantity}
                  onChange={(e) => updateItem(formState, setFormState, i, 'quantity', e.target.value)}
                />
              </div>

              {/* Unit Price */}
              <div className="col-span-2">
                <label className="text-xs text-gray-500 mb-1 block">Price (GHS)</label>
                <input
                  type="number"
                  step="0.01"
                  className="input text-xs w-full"
                  value={item.unit_price}
                  onChange={(e) => updateItem(formState, setFormState, i, 'unit_price', e.target.value)}
                  required
                />
              </div>

              {/* Amount */}
              <div className="col-span-2">
                <label className="text-xs text-gray-500 mb-1 block">Amount</label>
                <div className="text-sm font-medium text-gray-900 pt-1.5">
                  {formatCurrency((parseFloat(item.quantity || '0') * parseFloat(item.unit_price || '0')))}
                </div>
              </div>

              {/* Remove */}
              <div className="col-span-1 flex items-end pb-1">
                {formState.items.length > 1 && (
                  <button
                    type="button"
                    className="text-red-400 hover:text-red-600 text-xs"
                    onClick={() => removeItem(formState, setFormState, i)}
                  >
                    Remove
                  </button>
                )}
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Line items total */}
      <div className="mt-3 text-right">
        <span className="text-sm text-gray-500">Subtotal: </span>
        <span className="text-sm font-bold text-gray-900">
          {formatCurrency(formState.items.reduce((s: number, it: InvoiceItem) =>
            s + (parseFloat(it.quantity || '0') * parseFloat(it.unit_price || '0')), 0
          ))}
        </span>
      </div>
    </div>
  );

  // ══════════════════════════
  // DETAIL VIEW
  // ══════════════════════════
  if (viewMode === 'detail' && selectedInvoice) {
    return (
      <div className="space-y-6">
        {/* Back + Actions */}
        <div className="flex items-center justify-between">
          <button
            className="flex items-center gap-2 text-sm text-gray-600 hover:text-gray-900"
            onClick={() => { setViewMode('list'); setInvoiceDetail(null); setSelectedInvoice(null); }}
          >
            <ArrowLeftIcon className="h-4 w-4" /> Back to Invoices
          </button>
          <div className="flex gap-2">
            {invoiceDetail && ['draft', 'sent'].includes(invoiceDetail.status) && (
              <button className="btn-secondary btn-sm flex items-center gap-1.5" onClick={openEditModal}>
                <PencilSquareIcon className="h-4 w-4" /> Edit
              </button>
            )}
            {invoiceDetail && (
              <button className="btn-primary btn-sm flex items-center gap-1.5" onClick={handlePrint}>
                <PrinterIcon className="h-4 w-4" /> Print
              </button>
            )}
            {invoiceDetail && !['voided', 'paid'].includes(invoiceDetail.status) && (
              <button className="btn-danger btn-sm" onClick={() => voidInvoice(invoiceDetail.id)}>
                Void
              </button>
            )}
          </div>
        </div>

        {detailLoading ? (
          <PageLoading />
        ) : invoiceDetail ? (
          <div className="space-y-6">
            {/* Invoice header card */}
            <div className="card p-6">
              <div className="flex justify-between items-start">
                <div>
                  <h1 className="text-2xl font-bold text-gray-900">{invoiceDetail.invoice_number}</h1>
                  <p className="text-sm text-gray-500 mt-1 capitalize">{invoiceDetail.invoice_type} Invoice</p>
                </div>
                <StatusBadge status={invoiceDetail.status} className="text-sm" />
              </div>

              <div className="grid grid-cols-3 gap-6 mt-6">
                <div>
                  <h4 className="text-xs font-medium uppercase text-gray-400 mb-1">From</h4>
                  <p className="font-semibold text-gray-900">{invoiceDetail.company_name}</p>
                  <p className="text-xs text-gray-500">{invoiceDetail.company_address}</p>
                  {invoiceDetail.company_phone && <p className="text-xs text-gray-500">{invoiceDetail.company_phone}</p>}
                  {invoiceDetail.company_tin && <p className="text-xs text-gray-500">TIN: {invoiceDetail.company_tin}</p>}
                </div>
                <div>
                  <h4 className="text-xs font-medium uppercase text-gray-400 mb-1">Bill To</h4>
                  <p className="font-semibold text-gray-900">{invoiceDetail.contact_name}</p>
                  <p className="text-xs text-gray-500">{invoiceDetail.contact_address}</p>
                  {invoiceDetail.contact_email && <p className="text-xs text-gray-500">{invoiceDetail.contact_email}</p>}
                  {invoiceDetail.contact_phone && <p className="text-xs text-gray-500">{invoiceDetail.contact_phone}</p>}
                  {invoiceDetail.contact_tin && <p className="text-xs text-gray-500">TIN: {invoiceDetail.contact_tin}</p>}
                </div>
                <div className="text-right">
                  <div className="mb-3">
                    <h4 className="text-xs font-medium uppercase text-gray-400 mb-1">Invoice Date</h4>
                    <p className="text-sm font-medium">{formatDate(invoiceDetail.invoice_date)}</p>
                  </div>
                  <div className="mb-3">
                    <h4 className="text-xs font-medium uppercase text-gray-400 mb-1">Due Date</h4>
                    <p className="text-sm font-medium">{invoiceDetail.due_date ? formatDate(invoiceDetail.due_date) : '-'}</p>
                  </div>
                  {invoiceDetail.reference && (
                    <div>
                      <h4 className="text-xs font-medium uppercase text-gray-400 mb-1">Reference</h4>
                      <p className="text-sm font-medium">{invoiceDetail.reference}</p>
                    </div>
                  )}
                </div>
              </div>
            </div>

            {/* Items table */}
            <div className="card">
              <div className="table-container">
                <table className="table">
                  <thead>
                    <tr>
                      <th className="w-12">#</th>
                      <th>Description</th>
                      <th>SKU</th>
                      <th className="text-center">Qty</th>
                      <th className="text-right">Unit Price</th>
                      <th className="text-right">Amount</th>
                    </tr>
                  </thead>
                  <tbody>
                    {(invoiceDetail.items || []).map((item: any, idx: number) => (
                      <tr key={idx}>
                        <td className="text-xs text-gray-400">{idx + 1}</td>
                        <td className="font-medium">{item.description || item.product_name || '-'}</td>
                        <td className="text-xs text-gray-500">{item.sku || '-'}</td>
                        <td className="text-center">{item.quantity}</td>
                        <td className="text-right">{formatCurrency(item.unit_price)}</td>
                        <td className="text-right font-medium">{formatCurrency(item.line_total)}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              {/* Totals section */}
              <div className="border-t border-gray-200 px-5 py-4">
                <div className="ml-auto w-72 space-y-1.5">
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-500">Subtotal</span>
                    <span>{formatCurrency(invoiceDetail.subtotal)}</span>
                  </div>
                  {parseFloat(invoiceDetail.nhil_amount) > 0 && (
                    <div className="flex justify-between text-xs text-gray-400">
                      <span>NHIL (2.5%)</span>
                      <span>{formatCurrency(invoiceDetail.nhil_amount)}</span>
                    </div>
                  )}
                  {parseFloat(invoiceDetail.getfund_amount) > 0 && (
                    <div className="flex justify-between text-xs text-gray-400">
                      <span>GETFund (2.5%)</span>
                      <span>{formatCurrency(invoiceDetail.getfund_amount)}</span>
                    </div>
                  )}
                  {parseFloat(invoiceDetail.vat_amount) > 0 && (
                    <div className="flex justify-between text-xs text-gray-400">
                      <span>VAT (15%)</span>
                      <span>{formatCurrency(invoiceDetail.vat_amount)}</span>
                    </div>
                  )}
                  {parseFloat(invoiceDetail.discount_amount) > 0 && (
                    <div className="flex justify-between text-sm text-gray-500">
                      <span>Discount</span>
                      <span>-{formatCurrency(invoiceDetail.discount_amount)}</span>
                    </div>
                  )}
                  <div className="flex justify-between text-base font-bold border-t border-gray-200 pt-2">
                    <span>Total</span>
                    <span>{formatCurrency(invoiceDetail.total_amount)}</span>
                  </div>
                  {parseFloat(invoiceDetail.amount_paid) > 0 && (
                    <div className="flex justify-between text-sm text-emerald-600">
                      <span>Paid</span>
                      <span>-{formatCurrency(invoiceDetail.amount_paid)}</span>
                    </div>
                  )}
                  <div className={`flex justify-between text-sm font-bold ${parseFloat(invoiceDetail.balance_due) > 0 ? 'text-red-600' : 'text-emerald-600'}`}>
                    <span>Balance Due</span>
                    <span>{formatCurrency(invoiceDetail.balance_due)}</span>
                  </div>
                </div>
              </div>
            </div>

            {/* Payment History */}
            {(invoiceDetail.payments || []).length > 0 && (
              <div className="card">
                <div className="px-5 py-3 border-b border-gray-200">
                  <h3 className="text-sm font-semibold text-gray-900">Payment History</h3>
                </div>
                <div className="table-container">
                  <table className="table">
                    <thead>
                      <tr>
                        <th>Payment #</th>
                        <th>Date</th>
                        <th>Method</th>
                        <th>Reference</th>
                        <th className="text-right">Amount</th>
                      </tr>
                    </thead>
                    <tbody>
                      {invoiceDetail.payments.map((p: any, idx: number) => (
                        <tr key={idx}>
                          <td className="font-mono text-xs">{p.payment_number || '-'}</td>
                          <td>{formatDate(p.payment_date)}</td>
                          <td className="capitalize">{p.payment_method?.replace('_', ' ') || '-'}</td>
                          <td className="text-xs">{p.reference || '-'}</td>
                          <td className="text-right font-medium">{formatCurrency(p.amount)}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>
            )}

            {/* Notes */}
            {invoiceDetail.notes && (
              <div className="card p-5">
                <h3 className="text-sm font-semibold text-gray-900 mb-2">Notes</h3>
                <p className="text-sm text-gray-600">{invoiceDetail.notes}</p>
              </div>
            )}

            {/* Attachments */}
            <div className="card p-5">
              <FileAttachments entityType="invoice" entityId={invoiceDetail.id} />
            </div>
          </div>
        ) : (
          <div className="card p-12 text-center">
            <p className="text-sm text-gray-500">Invoice not found</p>
          </div>
        )}

        {/* Edit Modal */}
        {editForm && (
          <Modal open={showEditModal} onClose={() => { setShowEditModal(false); setEditForm(null); }} title="Edit Invoice" size="xl">
            <form onSubmit={updateInvoice} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="label">Customer / Supplier *</label>
                  <select className="select" value={editForm.contact_id} onChange={(e) => setEditForm({ ...editForm, contact_id: e.target.value })} required>
                    <option value="">Select...</option>
                    {contacts.map((c: any) => (
                      <option key={c.id} value={c.id}>{c.name}</option>
                    ))}
                  </select>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="label">Invoice Date *</label>
                    <DatePicker value={editForm.invoice_date} onChange={(v) => setEditForm({ ...editForm, invoice_date: v })} required />
                  </div>
                  <div>
                    <label className="label">Due Date</label>
                    <DatePicker value={editForm.due_date} onChange={(v) => setEditForm({ ...editForm, due_date: v })} />
                  </div>
                </div>
              </div>

              {renderLineItems(editForm, setEditForm)}

              <div>
                <label className="label">Notes</label>
                <textarea className="input" rows={2} value={editForm.notes} onChange={(e) => setEditForm({ ...editForm, notes: e.target.value })} />
              </div>

              <div className="flex justify-end gap-3 pt-2">
                <button type="button" className="btn-secondary" onClick={() => { setShowEditModal(false); setEditForm(null); }}>Cancel</button>
                <button type="submit" className="btn-primary">Update Invoice</button>
              </div>
            </form>
          </Modal>
        )}
      </div>
    );
  }

  // ══════════════════════════
  // LIST VIEW
  // ══════════════════════════
  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Invoices & Billing</h1>
        <button className="btn-primary" onClick={() => setShowModal(true)}>
          <PlusIcon className="h-4 w-4 mr-1" /> New Invoice
        </button>
      </div>

      {/* Filter tabs */}
      <div className="flex gap-2">
        {['all', 'draft', 'sent', 'partial', 'paid', 'overdue'].map((f) => (
          <button
            key={f}
            className={`px-3 py-1.5 text-xs font-medium rounded-lg capitalize ${
              filter === f ? 'bg-neutral-900 text-white' : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
            }`}
            onClick={() => setFilter(f)}
          >
            {f}
          </button>
        ))}
      </div>

      {/* Invoice list */}
      <div className="card">
        <div className="table-container">
          <table className="table">
            <thead>
              <tr>
                <th>Invoice #</th>
                <th>Type</th>
                <th>Customer / Supplier</th>
                <th>Date</th>
                <th>Due Date</th>
                <th>Status</th>
                <th className="text-right">Total</th>
                <th className="text-right">Balance</th>
                <th></th>
              </tr>
            </thead>
            <tbody>
              {filteredInvoices.length === 0 ? (
                <tr><td colSpan={9}><EmptyState title="No invoices found" icon={<DocumentTextIcon className="h-12 w-12" />} /></td></tr>
              ) : (
                filteredInvoices.map((inv: any) => (
                  <tr key={inv.id} className="hover:bg-gray-50">
                    <td className="font-mono text-xs font-medium">{inv.invoice_number}</td>
                    <td className="capitalize">{inv.invoice_type}</td>
                    <td>{inv.contact_name || '-'}</td>
                    <td>{formatDate(inv.invoice_date)}</td>
                    <td>{inv.due_date ? formatDate(inv.due_date) : '-'}</td>
                    <td><StatusBadge status={inv.status} /></td>
                    <td className="text-right font-medium">{formatCurrency(inv.total_amount)}</td>
                    <td className="text-right">{formatCurrency(inv.balance_due)}</td>
                    <td>
                      <button
                        className="text-neutral-500 hover:text-neutral-900 p-1"
                        onClick={() => handleViewInvoice(inv)}
                        title="View invoice"
                      >
                        <EyeIcon className="h-4 w-4" />
                      </button>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* New Invoice Modal */}
      <Modal open={showModal} onClose={() => setShowModal(false)} title="New Invoice" size="xl">
        <form onSubmit={createInvoice} className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Type *</label>
              <select className="select" value={form.invoice_type} onChange={(e) => setForm({ ...form, invoice_type: e.target.value })}>
                <option value="sales">Sales Invoice</option>
                <option value="purchase">Purchase Invoice</option>
              </select>
            </div>
            <div>
              <label className="label">{form.invoice_type === 'sales' ? 'Customer' : 'Supplier'} *</label>
              <select className="select" value={form.contact_id} onChange={(e) => setForm({ ...form, contact_id: e.target.value })} required>
                <option value="">Select...</option>
                {contacts
                  .filter((c: any) =>
                    form.invoice_type === 'sales' ? c.contact_type !== 'supplier' : c.contact_type !== 'customer'
                  )
                  .map((c: any) => (
                    <option key={c.id} value={c.id}>{c.name}</option>
                  ))}
              </select>
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Invoice Date *</label>
              <DatePicker value={form.invoice_date} onChange={(v) => setForm({ ...form, invoice_date: v })} required />
            </div>
            <div>
              <label className="label">Due Date</label>
              <DatePicker value={form.due_date} onChange={(v) => setForm({ ...form, due_date: v })} />
            </div>
          </div>

          {/* Line items with product picker */}
          {renderLineItems(form, setForm)}

          <div>
            <label className="label">Notes</label>
            <textarea className="input" rows={2} value={form.notes} onChange={(e) => setForm({ ...form, notes: e.target.value })} />
          </div>

          <div className="flex justify-end gap-3 pt-2">
            <button type="button" className="btn-secondary" onClick={() => setShowModal(false)}>Cancel</button>
            <button type="submit" className="btn-primary">Create Invoice</button>
          </div>
        </form>
      </Modal>
    </div>
  );
}
