'use client';

import React, { useState, useEffect } from 'react';
import { payrollAPI } from '@/lib/api';
import { formatCurrency, formatDate } from '@/lib/utils';
import { PageLoading, StatusBadge, Modal, EmptyState } from '@/components/ui';
import FileAttachments from '@/components/ui/FileAttachments';
import DatePicker from '@/components/ui/DatePicker';
import toast from 'react-hot-toast';
import { PlusIcon, UserGroupIcon } from '@heroicons/react/24/outline';
import { useAuth } from '@/context/AuthContext';

export default function PayrollPage() {
  const { user } = useAuth();
  const [tab, setTab] = useState<'employees' | 'periods'>('employees');
  const [employees, setEmployees] = useState<any[]>([]);
  const [periods, setPeriods] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showEmpModal, setShowEmpModal] = useState(false);
  const [showRunModal, setShowRunModal] = useState(false);

  const [empForm, setEmpForm] = useState({
    employee_id: '', first_name: '', last_name: '', email: '', phone: '',
    department: '', position: '', basic_salary: '', ssnit_number: '', tin_number: '',
    bank_name: '', bank_account: '', hire_date: new Date().toISOString().split('T')[0],
    tier3_rate: '0',
  });

  const [runForm, setRunForm] = useState({
    start_date: '', end_date: '', description: '',
  });

  useEffect(() => { loadData(); }, []);

  const loadData = async () => {
    try {
      const [empRes, periodRes] = await Promise.all([
        payrollAPI.listEmployees(),
        payrollAPI.listPeriods(),
      ]);
      setEmployees(empRes.data?.data || []);
      setPeriods(periodRes.data?.data || []);
    } catch (err) { console.error(err); }
    setLoading(false);
  };

  const createEmployee = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await payrollAPI.createEmployee(empForm);
      toast.success('Employee added');
      setShowEmpModal(false);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const runPayroll = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const { data } = await payrollAPI.runPayroll(runForm);
      toast.success(`Payroll processed - ${data?.data?.payslips_count || 0} payslips generated`);
      setShowRunModal(false);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const approvePeriod = async (id: string) => {
    try {
      await payrollAPI.approvePayroll(id);
      toast.success('Payroll approved');
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  if (loading) return <PageLoading />;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Payroll</h1>
        <div className="flex gap-2">
          <button className="btn-secondary" onClick={() => setShowEmpModal(true)}>
            <PlusIcon className="h-4 w-4 mr-1" /> Add Employee
          </button>
          <button className="btn-primary" onClick={() => setShowRunModal(true)}>
            Run Payroll
          </button>
        </div>
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="flex gap-6">
          {[
            { key: 'employees', label: 'Employees' },
            { key: 'periods', label: 'Payroll Periods' },
          ].map((t) => (
            <button key={t.key}
              className={`pb-3 text-sm font-medium border-b-2 transition-colors ${
                tab === t.key ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-gray-500'
              }`}
              onClick={() => setTab(t.key as any)}
            >{t.label}</button>
          ))}
        </nav>
      </div>

      {/* Employees */}
      {tab === 'employees' && (
        <div className="card">
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Employee ID</th>
                  <th>Name</th>
                  <th>Department</th>
                  <th>Position</th>
                  <th className="text-right">Basic Salary</th>
                  <th>SSNIT #</th>
                  <th>Status</th>
                </tr>
              </thead>
              <tbody>
                {employees.length === 0 ? (
                  <tr><td colSpan={7}><EmptyState title="No employees" icon={<UserGroupIcon className="h-12 w-12" />} /></td></tr>
                ) : (
                  employees.map((emp: any) => (
                    <tr key={emp.id}>
                      <td className="font-mono text-xs">{emp.employee_id}</td>
                      <td className="font-medium">{emp.first_name} {emp.last_name}</td>
                      <td>{emp.department || '-'}</td>
                      <td>{emp.position || '-'}</td>
                      <td className="text-right">{formatCurrency(emp.basic_salary)}</td>
                      <td className="text-xs">{emp.ssnit_number || '-'}</td>
                      <td><StatusBadge status={emp.status || 'active'} /></td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Payroll Periods */}
      {tab === 'periods' && (
        <div className="card">
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Period</th>
                  <th>Start</th>
                  <th>End</th>
                  <th>Status</th>
                  <th className="text-right">Gross Pay</th>
                  <th className="text-right">Net Pay</th>
                  <th className="text-right">PAYE</th>
                  <th className="text-right">SSNIT</th>
                  <th></th>
                </tr>
              </thead>
              <tbody>
                {periods.length === 0 ? (
                  <tr><td colSpan={9}><EmptyState title="No payroll periods" /></td></tr>
                ) : (
                  periods.map((p: any) => (
                    <tr key={p.id}>
                      <td className="font-medium">{p.description || p.period_name}</td>
                      <td>{formatDate(p.start_date)}</td>
                      <td>{formatDate(p.end_date)}</td>
                      <td><StatusBadge status={p.status} /></td>
                      <td className="text-right">{formatCurrency(p.total_gross || 0)}</td>
                      <td className="text-right">{formatCurrency(p.total_net || 0)}</td>
                      <td className="text-right">{formatCurrency(p.total_paye || 0)}</td>
                      <td className="text-right">{formatCurrency(p.total_ssnit || 0)}</td>
                      <td>
                        {p.status === 'draft' && (
                          <button className="btn-success btn-sm" onClick={() => approvePeriod(p.id)}>Approve</button>
                        )}
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Payroll Documents */}
      {user?.company?.id && (
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Payroll Documents</h3>
            <p className="text-xs text-gray-500 mt-1">Attach payroll-related documents (payslips, tax returns, SSNIT reports, etc.)</p>
          </div>
          <div className="card-body">
            <FileAttachments entityType="payroll" entityId={user.company.id} />
          </div>
        </div>
      )}

      {/* Add Employee Modal */}
      <Modal open={showEmpModal} onClose={() => setShowEmpModal(false)} title="Add Employee" size="lg">
        <form onSubmit={createEmployee} className="space-y-4">
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">Employee ID *</label>
              <input className="input" value={empForm.employee_id} onChange={(e) => setEmpForm({ ...empForm, employee_id: e.target.value })} required />
            </div>
            <div>
              <label className="label">First Name *</label>
              <input className="input" value={empForm.first_name} onChange={(e) => setEmpForm({ ...empForm, first_name: e.target.value })} required />
            </div>
            <div>
              <label className="label">Last Name *</label>
              <input className="input" value={empForm.last_name} onChange={(e) => setEmpForm({ ...empForm, last_name: e.target.value })} required />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Email</label>
              <input type="email" className="input" value={empForm.email} onChange={(e) => setEmpForm({ ...empForm, email: e.target.value })} />
            </div>
            <div>
              <label className="label">Phone</label>
              <input className="input" value={empForm.phone} onChange={(e) => setEmpForm({ ...empForm, phone: e.target.value })} />
            </div>
          </div>
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">Department</label>
              <input className="input" value={empForm.department} onChange={(e) => setEmpForm({ ...empForm, department: e.target.value })} />
            </div>
            <div>
              <label className="label">Position</label>
              <input className="input" value={empForm.position} onChange={(e) => setEmpForm({ ...empForm, position: e.target.value })} />
            </div>
            <div>
              <label className="label">Basic Salary (GHS) *</label>
              <input type="number" step="0.01" className="input" value={empForm.basic_salary} onChange={(e) => setEmpForm({ ...empForm, basic_salary: e.target.value })} required />
            </div>
          </div>
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">SSNIT Number</label>
              <input className="input" value={empForm.ssnit_number} onChange={(e) => setEmpForm({ ...empForm, ssnit_number: e.target.value })} />
            </div>
            <div>
              <label className="label">TIN Number</label>
              <input className="input" value={empForm.tin_number} onChange={(e) => setEmpForm({ ...empForm, tin_number: e.target.value })} />
            </div>
            <div>
              <label className="label">Hire Date *</label>
              <DatePicker value={empForm.hire_date} onChange={(v) => setEmpForm({ ...empForm, hire_date: v })} required />
            </div>
          </div>
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">Bank Name</label>
              <input className="input" value={empForm.bank_name} onChange={(e) => setEmpForm({ ...empForm, bank_name: e.target.value })} />
            </div>
            <div>
              <label className="label">Bank Account #</label>
              <input className="input" value={empForm.bank_account} onChange={(e) => setEmpForm({ ...empForm, bank_account: e.target.value })} />
            </div>
            <div>
              <label className="label">Tier 3 Rate (%)</label>
              <input type="number" step="0.1" className="input" value={empForm.tier3_rate} onChange={(e) => setEmpForm({ ...empForm, tier3_rate: e.target.value })} />
            </div>
          </div>
          <div className="flex justify-end gap-3 pt-2">
            <button type="button" className="btn-secondary" onClick={() => setShowEmpModal(false)}>Cancel</button>
            <button type="submit" className="btn-primary">Add Employee</button>
          </div>
        </form>
      </Modal>

      {/* Run Payroll Modal */}
      <Modal open={showRunModal} onClose={() => setShowRunModal(false)} title="Run Payroll">
        <form onSubmit={runPayroll} className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Period Start *</label>
              <DatePicker value={runForm.start_date} onChange={(v) => setRunForm({ ...runForm, start_date: v })} required />
            </div>
            <div>
              <label className="label">Period End *</label>
              <DatePicker value={runForm.end_date} onChange={(v) => setRunForm({ ...runForm, end_date: v })} required />
            </div>
          </div>
          <div>
            <label className="label">Description</label>
            <input className="input" value={runForm.description} onChange={(e) => setRunForm({ ...runForm, description: e.target.value })} placeholder="e.g. January 2026 Payroll" />
          </div>
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-3 text-sm text-yellow-800">
            This will calculate PAYE, SSNIT (Tier 1 & 2), and generate payslips for all active employees.
          </div>
          <div className="flex justify-end gap-3 pt-2">
            <button type="button" className="btn-secondary" onClick={() => setShowRunModal(false)}>Cancel</button>
            <button type="submit" className="btn-primary">Process Payroll</button>
          </div>
        </form>
      </Modal>
    </div>
  );
}
