'use client';

import React, { useState, useEffect, useCallback } from 'react';
import { posAPI } from '@/lib/api';
import { formatCurrency } from '@/lib/utils';
import { PageLoading, Modal, StatusBadge, StatCard, EmptyState } from '@/components/ui';
import ReceiptPopup from '@/components/ui/ReceiptPopup';
import toast from 'react-hot-toast';
import {
  ShoppingCartIcon,
  MagnifyingGlassIcon,
  EyeIcon,
  ArrowPathIcon,
  FunnelIcon,
  ReceiptPercentIcon,
  BanknotesIcon,
  DevicePhoneMobileIcon,
  CreditCardIcon,
  PrinterIcon,
  ArrowUturnLeftIcon,
  ExclamationTriangleIcon,
  MinusIcon,
  PlusIcon,
} from '@heroicons/react/24/outline';

const RETURN_REASONS = [
  'Defective product',
  'Wrong item sold',
  'Customer changed mind',
  'Overcharged / pricing error',
  'Duplicate transaction',
  'Quality issue',
  'Expired product',
  'Other',
];

export default function POSHistoryPage() {
  const [loading, setLoading] = useState(true);
  const [sales, setSales] = useState<any[]>([]);
  const [total, setTotal] = useState(0);
  const [page, setPage] = useState(1);

  // Filters
  const [dateFrom, setDateFrom] = useState('');
  const [dateTo, setDateTo] = useState('');
  const [paymentMethod, setPaymentMethod] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [showFilters, setShowFilters] = useState(false);

  // Sale detail modal
  const [showDetail, setShowDetail] = useState(false);
  const [saleDetail, setSaleDetail] = useState<any>(null);
  const [detailLoading, setDetailLoading] = useState(false);

  // Return modal
  const [showReturn, setShowReturn] = useState(false);
  const [returnSale, setReturnSale] = useState<any>(null);
  const [returnReason, setReturnReason] = useState('');
  const [customReason, setCustomReason] = useState('');
  const [returnType, setReturnType] = useState<'full' | 'partial'>('full');
  const [returnItems, setReturnItems] = useState<any[]>([]);
  const [processing, setProcessing] = useState(false);

  // Returns history
  const [showReturns, setShowReturns] = useState(false);
  const [returns, setReturns] = useState<any[]>([]);

  // Receipt popup
  const [showReceipt, setShowReceipt] = useState(false);
  const [receiptData, setReceiptData] = useState<any>(null);

  // Daily summary
  const [dailyReport, setDailyReport] = useState<any>(null);

  const loadSales = useCallback(async () => {
    try {
      const params: any = { page, per_page: 20 };
      if (dateFrom) params.date_from = dateFrom;
      if (dateTo) params.date_to = dateTo;
      if (paymentMethod) params.payment_method = paymentMethod;

      const [salesRes, reportRes] = await Promise.all([
        posAPI.listSales(params),
        posAPI.dailyReport({ date: new Date().toISOString().split('T')[0] }),
      ]);

      let salesData = salesRes.data?.data || [];
      if (statusFilter) {
        salesData = salesData.filter((s: any) => s.status === statusFilter);
      }
      setSales(salesData);
      setTotal(salesRes.data?.pagination?.total || 0);
      setDailyReport(reportRes.data?.data || null);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  }, [page, dateFrom, dateTo, paymentMethod, statusFilter]);

  useEffect(() => { loadSales(); }, [loadSales]);

  const viewSale = async (id: string) => {
    setDetailLoading(true);
    setShowDetail(true);
    try {
      const { data } = await posAPI.getSale(id);
      setSaleDetail(data.data);
    } catch {
      toast.error('Failed to load sale details');
      setShowDetail(false);
    }
    setDetailLoading(false);
  };

  const openReturn = (sale: any) => {
    if (sale.status === 'refunded') {
      toast.error('This sale is already fully refunded');
      return;
    }
    setReturnSale(sale);
    setReturnReason('');
    setCustomReason('');
    setReturnType('full');
    setReturnItems((sale.items || saleDetail?.items || []).map((item: any) => ({
      sale_item_id: item.id,
      product_name: item.product_name,
      original_quantity: parseFloat(item.quantity),
      unit_price: parseFloat(item.unit_price),
      line_total: parseFloat(item.line_total),
      return_quantity: 0,
      selected: false,
    })));
    setShowReturn(true);
  };

  const handleReturn = async () => {
    const reason = returnReason === 'Other' ? customReason : returnReason;
    if (!reason.trim()) {
      toast.error('Please provide a reason for the return');
      return;
    }

    setProcessing(true);
    try {
      const payload: any = { reason };

      if (returnType === 'partial') {
        const selectedItems = returnItems.filter(i => i.selected && i.return_quantity > 0);
        if (selectedItems.length === 0) {
          toast.error('Select items and quantities to return');
          setProcessing(false);
          return;
        }
        payload.items = selectedItems.map(i => ({
          sale_item_id: i.sale_item_id,
          quantity: i.return_quantity,
        }));
      }

      const { data } = await posAPI.refund(returnSale.id, payload);
      toast.success(`Return processed: ${data.data?.return_number || 'Success'}`);
      setShowReturn(false);
      setShowDetail(false);
      loadSales();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Return failed');
    }
    setProcessing(false);
  };

  const toggleReturnItem = (idx: number) => {
    setReturnItems(prev => prev.map((item, i) =>
      i === idx ? { ...item, selected: !item.selected, return_quantity: !item.selected ? item.original_quantity : 0 } : item
    ));
  };

  const updateReturnQty = (idx: number, qty: number) => {
    setReturnItems(prev => prev.map((item, i) =>
      i === idx ? { ...item, return_quantity: Math.max(0, Math.min(qty, item.original_quantity)) } : item
    ));
  };

  const returnSubtotal = returnType === 'full'
    ? returnItems.reduce((sum, i) => sum + i.line_total, 0)
    : returnItems.filter(i => i.selected).reduce((sum, i) => sum + (i.return_quantity * i.unit_price), 0);

  const viewReceipt = (sale: any) => {
    setReceiptData({
      receipt_number: sale.sale_number,
      date: new Date(sale.sale_date || sale.created_at).toLocaleString('en-GH', { year: 'numeric', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit' }),
      items: (sale.items || []).map((item: any) => ({
        name: item.product_name,
        quantity: parseFloat(item.quantity),
        unit_price: parseFloat(item.unit_price),
        total: parseFloat(item.line_total),
      })),
      subtotal: parseFloat(sale.subtotal),
      nhil: parseFloat(sale.nhil_amount || 0),
      getfund: parseFloat(sale.getfund_amount || 0),
      vat: parseFloat(sale.vat_amount || 0),
      total_tax: parseFloat(sale.tax_amount || 0),
      grand_total: parseFloat(sale.total_amount),
      payment_method: sale.payment_method,
      amount_tendered: parseFloat(sale.amount_tendered || sale.total_amount),
      change: parseFloat(sale.change_given || 0),
      company_name: sale.company_name || 'CashBook ERP',
      cashier_name: sale.cashier_name || '',
    });
    setShowReceipt(true);
  };

  const loadReturns = async () => {
    try {
      const { data } = await posAPI.getReturns({ per_page: 50 });
      setReturns(data.data || []);
      setShowReturns(true);
    } catch {
      toast.error('Failed to load returns');
    }
  };

  const totalPages = Math.ceil(total / 20);

  if (loading) return <PageLoading />;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-neutral-900">POS Sales History</h1>
          <p className="text-sm text-neutral-500 mt-1">View all POS sales, process returns, and reprint receipts</p>
        </div>
        <div className="flex gap-3">
          <button className="btn-secondary flex items-center gap-2" onClick={loadReturns}>
            <ArrowUturnLeftIcon className="h-4 w-4" /> View Returns
          </button>
          <button className="btn-secondary flex items-center gap-2" onClick={loadSales}>
            <ArrowPathIcon className="h-4 w-4" /> Refresh
          </button>
        </div>
      </div>

      {/* Today's Summary */}
      {dailyReport && (
        <div className="grid grid-cols-2 sm:grid-cols-4 lg:grid-cols-6 gap-4">
          <StatCard title="Today's Sales" value={formatCurrency(dailyReport.total_sales)} subtitle={`${dailyReport.total_transactions} transactions`} icon={<ShoppingCartIcon className="h-5 w-5" />} />
          <StatCard title="Cash Sales" value={formatCurrency(dailyReport.cash_sales)} icon={<BanknotesIcon className="h-5 w-5 text-emerald-600" />} />
          <StatCard title="MoMo Sales" value={formatCurrency(dailyReport.momo_sales)} icon={<DevicePhoneMobileIcon className="h-5 w-5 text-yellow-600" />} />
          <StatCard title="Card Sales" value={formatCurrency(dailyReport.card_sales)} icon={<CreditCardIcon className="h-5 w-5 text-blue-600" />} />
          <StatCard title="Refunds" value={formatCurrency(dailyReport.refunds)} icon={<ArrowUturnLeftIcon className="h-5 w-5 text-red-600" />} />
          <StatCard title="VAT Collected" value={formatCurrency(dailyReport.total_vat)} icon={<ReceiptPercentIcon className="h-5 w-5 text-purple-600" />} />
        </div>
      )}

      {/* Filters */}
      <div className="card p-4">
        <div className="flex flex-wrap items-center gap-3">
          <select className="select w-auto" value={paymentMethod} onChange={(e) => { setPaymentMethod(e.target.value); setPage(1); }}>
            <option value="">All Methods</option>
            <option value="cash">Cash</option>
            <option value="mobile_money">Mobile Money</option>
            <option value="card">Card</option>
            <option value="split">Split</option>
          </select>
          <select className="select w-auto" value={statusFilter} onChange={(e) => { setStatusFilter(e.target.value); setPage(1); }}>
            <option value="">All Status</option>
            <option value="completed">Completed</option>
            <option value="refunded">Refunded</option>
            <option value="partially_refunded">Partially Refunded</option>
            <option value="voided">Voided</option>
          </select>
          <button className="btn-secondary flex items-center gap-1" onClick={() => setShowFilters(!showFilters)}>
            <FunnelIcon className="h-4 w-4" /> Date Range
          </button>
        </div>
        {showFilters && (
          <div className="flex items-center gap-3 mt-3 pt-3 border-t border-neutral-200">
            <div>
              <label className="text-xs text-neutral-500">From</label>
              <input type="date" className="input" value={dateFrom} onChange={(e) => { setDateFrom(e.target.value); setPage(1); }} />
            </div>
            <div>
              <label className="text-xs text-neutral-500">To</label>
              <input type="date" className="input" value={dateTo} onChange={(e) => { setDateTo(e.target.value); setPage(1); }} />
            </div>
            <button className="btn-secondary btn-sm mt-4" onClick={() => { setDateFrom(''); setDateTo(''); setPage(1); }}>Clear</button>
          </div>
        )}
      </div>

      {/* Sales Table */}
      <div className="card">
        <div className="table-container">
          <table className="table">
            <thead>
              <tr>
                <th>Sale #</th>
                <th>Date</th>
                <th>Cashier</th>
                <th>Customer</th>
                <th>Payment</th>
                <th className="text-right">Subtotal</th>
                <th className="text-right">Tax</th>
                <th className="text-right">Total</th>
                <th>Status</th>
                <th className="text-right">Actions</th>
              </tr>
            </thead>
            <tbody>
              {sales.length === 0 ? (
                <tr><td colSpan={10} className="text-center py-8">
                  <EmptyState icon={<ShoppingCartIcon className="h-12 w-12" />} title="No sales found" description="Sales will appear here after they are recorded at the POS" />
                </td></tr>
              ) : (
                sales.map((sale: any) => (
                  <tr key={sale.id} className="hover:bg-neutral-50 cursor-pointer" onClick={() => viewSale(sale.id)}>
                    <td className="font-mono text-xs font-medium text-blue-600">{sale.sale_number}</td>
                    <td className="text-xs">{new Date(sale.sale_date || sale.created_at).toLocaleString('en-GH', { day: 'numeric', month: 'short', year: '2-digit', hour: '2-digit', minute: '2-digit' })}</td>
                    <td className="text-sm">{sale.cashier_name || '-'}</td>
                    <td className="text-sm">{sale.customer_name || '-'}</td>
                    <td>
                      <span className="inline-flex items-center gap-1 text-xs capitalize">
                        {sale.payment_method === 'cash' && <BanknotesIcon className="h-3.5 w-3.5 text-emerald-600" />}
                        {sale.payment_method === 'mobile_money' && <DevicePhoneMobileIcon className="h-3.5 w-3.5 text-yellow-600" />}
                        {sale.payment_method === 'card' && <CreditCardIcon className="h-3.5 w-3.5 text-blue-600" />}
                        {sale.payment_method?.replace(/_/g, ' ')}
                      </span>
                    </td>
                    <td className="text-right text-sm">{formatCurrency(sale.subtotal)}</td>
                    <td className="text-right text-xs text-neutral-500">{formatCurrency(sale.tax_amount)}</td>
                    <td className="text-right font-semibold">{formatCurrency(sale.total_amount)}</td>
                    <td><StatusBadge status={sale.status} /></td>
                    <td className="text-right" onClick={(e) => e.stopPropagation()}>
                      <div className="flex items-center justify-end gap-1">
                        <button onClick={() => viewSale(sale.id)} className="p-1.5 rounded hover:bg-neutral-100 text-neutral-400 hover:text-neutral-700" title="View Details">
                          <EyeIcon className="h-4 w-4" />
                        </button>
                        {sale.status !== 'refunded' && sale.status !== 'voided' && (
                          <button
                            onClick={() => {
                              viewSale(sale.id).then(() => {
                                // Open return after detail loads
                              });
                            }}
                            className="p-1.5 rounded hover:bg-red-50 text-neutral-400 hover:text-red-600"
                            title="Process Return"
                          >
                            <ArrowUturnLeftIcon className="h-4 w-4" />
                          </button>
                        )}
                      </div>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        {totalPages > 1 && (
          <div className="flex items-center justify-between px-4 py-3 border-t border-neutral-200">
            <p className="text-sm text-neutral-500">Showing {(page - 1) * 20 + 1}-{Math.min(page * 20, total)} of {total}</p>
            <div className="flex gap-2">
              <button className="btn-secondary btn-sm" disabled={page <= 1} onClick={() => setPage(page - 1)}>Previous</button>
              <button className="btn-secondary btn-sm" disabled={page >= totalPages} onClick={() => setPage(page + 1)}>Next</button>
            </div>
          </div>
        )}
      </div>

      {/* Sale Detail Modal */}
      <Modal open={showDetail} onClose={() => { setShowDetail(false); setSaleDetail(null); }} title="Sale Details" size="xl">
        {detailLoading ? (
          <div className="flex items-center justify-center py-12"><div className="animate-spin rounded-full h-8 w-8 border-2 border-neutral-200 border-t-neutral-900" /></div>
        ) : saleDetail ? (
          <div className="space-y-4">
            {/* Sale header */}
            <div className="flex items-start justify-between">
              <div>
                <p className="font-mono text-sm text-neutral-500">{saleDetail.sale_number}</p>
                <p className="text-2xl font-bold text-neutral-900">{formatCurrency(saleDetail.total_amount)}</p>
                <p className="text-xs text-neutral-500 mt-1">
                  {new Date(saleDetail.sale_date || saleDetail.created_at).toLocaleString('en-GH', { weekday: 'short', day: 'numeric', month: 'long', year: 'numeric', hour: '2-digit', minute: '2-digit' })}
                </p>
              </div>
              <div className="flex items-center gap-2">
                <StatusBadge status={saleDetail.status} />
                <span className="inline-flex items-center gap-1 px-2 py-1 rounded-md bg-neutral-100 text-xs capitalize">
                  {saleDetail.payment_method === 'cash' && <BanknotesIcon className="h-3.5 w-3.5" />}
                  {saleDetail.payment_method === 'mobile_money' && <DevicePhoneMobileIcon className="h-3.5 w-3.5" />}
                  {saleDetail.payment_method === 'card' && <CreditCardIcon className="h-3.5 w-3.5" />}
                  {saleDetail.payment_method?.replace(/_/g, ' ')}
                </span>
              </div>
            </div>

            {/* Sale info */}
            <div className="grid grid-cols-2 sm:grid-cols-4 gap-3 bg-neutral-50 rounded-lg p-3 text-sm">
              <div><span className="text-neutral-500 text-xs block">Cashier</span>{saleDetail.cashier_name || '-'}</div>
              <div><span className="text-neutral-500 text-xs block">Customer</span>{saleDetail.customer_name || 'Walk-in'}</div>
              <div><span className="text-neutral-500 text-xs block">Amount Tendered</span>{formatCurrency(saleDetail.amount_tendered || 0)}</div>
              <div><span className="text-neutral-500 text-xs block">Change Given</span>{formatCurrency(saleDetail.change_given || 0)}</div>
            </div>

            {/* Items */}
            <div>
              <h4 className="text-sm font-semibold text-neutral-900 mb-2">Items ({saleDetail.items?.length || 0})</h4>
              <div className="border border-neutral-200 rounded-lg overflow-hidden">
                <table className="w-full text-sm">
                  <thead className="bg-neutral-50">
                    <tr>
                      <th className="text-left px-3 py-2 text-xs font-medium text-neutral-500">Product</th>
                      <th className="text-right px-3 py-2 text-xs font-medium text-neutral-500">Qty</th>
                      <th className="text-right px-3 py-2 text-xs font-medium text-neutral-500">Unit Price</th>
                      <th className="text-right px-3 py-2 text-xs font-medium text-neutral-500">Total</th>
                    </tr>
                  </thead>
                  <tbody>
                    {(saleDetail.items || []).map((item: any, idx: number) => (
                      <tr key={idx} className="border-t border-neutral-100">
                        <td className="px-3 py-2 font-medium">{item.product_name}</td>
                        <td className="px-3 py-2 text-right">{item.quantity}</td>
                        <td className="px-3 py-2 text-right">{formatCurrency(item.unit_price)}</td>
                        <td className="px-3 py-2 text-right font-medium">{formatCurrency(item.line_total)}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>

            {/* Totals */}
            <div className="bg-neutral-50 rounded-lg p-4 space-y-1.5">
              <div className="flex justify-between text-sm"><span className="text-neutral-500">Subtotal</span><span>{formatCurrency(saleDetail.subtotal)}</span></div>
              {parseFloat(saleDetail.discount_amount) > 0 && (
                <div className="flex justify-between text-sm text-red-600"><span>Discount</span><span>-{formatCurrency(saleDetail.discount_amount)}</span></div>
              )}
              <div className="flex justify-between text-xs text-neutral-400"><span>NHIL (2.5%)</span><span>{formatCurrency(saleDetail.nhil_amount)}</span></div>
              <div className="flex justify-between text-xs text-neutral-400"><span>GETFund (2.5%)</span><span>{formatCurrency(saleDetail.getfund_amount)}</span></div>
              <div className="flex justify-between text-xs text-neutral-400"><span>VAT (15%)</span><span>{formatCurrency(saleDetail.vat_amount)}</span></div>
              <div className="flex justify-between text-base font-bold border-t border-neutral-200 pt-2 mt-2"><span>Total</span><span>{formatCurrency(saleDetail.total_amount)}</span></div>
            </div>

            {/* Split payments */}
            {saleDetail.split_payments && saleDetail.split_payments.length > 0 && (
              <div>
                <h4 className="text-sm font-semibold mb-2">Split Payments</h4>
                <div className="space-y-1">
                  {saleDetail.split_payments.map((sp: any, i: number) => (
                    <div key={i} className="flex justify-between text-sm bg-neutral-50 rounded px-3 py-2">
                      <span className="capitalize">{sp.payment_method?.replace(/_/g, ' ')} {sp.reference ? `(${sp.reference})` : ''}</span>
                      <span className="font-medium">{formatCurrency(sp.amount)}</span>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Notes */}
            {saleDetail.notes && (
              <div className="text-sm">
                <span className="text-neutral-500 font-medium">Notes:</span>
                <p className="text-neutral-700 mt-1 whitespace-pre-line">{saleDetail.notes}</p>
              </div>
            )}

            {/* Actions */}
            <div className="flex items-center justify-between pt-4 border-t border-neutral-200">
              <div className="flex gap-2">
                {saleDetail.status !== 'refunded' && saleDetail.status !== 'voided' && (
                  <button className="btn-danger btn-sm flex items-center gap-1" onClick={() => openReturn(saleDetail)}>
                    <ArrowUturnLeftIcon className="h-4 w-4" /> Process Return
                  </button>
                )}
              </div>
              <div className="flex gap-2">
                <button className="btn-secondary btn-sm flex items-center gap-1" onClick={() => viewReceipt(saleDetail)}>
                  <PrinterIcon className="h-4 w-4" /> Print Receipt
                </button>
                <button className="btn-secondary btn-sm" onClick={() => { setShowDetail(false); setSaleDetail(null); }}>Close</button>
              </div>
            </div>
          </div>
        ) : null}
      </Modal>

      {/* Return Modal */}
      <Modal open={showReturn} onClose={() => setShowReturn(false)} title="Process Return / Refund" size="lg">
        {returnSale && (
          <div className="space-y-4">
            <div className="bg-amber-50 border border-amber-200 rounded-lg p-3 flex items-start gap-2">
              <ExclamationTriangleIcon className="h-5 w-5 text-amber-600 flex-shrink-0 mt-0.5" />
              <div className="text-sm">
                <p className="font-medium text-amber-800">Return for Sale {returnSale.sale_number || saleDetail?.sale_number}</p>
                <p className="text-amber-700 text-xs mt-0.5">Returned items will be added back to inventory and the refund amount will be deducted from sales.</p>
              </div>
            </div>

            {/* Return Type */}
            <div>
              <label className="label">Return Type</label>
              <div className="flex gap-2">
                <button
                  className={`flex-1 p-3 rounded-lg border-2 text-sm font-medium transition-all ${returnType === 'full' ? 'border-red-500 bg-red-50 text-red-700' : 'border-neutral-200 hover:border-neutral-300'}`}
                  onClick={() => setReturnType('full')}
                >
                  Full Return
                  <p className="text-xs font-normal mt-0.5 opacity-70">Return all items</p>
                </button>
                <button
                  className={`flex-1 p-3 rounded-lg border-2 text-sm font-medium transition-all ${returnType === 'partial' ? 'border-orange-500 bg-orange-50 text-orange-700' : 'border-neutral-200 hover:border-neutral-300'}`}
                  onClick={() => setReturnType('partial')}
                >
                  Partial Return
                  <p className="text-xs font-normal mt-0.5 opacity-70">Select specific items</p>
                </button>
              </div>
            </div>

            {/* Partial return items */}
            {returnType === 'partial' && (
              <div>
                <label className="label">Select Items to Return</label>
                <div className="border border-neutral-200 rounded-lg overflow-hidden">
                  {returnItems.map((item, idx) => (
                    <div key={idx} className={`flex items-center gap-3 px-3 py-2.5 border-b border-neutral-100 last:border-0 ${item.selected ? 'bg-red-50' : ''}`}>
                      <input
                        type="checkbox"
                        checked={item.selected}
                        onChange={() => toggleReturnItem(idx)}
                        className="w-4 h-4 rounded border-neutral-300"
                      />
                      <div className="flex-1 min-w-0">
                        <p className="text-sm font-medium truncate">{item.product_name}</p>
                        <p className="text-xs text-neutral-500">{formatCurrency(item.unit_price)} x {item.original_quantity} = {formatCurrency(item.line_total)}</p>
                      </div>
                      {item.selected && (
                        <div className="flex items-center gap-1">
                          <button onClick={() => updateReturnQty(idx, item.return_quantity - 1)} className="p-1 hover:bg-neutral-200 rounded">
                            <MinusIcon className="h-4 w-4" />
                          </button>
                          <input
                            type="number"
                            min="1"
                            max={item.original_quantity}
                            className="w-14 text-center text-sm font-bold border border-neutral-200 rounded px-1 py-0.5"
                            value={item.return_quantity}
                            onChange={(e) => updateReturnQty(idx, parseInt(e.target.value) || 0)}
                          />
                          <button onClick={() => updateReturnQty(idx, item.return_quantity + 1)} className="p-1 hover:bg-neutral-200 rounded">
                            <PlusIcon className="h-4 w-4" />
                          </button>
                          <span className="text-xs text-neutral-400 ml-1">/ {item.original_quantity}</span>
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Reason */}
            <div>
              <label className="label">Reason for Return *</label>
              <select className="select" value={returnReason} onChange={(e) => setReturnReason(e.target.value)}>
                <option value="">Select a reason</option>
                {RETURN_REASONS.map((r) => <option key={r} value={r}>{r}</option>)}
              </select>
              {returnReason === 'Other' && (
                <input className="input mt-2" placeholder="Describe the reason..."
                  value={customReason} onChange={(e) => setCustomReason(e.target.value)} />
              )}
            </div>

            {/* Refund summary */}
            <div className="bg-neutral-50 rounded-lg p-4">
              <h4 className="text-sm font-semibold text-neutral-900 mb-2">Refund Summary</h4>
              <div className="flex justify-between text-sm">
                <span className="text-neutral-500">Refund Subtotal</span>
                <span className="font-medium">{formatCurrency(returnSubtotal)}</span>
              </div>
              <div className="flex justify-between text-xs text-neutral-400 mt-1">
                <span>+ Proportional Tax</span>
                <span>Est. {formatCurrency(returnSubtotal * 0.2125)}</span>
              </div>
              <div className="flex justify-between text-lg font-bold border-t border-neutral-200 mt-2 pt-2">
                <span>Est. Total Refund</span>
                <span className="text-red-600">{formatCurrency(returnSubtotal * 1.2125)}</span>
              </div>
            </div>

            <div className="flex justify-end gap-3 pt-4 border-t border-neutral-200">
              <button className="btn-secondary" onClick={() => setShowReturn(false)}>Cancel</button>
              <button
                className="btn-danger flex items-center gap-2"
                onClick={handleReturn}
                disabled={processing}
              >
                <ArrowUturnLeftIcon className="h-4 w-4" />
                {processing ? 'Processing...' : 'Confirm Return'}
              </button>
            </div>
          </div>
        )}
      </Modal>

      {/* Returns History Modal */}
      <Modal open={showReturns} onClose={() => setShowReturns(false)} title="Returns History" size="xl">
        <div className="space-y-4">
          {returns.length === 0 ? (
            <EmptyState icon={<ArrowUturnLeftIcon className="h-12 w-12" />} title="No returns yet" description="Processed returns will appear here" />
          ) : (
            <div className="table-container">
              <table className="table">
                <thead>
                  <tr>
                    <th>Return #</th>
                    <th>Sale #</th>
                    <th>Product</th>
                    <th className="text-right">Qty</th>
                    <th className="text-right">Refund</th>
                    <th>Reason</th>
                    <th>Processed By</th>
                    <th>Date</th>
                  </tr>
                </thead>
                <tbody>
                  {returns.map((ret: any) => (
                    <tr key={ret.id}>
                      <td className="font-mono text-xs">{ret.return_number}</td>
                      <td className="font-mono text-xs text-blue-600">{ret.sale_number}</td>
                      <td className="text-sm">{ret.product_name}</td>
                      <td className="text-right">{ret.quantity_returned}</td>
                      <td className="text-right font-medium text-red-600">{formatCurrency(ret.refund_amount)}</td>
                      <td className="text-xs text-neutral-500 max-w-[150px] truncate">{ret.reason}</td>
                      <td className="text-sm">{ret.processed_by_name || '-'}</td>
                      <td className="text-xs">{new Date(ret.created_at).toLocaleDateString('en-GH', { day: 'numeric', month: 'short', hour: '2-digit', minute: '2-digit' })}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </div>
      </Modal>

      {/* Receipt Popup */}
      <ReceiptPopup open={showReceipt} onClose={() => { setShowReceipt(false); setReceiptData(null); }} receipt={receiptData} />
    </div>
  );
}
