'use client';

import React, { useState, useEffect, useRef } from 'react';
import { posAPI, inventoryAPI } from '@/lib/api';
import { formatCurrency } from '@/lib/utils';
import { useAuth } from '@/context/AuthContext';
import { PageLoading, Modal } from '@/components/ui';
import ReceiptPopup from '@/components/ui/ReceiptPopup';
import toast from 'react-hot-toast';
import {
  ShoppingCartIcon,
  MagnifyingGlassIcon,
  TrashIcon,
  PlusIcon,
  MinusIcon,
  BanknotesIcon,
  DevicePhoneMobileIcon,
  CreditCardIcon,
  ClockIcon,
} from '@heroicons/react/24/outline';

interface CartItem {
  product_id: string;
  name: string;
  price: number;
  quantity: number;
}

export default function POSPage() {
  const { user } = useAuth();
  const [session, setSession] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [cart, setCart] = useState<CartItem[]>([]);
  const [products, setProducts] = useState<any[]>([]);
  const [search, setSearch] = useState('');
  const [showPayment, setShowPayment] = useState(false);
  const [paymentMethod, setPaymentMethod] = useState('cash');
  const [amountTendered, setAmountTendered] = useState('');
  const [processing, setProcessing] = useState(false);
  const searchRef = useRef<HTMLInputElement>(null);

  // Open/Close session modals
  const [showOpenSession, setShowOpenSession] = useState(false);
  const [openingCash, setOpeningCash] = useState('0');
  const [showCloseSession, setShowCloseSession] = useState(false);
  const [closingCash, setClosingCash] = useState('0');

  // Terminals
  const [terminals, setTerminals] = useState<any[]>([]);
  const [selectedTerminalId, setSelectedTerminalId] = useState('');

  // Receipt popup
  const [showReceipt, setShowReceipt] = useState(false);
  const [receiptData, setReceiptData] = useState<any>(null);

  // Sales history
  const [posTab, setPosTab] = useState<'register' | 'sales'>('register');
  const [sales, setSales] = useState<any[]>([]);

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [sessionRes, prodRes, salesRes, termRes] = await Promise.all([
        posAPI.currentSession().catch(() => null),
        inventoryAPI.listProducts({ limit: 200 }),
        posAPI.listSales({ limit: 50 }).catch(() => null),
        posAPI.getTerminals().catch(() => null),
      ]);
      if (sessionRes?.data?.data) setSession(sessionRes.data.data);
      setProducts(prodRes.data?.data || []);
      setSales(Array.isArray(salesRes?.data?.data) ? salesRes.data.data : []);
      const termList = termRes?.data?.data || [];
      setTerminals(termList);
      if (termList.length > 0) setSelectedTerminalId(termList[0].id);
    } catch (err) { console.error(err); }
    setLoading(false);
  };

  const openSession = async () => {
    try {
      let terminalId = selectedTerminalId;

      // If no terminals exist, auto-create a default one
      if (!terminalId) {
        const { data: termData } = await posAPI.createTerminal({ terminal_name: 'POS-1', location: 'Main' });
        terminalId = termData?.data?.id;
        if (!terminalId) {
          toast.error('Failed to create POS terminal');
          return;
        }
        setTerminals(prev => [...prev, termData.data]);
        setSelectedTerminalId(terminalId);
      }

      const { data } = await posAPI.openSession({
        terminal_id: terminalId,
        opening_balance: parseFloat(openingCash || '0'),
      });
      setSession(data.data);
      setShowOpenSession(false);
      setOpeningCash('0');
      toast.success('POS session opened');
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error opening session');
    }
  };

  const closeSession = async () => {
    if (!session) return;
    try {
      await posAPI.closeSession(session.id, {
        closing_balance: parseFloat(closingCash || '0'),
      });
      setSession(null);
      setCart([]);
      setShowCloseSession(false);
      setClosingCash('0');
      toast.success('Session closed successfully');
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error closing session');
    }
  };

  const addToCart = (product: any) => {
    const existing = cart.find((c) => c.product_id === product.id);
    if (existing) {
      setCart(cart.map((c) => c.product_id === product.id ? { ...c, quantity: c.quantity + 1 } : c));
    } else {
      setCart([...cart, {
        product_id: product.id,
        name: product.name,
        price: parseFloat(product.selling_price),
        quantity: 1,
      }]);
    }
  };

  const updateQty = (productId: string, delta: number) => {
    setCart(cart.map((c) => {
      if (c.product_id === productId) {
        const newQty = c.quantity + delta;
        return newQty > 0 ? { ...c, quantity: newQty } : c;
      }
      return c;
    }));
  };

  const setQty = (productId: string, qty: number) => {
    if (qty < 1) qty = 1;
    setCart(cart.map((c) => c.product_id === productId ? { ...c, quantity: qty } : c));
  };

  const removeFromCart = (productId: string) => {
    setCart(cart.filter((c) => c.product_id !== productId));
  };

  const subtotal = cart.reduce((sum, c) => sum + c.price * c.quantity, 0);

  // Ghana composite tax calculation
  const nhil = subtotal * 0.025;
  const getfund = subtotal * 0.025;
  const vatBase = subtotal + nhil + getfund;
  const vat = vatBase * 0.15;
  const totalTax = nhil + getfund + vat;
  const grandTotal = subtotal + totalTax;

  const change = parseFloat(amountTendered || '0') - grandTotal;

  const completeSale = async () => {
    if (cart.length === 0) return;
    setProcessing(true);
    try {
      const saleData = {
        session_id: session.id,
        payment_method: paymentMethod,
        amount_tendered: paymentMethod === 'cash' ? parseFloat(amountTendered || '0') : grandTotal,
        items: cart.map((c) => ({
          product_id: c.product_id,
          quantity: c.quantity,
          unit_price: c.price,
        })),
      };
      const { data } = await posAPI.createSale(saleData);

      // Build receipt data for popup
      setReceiptData({
        receipt_number: data?.data?.sale_number || data?.data?.receipt_number || 'N/A',
        date: new Date().toLocaleString('en-GH', {
          year: 'numeric',
          month: 'short',
          day: 'numeric',
          hour: '2-digit',
          minute: '2-digit',
        }),
        items: cart.map((c) => ({
          name: c.name,
          quantity: c.quantity,
          unit_price: c.price,
          total: c.price * c.quantity,
        })),
        subtotal,
        nhil,
        getfund,
        vat,
        total_tax: totalTax,
        grand_total: grandTotal,
        payment_method: paymentMethod,
        amount_tendered: parseFloat(amountTendered || '0'),
        change: Math.max(0, change),
        company_name: user?.company?.name || 'CashBook ERP',
        cashier_name: `${user?.first_name || ''} ${user?.last_name || ''}`.trim(),
      });

      toast.success('Sale completed');
      setShowPayment(false);
      setShowReceipt(true);
      // Reload sales list
      posAPI.listSales({ limit: 50 }).then(res => setSales(res.data?.data || res.data || [])).catch(() => {});
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Sale failed');
    }
    setProcessing(false);
  };

  const handleReceiptClose = () => {
    setShowReceipt(false);
    setReceiptData(null);
    setCart([]);
    setAmountTendered('');
    searchRef.current?.focus();
  };

  const filteredProducts = products.filter((p: any) =>
    p.name?.toLowerCase().includes(search.toLowerCase()) ||
    p.sku?.toLowerCase().includes(search.toLowerCase()) ||
    p.barcode?.includes(search)
  );

  if (loading) return <PageLoading />;

  // If no session, show open session prompt
  if (!session) {
    return (
      <>
        <div className="flex items-center justify-center h-96">
          <div className="text-center">
            <ShoppingCartIcon className="h-16 w-16 text-neutral-300 mx-auto mb-4" />
            <h2 className="text-xl font-bold text-neutral-900 mb-2">Start a POS Session</h2>
            <p className="text-sm text-neutral-500 mb-6">
              Open a session to begin processing sales
            </p>
            <button
              className="btn-primary btn-lg"
              onClick={() => setShowOpenSession(true)}
            >
              Open Session
            </button>
          </div>
        </div>

        {/* Open Session Modal */}
        <Modal open={showOpenSession} onClose={() => setShowOpenSession(false)} title="Open POS Session">
          <div className="space-y-4">
            {terminals.length > 0 && (
              <div>
                <label className="label">Terminal</label>
                <select
                  className="select"
                  value={selectedTerminalId}
                  onChange={(e) => setSelectedTerminalId(e.target.value)}
                >
                  {terminals.map((t: any) => (
                    <option key={t.id} value={t.id}>{t.terminal_name} {t.location ? `(${t.location})` : ''}</option>
                  ))}
                </select>
              </div>
            )}
            {terminals.length === 0 && (
              <div className="p-3 bg-blue-50 rounded-md">
                <p className="text-xs text-blue-700">A default POS terminal will be created automatically.</p>
              </div>
            )}
            <div>
              <label className="label">Opening Cash Balance (GHS)</label>
              <input
                type="number"
                step="0.01"
                min="0"
                className="input text-lg font-semibold text-center"
                value={openingCash}
                onChange={(e) => setOpeningCash(e.target.value)}
                autoFocus
                placeholder="0.00"
              />
              <p className="text-xs text-neutral-400 mt-1">Count the cash in the register before starting</p>
            </div>
            <div className="flex justify-end gap-3 pt-2">
              <button className="btn-secondary" onClick={() => setShowOpenSession(false)}>
                Cancel
              </button>
              <button className="btn-primary" onClick={openSession}>
                Open Session
              </button>
            </div>
          </div>
        </Modal>
      </>
    );
  }

  return (
    <div className="space-y-4">
      {/* POS Tabs */}
      <div className="flex items-center justify-between">
        <div className="flex gap-4">
          <button
            className={`pb-2 text-sm font-medium border-b-2 transition-colors ${posTab === 'register' ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-neutral-500'}`}
            onClick={() => setPosTab('register')}
          >
            <ShoppingCartIcon className="h-4 w-4 inline mr-1.5" />Register
          </button>
          <button
            className={`pb-2 text-sm font-medium border-b-2 transition-colors ${posTab === 'sales' ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-neutral-500'}`}
            onClick={() => setPosTab('sales')}
          >
            <ClockIcon className="h-4 w-4 inline mr-1.5" />Sales History
          </button>
        </div>
        {session && (
          <div className="text-xs text-neutral-500">
            Session: <span className="font-medium text-neutral-700">{session.terminal_name || 'POS-1'}</span>
            {' '} | Sales: <span className="font-medium text-neutral-700">{session.sales_summary?.sale_count || 0}</span>
            {' '} | Total: <span className="font-medium text-neutral-700">{formatCurrency(session.sales_summary?.total || 0)}</span>
          </div>
        )}
      </div>

      {posTab === 'sales' ? (
        /* Sales History Tab */
        <div className="card">
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Sale #</th>
                  <th>Date</th>
                  <th>Cashier</th>
                  <th>Payment</th>
                  <th className="text-right">Subtotal</th>
                  <th className="text-right">Tax</th>
                  <th className="text-right">Total</th>
                  <th>Status</th>
                </tr>
              </thead>
              <tbody>
                {sales.length === 0 ? (
                  <tr><td colSpan={8} className="text-center text-neutral-400 py-8">No sales recorded yet</td></tr>
                ) : (
                  sales.map((sale: any) => (
                    <tr key={sale.id}>
                      <td className="font-mono text-xs font-medium">{sale.sale_number}</td>
                      <td className="text-xs">{new Date(sale.sale_date || sale.created_at).toLocaleString('en-GH', { day: 'numeric', month: 'short', hour: '2-digit', minute: '2-digit' })}</td>
                      <td className="text-sm">{sale.cashier_name || '-'}</td>
                      <td className="capitalize text-xs">{sale.payment_method?.replace('_', ' ')}</td>
                      <td className="text-right text-sm">{formatCurrency(sale.subtotal)}</td>
                      <td className="text-right text-xs text-neutral-500">{formatCurrency(sale.tax_amount)}</td>
                      <td className="text-right font-medium">{formatCurrency(sale.total_amount)}</td>
                      <td><span className={`inline-flex items-center px-2 py-0.5 rounded text-xs font-medium ${sale.status === 'completed' ? 'bg-green-50 text-green-700' : sale.status === 'refunded' ? 'bg-red-50 text-red-700' : 'bg-neutral-100 text-neutral-700'}`}>{sale.status}</span></td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      ) : (
      /* Register Tab */
    <div className="flex gap-6 h-[calc(100vh-10rem)]">
      {/* Left: Product Grid */}
      <div className="flex-1 flex flex-col min-w-0">
        <div className="flex items-center gap-4 mb-4">
          <div className="relative flex-1">
            <MagnifyingGlassIcon className="h-5 w-5 absolute left-3 top-2.5 text-gray-400" />
            <input
              ref={searchRef}
              className="input pl-10"
              placeholder="Search products or scan barcode..."
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              autoFocus
            />
          </div>
          <button
            className="btn-danger btn-sm"
            onClick={() => setShowCloseSession(true)}
          >
            Close Session
          </button>
        </div>

        <div className="flex-1 overflow-y-auto">
          <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-4 gap-3">
            {filteredProducts.map((product: any) => (
              <button
                key={product.id}
                onClick={() => addToCart(product)}
                className="card p-4 text-left hover:border-neutral-400 hover:shadow-md transition-all"
              >
                <p className="text-sm font-medium text-gray-900 truncate">{product.name}</p>
                <p className="text-xs text-gray-500 mt-1">{product.sku}</p>
                <p className="text-sm font-bold text-neutral-900 mt-2">{formatCurrency(product.selling_price)}</p>
                <p className="text-xs text-gray-400">Stock: {product.quantity_on_hand ?? product.stock_quantity}</p>
              </button>
            ))}
          </div>
        </div>
      </div>

      {/* Right: Cart */}
      <div className="w-96 flex flex-col bg-white rounded-xl border border-gray-200 shadow-sm">
        <div className="px-4 py-3 border-b border-gray-200">
          <h3 className="font-semibold text-gray-900 flex items-center gap-2">
            <ShoppingCartIcon className="h-5 w-5" />
            Cart ({cart.length} items)
          </h3>
        </div>

        {/* Cart items */}
        <div className="flex-1 overflow-y-auto p-4 space-y-3">
          {cart.length === 0 ? (
            <p className="text-sm text-gray-400 text-center py-8">Cart is empty</p>
          ) : (
            cart.map((item) => (
              <div key={item.product_id} className="flex items-center gap-3 bg-gray-50 rounded-lg p-3">
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium text-gray-900 truncate">{item.name}</p>
                  <p className="text-xs text-gray-500">{formatCurrency(item.price)} each</p>
                </div>
                <div className="flex items-center gap-1">
                  <button onClick={() => updateQty(item.product_id, -1)} className="p-1 hover:bg-gray-200 rounded">
                    <MinusIcon className="h-4 w-4" />
                  </button>
                  <input
                    type="number"
                    min="1"
                    className="w-12 text-sm font-bold text-center border border-neutral-200 rounded px-1 py-0.5 [appearance:textfield] [&::-webkit-outer-spin-button]:appearance-none [&::-webkit-inner-spin-button]:appearance-none"
                    value={item.quantity}
                    onChange={(e) => setQty(item.product_id, parseInt(e.target.value) || 1)}
                  />
                  <button onClick={() => updateQty(item.product_id, 1)} className="p-1 hover:bg-gray-200 rounded">
                    <PlusIcon className="h-4 w-4" />
                  </button>
                </div>
                <p className="text-sm font-bold w-20 text-right">{formatCurrency(item.price * item.quantity)}</p>
                <button onClick={() => removeFromCart(item.product_id)} className="text-red-400 hover:text-red-600">
                  <TrashIcon className="h-4 w-4" />
                </button>
              </div>
            ))
          )}
        </div>

        {/* Totals */}
        <div className="border-t border-neutral-200 p-4 space-y-2">
          <div className="flex justify-between text-sm">
            <span className="text-neutral-500">Subtotal</span>
            <span>{formatCurrency(subtotal)}</span>
          </div>
          <div className="flex justify-between text-xs text-neutral-400">
            <span>NHIL (2.5%)</span><span>{formatCurrency(nhil)}</span>
          </div>
          <div className="flex justify-between text-xs text-neutral-400">
            <span>GETFund (2.5%)</span><span>{formatCurrency(getfund)}</span>
          </div>
          <div className="flex justify-between text-xs text-neutral-400">
            <span>VAT (15%)</span><span>{formatCurrency(vat)}</span>
          </div>
          <div className="flex justify-between text-lg font-semibold border-t border-neutral-200 pt-2">
            <span>Total</span>
            <span className="text-neutral-900">{formatCurrency(grandTotal)}</span>
          </div>

          <button
            className="btn-primary w-full btn-lg mt-4"
            disabled={cart.length === 0}
            onClick={() => setShowPayment(true)}
          >
            Charge {formatCurrency(grandTotal)}
          </button>
        </div>
      </div>

      {/* Payment Modal */}
      <Modal open={showPayment} onClose={() => setShowPayment(false)} title="Payment">
        <div className="space-y-6">
          <div className="text-center">
            <p className="text-3xl font-bold text-gray-900">{formatCurrency(grandTotal)}</p>
            <p className="text-sm text-gray-500 mt-1">Amount Due</p>
          </div>

          {/* Payment method selection */}
          <div className="grid grid-cols-3 gap-3">
            {[
              { value: 'cash', label: 'Cash', icon: BanknotesIcon },
              { value: 'mobile_money', label: 'Mobile Money', icon: DevicePhoneMobileIcon },
              { value: 'card', label: 'Card', icon: CreditCardIcon },
            ].map((method) => (
              <button
                key={method.value}
                className={`p-4 rounded-xl border-2 text-center transition-all ${
                  paymentMethod === method.value
                    ? 'border-neutral-900 bg-neutral-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
                onClick={() => setPaymentMethod(method.value)}
              >
                <method.icon className="h-6 w-6 mx-auto mb-1" />
                <p className="text-xs font-medium">{method.label}</p>
              </button>
            ))}
          </div>

          {paymentMethod === 'cash' && (
            <div>
              <label className="label">Amount Tendered (GHS)</label>
              <input
                type="number"
                step="0.01"
                className="input text-lg text-center font-bold"
                value={amountTendered}
                onChange={(e) => setAmountTendered(e.target.value)}
                autoFocus
              />

              {/* Quick denomination buttons */}
              <div className="mt-3">
                <p className="text-xs text-neutral-500 font-medium mb-2">Quick Cash Denominations</p>
                <div className="grid grid-cols-4 gap-1.5">
                  {[200, 100, 50, 20, 10, 5, 2, 1].map((denom) => (
                    <button
                      key={denom}
                      type="button"
                      className="py-2 px-1 text-sm font-semibold rounded-lg border border-neutral-200 bg-white hover:bg-neutral-50 hover:border-neutral-400 transition-all text-neutral-800"
                      onClick={() => setAmountTendered((prev) => ((parseFloat(prev) || 0) + denom).toFixed(2))}
                    >
                      {denom}
                    </button>
                  ))}
                </div>
                <div className="grid grid-cols-3 gap-1.5 mt-1.5">
                  {[0.50, 0.20, 0.10].map((denom) => (
                    <button
                      key={denom}
                      type="button"
                      className="py-1.5 px-1 text-xs font-medium rounded-lg border border-neutral-200 bg-neutral-50 hover:bg-neutral-100 hover:border-neutral-400 transition-all text-neutral-600"
                      onClick={() => setAmountTendered((prev) => ((parseFloat(prev) || 0) + denom).toFixed(2))}
                    >
                      {denom.toFixed(2)}
                    </button>
                  ))}
                </div>
                <div className="flex gap-1.5 mt-2">
                  <button
                    type="button"
                    className="flex-1 py-1.5 text-xs font-medium rounded-lg border border-emerald-200 bg-emerald-50 hover:bg-emerald-100 text-emerald-700 transition-all"
                    onClick={() => setAmountTendered(Math.ceil(grandTotal).toFixed(2))}
                  >
                    Exact ({formatCurrency(Math.ceil(grandTotal))})
                  </button>
                  <button
                    type="button"
                    className="flex-1 py-1.5 text-xs font-medium rounded-lg border border-red-200 bg-red-50 hover:bg-red-100 text-red-600 transition-all"
                    onClick={() => setAmountTendered('')}
                  >
                    Clear
                  </button>
                </div>
              </div>

              {change >= 0 && amountTendered && (
                <div className="mt-3 p-3 bg-emerald-50 rounded-md text-center">
                  <p className="text-xs text-emerald-600 font-medium">Change Due</p>
                  <p className="text-xl font-bold text-emerald-700">
                    {formatCurrency(change)}
                  </p>
                </div>
              )}
              {change < 0 && amountTendered && (
                <div className="mt-3 p-3 bg-red-50 rounded-md text-center">
                  <p className="text-xs text-red-600 font-medium">Insufficient</p>
                  <p className="text-xl font-bold text-red-700">
                    -{formatCurrency(Math.abs(change))}
                  </p>
                </div>
              )}
            </div>
          )}

          <button
            className="btn-success w-full btn-lg"
            onClick={completeSale}
            disabled={processing || (paymentMethod === 'cash' && change < 0)}
          >
            {processing ? 'Processing...' : 'Complete Sale'}
          </button>
        </div>
      </Modal>

      {/* Close Session Modal */}
      <Modal
        open={showCloseSession}
        onClose={() => setShowCloseSession(false)}
        title="Close POS Session"
      >
        <div className="space-y-4">
          <div>
            <label className="label">Closing Cash Count (GHS)</label>
            <input
              type="number"
              step="0.01"
              min="0"
              className="input text-lg font-semibold text-center"
              value={closingCash}
              onChange={(e) => setClosingCash(e.target.value)}
              autoFocus
              placeholder="0.00"
            />
            <p className="text-xs text-neutral-400 mt-1">
              Count all cash in the register before closing
            </p>
          </div>
          <div className="flex justify-end gap-3 pt-2">
            <button
              className="btn-secondary"
              onClick={() => setShowCloseSession(false)}
            >
              Cancel
            </button>
            <button className="btn-danger" onClick={closeSession}>
              Close Session
            </button>
          </div>
        </div>
      </Modal>

      {/* Receipt Popup */}
      <ReceiptPopup
        open={showReceipt}
        onClose={handleReceiptClose}
        receipt={receiptData}
      />
    </div>
      )}
    </div>
  );
}
