'use client';

import React, { useState } from 'react';
import { taxAPI } from '@/lib/api';
import { formatCurrency } from '@/lib/utils';
import { StatCard, Modal, StatusBadge } from '@/components/ui';
import FileAttachments from '@/components/ui/FileAttachments';
import DatePicker from '@/components/ui/DatePicker';
import toast from 'react-hot-toast';
import { CalculatorIcon, PlusIcon } from '@heroicons/react/24/outline';
import { useAuth } from '@/context/AuthContext';

export default function TaxPage() {
  const { user } = useAuth();
  const [tab, setTab] = useState<'calculator' | 'paye' | 'filings'>('calculator');
  const [amount, setAmount] = useState('');
  const [scheme, setScheme] = useState('standard');
  const [result, setResult] = useState<any>(null);

  const [payeIncome, setPayeIncome] = useState('');
  const [payeResult, setPayeResult] = useState<any>(null);

  const [filings, setFilings] = useState<any[]>([]);
  const [showFilingModal, setShowFilingModal] = useState(false);
  const [filingForm, setFilingForm] = useState({
    tax_type: 'vat',
    period_start: '',
    period_end: '',
  });

  const calculateTax = async () => {
    if (!amount) return;
    try {
      const { data } = await taxAPI.calculateTax({ amount: parseFloat(amount), scheme });
      setResult(data.data);
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const calculatePAYE = async () => {
    if (!payeIncome) return;
    try {
      const { data } = await taxAPI.calculatePAYE({ monthly_income: parseFloat(payeIncome) });
      setPayeResult(data.data);
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const loadFilings = async () => {
    try {
      const { data } = await taxAPI.getFilings();
      setFilings(data?.data || []);
    } catch (err) { console.error(err); }
  };

  const createFiling = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await taxAPI.createFiling(filingForm);
      toast.success('Tax filing created');
      setShowFilingModal(false);
      loadFilings();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  React.useEffect(() => {
    if (tab === 'filings') loadFilings();
  }, [tab]);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Tax Compliance</h1>
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="flex gap-6">
          {[
            { key: 'calculator', label: 'VAT Calculator' },
            { key: 'paye', label: 'PAYE Calculator' },
            { key: 'filings', label: 'Tax Filings' },
          ].map((t) => (
            <button key={t.key}
              className={`pb-3 text-sm font-medium border-b-2 transition-colors ${
                tab === t.key ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-gray-500'
              }`}
              onClick={() => setTab(t.key as any)}
            >{t.label}</button>
          ))}
        </nav>
      </div>

      {/* VAT Calculator */}
      {tab === 'calculator' && (
        <div className="grid lg:grid-cols-2 gap-8">
          <div className="card p-6">
            <h3 className="text-lg font-semibold mb-4">Ghana Composite Tax Calculator</h3>
            <div className="space-y-4">
              <div>
                <label className="label">Taxable Amount (GHS)</label>
                <input type="number" step="0.01" className="input text-lg" value={amount}
                  onChange={(e) => setAmount(e.target.value)} placeholder="Enter amount" />
              </div>
              <div>
                <label className="label">VAT Scheme</label>
                <select className="select" value={scheme} onChange={(e) => setScheme(e.target.value)}>
                  <option value="standard">Standard Rate (15%)</option>
                  <option value="flat_rate">Flat Rate (4%)</option>
                </select>
              </div>
              <button className="btn-primary w-full" onClick={calculateTax}>Calculate Tax</button>
            </div>
          </div>

          {result && (
            <div className="card p-6">
              <h3 className="text-lg font-semibold mb-4">Tax Breakdown</h3>
              <div className="space-y-3">
                <div className="flex justify-between py-2 border-b">
                  <span className="text-gray-600">Taxable Amount</span>
                  <span className="font-medium">{formatCurrency(result.taxable_amount)}</span>
                </div>
                {result.scheme === 'standard' && (
                  <>
                    <div className="flex justify-between py-2 border-b border-neutral-200">
                      <span className="text-neutral-600">NHIL (2.5%)</span>
                      <span>{formatCurrency(result.nhil?.amount || 0)}</span>
                    </div>
                    <div className="flex justify-between py-2 border-b border-neutral-200">
                      <span className="text-neutral-600">GETFund (2.5%)</span>
                      <span>{formatCurrency(result.getfund?.amount || 0)}</span>
                    </div>
                    <div className="flex justify-between py-2 border-b border-neutral-200">
                      <span className="text-neutral-600">VAT (15%)</span>
                      <span>{formatCurrency(result.vat?.amount || 0)}</span>
                    </div>
                  </>
                )}
                {result.scheme === 'flat_rate' && (
                  <div className="flex justify-between py-2 border-b border-neutral-200">
                    <span className="text-neutral-600">VAT Flat Rate (4%)</span>
                    <span>{formatCurrency(result.vat_amount || 0)}</span>
                  </div>
                )}
                <div className="flex justify-between py-2 border-b border-neutral-200 font-medium">
                  <span>Total Tax</span>
                  <span className="text-red-600">{formatCurrency(result.total_tax || result.vat_amount || 0)}</span>
                </div>
                <div className="flex justify-between py-2 text-lg font-semibold">
                  <span>Grand Total</span>
                  <span className="text-neutral-900">{formatCurrency(result.grand_total || result.total)}</span>
                </div>
                {result.effective_tax_rate && (
                  <p className="text-sm text-gray-500 text-center">Effective tax rate: {result.effective_tax_rate}</p>
                )}
              </div>
            </div>
          )}
        </div>
      )}

      {/* PAYE Calculator */}
      {tab === 'paye' && (
        <div className="grid lg:grid-cols-2 gap-8">
          <div className="card p-6">
            <h3 className="text-lg font-semibold mb-4">Ghana PAYE Tax Calculator</h3>
            <div className="space-y-4">
              <div>
                <label className="label">Monthly Gross Salary (GHS)</label>
                <input type="number" step="0.01" className="input text-lg" value={payeIncome}
                  onChange={(e) => setPayeIncome(e.target.value)} placeholder="Enter monthly salary" />
              </div>
              <button className="btn-primary w-full" onClick={calculatePAYE}>Calculate PAYE</button>
            </div>
          </div>

          {payeResult && (
            <div className="card p-6">
              <h3 className="text-lg font-semibold mb-4">PAYE Breakdown</h3>
              <div className="space-y-3">
                <div className="flex justify-between py-2 border-b">
                  <span className="text-gray-600">Monthly Income</span>
                  <span className="font-medium">{formatCurrency(payeResult.monthly_income)}</span>
                </div>
                <div className="flex justify-between py-2 border-b">
                  <span className="text-gray-600">SSNIT Deduction (5.5%)</span>
                  <span className="text-red-600">-{formatCurrency(payeResult.ssnit_deduction)}</span>
                </div>
                <div className="flex justify-between py-2 border-b font-medium">
                  <span>Taxable Income</span>
                  <span>{formatCurrency(payeResult.taxable_income)}</span>
                </div>

                <h4 className="text-sm font-semibold text-gray-700 mt-4">Tax Brackets</h4>
                {payeResult.breakdown?.map((b: any, i: number) => (
                  <div key={i} className="flex justify-between py-1 text-sm">
                    <span className="text-gray-500">{b.bracket} @ {b.rate}</span>
                    <span>{formatCurrency(b.tax)}</span>
                  </div>
                ))}

                <div className="flex justify-between py-2 border-t font-medium">
                  <span>PAYE Tax</span>
                  <span className="text-red-600">{formatCurrency(payeResult.paye_tax)}</span>
                </div>
                <div className="flex justify-between py-2 text-lg font-bold bg-emerald-50 rounded-md px-3">
                  <span>Net Take-Home</span>
                  <span className="text-green-600">{formatCurrency(payeResult.net_income)}</span>
                </div>
                <p className="text-sm text-gray-500 text-center">Effective rate: {payeResult.effective_rate}</p>
              </div>
            </div>
          )}
        </div>
      )}

      {/* Tax Filings */}
      {tab === 'filings' && (
        <>
          <div className="flex justify-end">
            <button className="btn-primary" onClick={() => setShowFilingModal(true)}>
              <PlusIcon className="h-4 w-4 mr-1" /> New Filing
            </button>
          </div>
          <div className="card">
            <div className="table-container">
              <table className="table">
                <thead>
                  <tr>
                    <th>Tax Type</th>
                    <th>Period</th>
                    <th>Status</th>
                    <th className="text-right">Output Tax</th>
                    <th className="text-right">Input Tax</th>
                    <th className="text-right">Tax Payable</th>
                  </tr>
                </thead>
                <tbody>
                  {filings.map((f: any) => (
                    <tr key={f.id}>
                      <td className="uppercase font-medium">{f.tax_type}</td>
                      <td>{f.filing_period}</td>
                      <td><StatusBadge status={f.status} /></td>
                      <td className="text-right">{formatCurrency(f.output_tax)}</td>
                      <td className="text-right">{formatCurrency(f.input_tax)}</td>
                      <td className="text-right font-bold">{formatCurrency(f.tax_payable)}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          <Modal open={showFilingModal} onClose={() => setShowFilingModal(false)} title="Create Tax Filing">
            <form onSubmit={createFiling} className="space-y-4">
              <div>
                <label className="label">Tax Type</label>
                <select className="select" value={filingForm.tax_type} onChange={(e) => setFilingForm({ ...filingForm, tax_type: e.target.value })}>
                  <option value="vat">VAT / NHIL / GETFund</option>
                  <option value="paye">PAYE</option>
                  <option value="withholding">Withholding Tax</option>
                </select>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="label">Period Start</label>
                  <DatePicker value={filingForm.period_start} onChange={(v) => setFilingForm({ ...filingForm, period_start: v })} required />
                </div>
                <div>
                  <label className="label">Period End</label>
                  <DatePicker value={filingForm.period_end} onChange={(v) => setFilingForm({ ...filingForm, period_end: v })} required />
                </div>
              </div>
              <div className="bg-neutral-50 border border-neutral-200 rounded-md p-3 text-sm text-neutral-700">
                Tax amounts will be auto-calculated from your invoices, POS sales, and payroll data.
              </div>
              <div className="flex justify-end gap-3">
                <button type="button" className="btn-secondary" onClick={() => setShowFilingModal(false)}>Cancel</button>
                <button type="submit" className="btn-primary">Create Filing</button>
              </div>
            </form>
          </Modal>
        </>
      )}

      {/* Tax Documents */}
      {user?.company?.id && (
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Tax Documents</h3>
            <p className="text-xs text-gray-500 mt-1">Attach tax-related documents (VAT returns, GRA filings, receipts, etc.)</p>
          </div>
          <div className="card-body">
            <FileAttachments entityType="tax" entityId={user.company.id} />
          </div>
        </div>
      )}
    </div>
  );
}
