'use client';

import React, { useState, useEffect } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import { useAuth } from '@/context/AuthContext';
import toast from 'react-hot-toast';
import { BookOpenIcon, CalculatorIcon, ClipboardDocumentListIcon, ChartBarSquareIcon, DocumentTextIcon, BanknotesIcon } from '@heroicons/react/24/outline';

export default function LoginPage() {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [loading, setLoading] = useState(false);
  const { login, isAuthenticated, loading: authLoading } = useAuth();
  const router = useRouter();

  // Redirect to dashboard if already authenticated
  useEffect(() => {
    if (!authLoading && isAuthenticated) {
      router.replace('/dashboard');
    }
  }, [authLoading, isAuthenticated, router]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      await login(email, password);
      toast.success('Welcome back!');
      router.push('/dashboard');
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Invalid credentials');
    }
    setLoading(false);
  };

  return (
    <div className="min-h-screen flex relative overflow-hidden">

      {/* ---- LEFT: Bookkeeping-themed background panel ---- */}
      <div className="hidden lg:flex lg:w-[55%] bg-neutral-900 relative flex-col justify-between p-12 overflow-hidden">
        {/* Grid pattern overlay */}
        <div className="absolute inset-0 opacity-[0.04]"
          style={{
            backgroundImage: 'linear-gradient(rgba(255,255,255,.5) 1px, transparent 1px), linear-gradient(90deg, rgba(255,255,255,.5) 1px, transparent 1px)',
            backgroundSize: '40px 40px',
          }}
        />

        {/* Floating accounting icons */}
        <div className="absolute top-16 right-16 opacity-10">
          <CalculatorIcon className="h-32 w-32 text-white" />
        </div>
        <div className="absolute bottom-32 left-16 opacity-[0.07]">
          <ClipboardDocumentListIcon className="h-28 w-28 text-white" />
        </div>
        <div className="absolute top-1/2 right-1/3 -translate-y-1/2 opacity-[0.06]">
          <ChartBarSquareIcon className="h-24 w-24 text-white" />
        </div>
        <div className="absolute bottom-16 right-32 opacity-[0.05]">
          <BanknotesIcon className="h-20 w-20 text-white" />
        </div>

        {/* Logo */}
        <div className="relative z-10">
          <Link href="/" className="flex items-center gap-2.5">
            <div className="h-9 w-9 bg-white rounded-lg flex items-center justify-center">
              <BookOpenIcon className="h-5 w-5 text-neutral-900" />
            </div>
            <span className="text-xl font-semibold text-white tracking-tight">CashBook</span>
          </Link>
        </div>

        {/* Center content — ledger mockup */}
        <div className="relative z-10 flex-1 flex items-center justify-center">
          <div className="max-w-sm w-full">
            {/* Ledger-style card */}
            <div className="bg-white/[0.06] backdrop-blur-sm rounded-2xl border border-white/10 p-6 shadow-2xl">
              <div className="flex items-center gap-2 mb-4">
                <DocumentTextIcon className="h-5 w-5 text-emerald-400" />
                <span className="text-sm font-medium text-white">General Ledger</span>
              </div>
              {/* Ledger rows */}
              <div className="space-y-0">
                {/* Header */}
                <div className="grid grid-cols-4 gap-2 text-[10px] text-neutral-400 uppercase tracking-wider border-b border-white/10 pb-2 mb-2">
                  <span>Date</span><span>Description</span><span className="text-right">Debit</span><span className="text-right">Credit</span>
                </div>
                {[
                  { date: 'Feb 01', desc: 'Sales Revenue', dr: '12,500', cr: '' },
                  { date: 'Feb 03', desc: 'Office Rent', dr: '', cr: '2,800' },
                  { date: 'Feb 07', desc: 'Client Payment', dr: '8,340', cr: '' },
                  { date: 'Feb 10', desc: 'Payroll - SSNIT', dr: '', cr: '1,250' },
                  { date: 'Feb 14', desc: 'POS Sales', dr: '4,680', cr: '' },
                ].map((row, i) => (
                  <div key={i} className="grid grid-cols-4 gap-2 text-[11px] py-1.5 border-b border-white/5 last:border-0">
                    <span className="text-neutral-500">{row.date}</span>
                    <span className="text-neutral-300">{row.desc}</span>
                    <span className={`text-right ${row.dr ? 'text-emerald-400' : 'text-neutral-600'}`}>{row.dr ? `GHS ${row.dr}` : '—'}</span>
                    <span className={`text-right ${row.cr ? 'text-red-400' : 'text-neutral-600'}`}>{row.cr ? `GHS ${row.cr}` : '—'}</span>
                  </div>
                ))}
              </div>
              {/* Balance */}
              <div className="mt-3 pt-3 border-t border-white/10 flex justify-between items-center">
                <span className="text-[10px] text-neutral-500 uppercase tracking-wider">Balance</span>
                <span className="text-sm font-bold text-emerald-400">GHS 21,470.00</span>
              </div>
            </div>

            {/* Mini stat cards below */}
            <div className="grid grid-cols-3 gap-3 mt-4">
              {[
                { label: 'Accounts', value: '42' },
                { label: 'Invoices', value: '128' },
                { label: 'Tax Filed', value: '12' },
              ].map((stat) => (
                <div key={stat.label} className="bg-white/[0.05] rounded-lg border border-white/10 px-3 py-2 text-center">
                  <div className="text-sm font-bold text-white">{stat.value}</div>
                  <div className="text-[9px] text-neutral-500">{stat.label}</div>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Bottom tagline */}
        <div className="relative z-10">
          <p className="text-sm text-neutral-400 leading-relaxed max-w-xs">
            Professional bookkeeping, accounting &amp; tax compliance — built for Ghanaian businesses.
          </p>
        </div>
      </div>

      {/* ---- RIGHT: Login form ---- */}
      <div className="flex-1 flex items-center justify-center px-6 py-12 bg-white">
        <div className="max-w-sm w-full">
          {/* Mobile logo (hidden on lg) */}
          <div className="text-center mb-8 lg:hidden">
            <Link href="/" className="inline-flex items-center gap-2">
              <div className="h-10 w-10 bg-neutral-900 rounded-lg flex items-center justify-center">
                <BookOpenIcon className="h-6 w-6 text-white" />
              </div>
              <span className="text-2xl font-semibold text-neutral-900">CashBook</span>
            </Link>
          </div>

          <div className="mb-8">
            <h1 className="text-2xl font-bold text-neutral-900 tracking-tight">Welcome back</h1>
            <p className="mt-1 text-sm text-neutral-500">Sign in to your account to continue</p>
          </div>

          <form onSubmit={handleSubmit} className="space-y-5">
            <div>
              <label className="label">Email Address</label>
              <input
                type="email"
                className="input"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                placeholder="you@company.com"
                required
              />
            </div>

            <div>
              <label className="label">Password</label>
              <input
                type="password"
                className="input"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                placeholder="Enter your password"
                required
              />
            </div>

            <button type="submit" className="btn-primary w-full" disabled={loading}>
              {loading ? 'Signing in...' : 'Sign In'}
            </button>
          </form>

          <p className="text-center text-sm text-neutral-500 mt-8">
            Don&apos;t have an account?{' '}
            <Link href="/register" className="text-neutral-900 font-medium hover:underline">
              Create one free
            </Link>
          </p>
        </div>
      </div>
    </div>
  );
}
