'use client';

import React, { useState, useEffect, useCallback } from 'react';
import Link from 'next/link';
import useEmblaCarousel from 'embla-carousel-react';
import Autoplay from 'embla-carousel-autoplay';
import {
  BookOpenIcon,
  ChartBarIcon,
  ShoppingCartIcon,
  DocumentTextIcon,
  BanknotesIcon,
  ShieldCheckIcon,
  CubeIcon,
  UserGroupIcon,
  Bars3Icon,
  XMarkIcon,
  CheckIcon,
  ArrowRightIcon,
} from '@heroicons/react/24/outline';
import { CheckCircleIcon, StarIcon } from '@heroicons/react/24/solid';

/* ---- DATA ---- */

const features = [
  {
    name: 'Double-Entry Accounting',
    description: 'Full chart of accounts, journal entries, trial balance, and general ledger built on Ghana-standard account structures.',
    icon: BookOpenIcon,
  },
  {
    name: 'Professional Invoicing',
    description: 'Create and send beautiful invoices with automatic Ghana VAT, NHIL, and GETFund levy calculations.',
    icon: DocumentTextIcon,
  },
  {
    name: 'Point of Sale',
    description: 'Integrated POS terminal with barcode scanning, split payments, mobile money, and real-time inventory sync.',
    icon: ShoppingCartIcon,
  },
  {
    name: 'Payroll & HR',
    description: 'Automated PAYE with all 7 Ghana tax brackets, SSNIT Tier 1/2/3 deductions, and instant payslip generation.',
    icon: BanknotesIcon,
  },
  {
    name: 'Financial Reports',
    description: 'Income Statement, Balance Sheet, Cash Flow, AR Aging, and tax summaries - exportable as PDF and Excel.',
    icon: ChartBarIcon,
  },
  {
    name: 'GRA Tax Compliance',
    description: 'Built-in VAT, NHIL, GETFund, withholding tax tracking, and GRA e-filing ready infrastructure.',
    icon: ShieldCheckIcon,
  },
  {
    name: 'Inventory Management',
    description: 'Track stock levels, set auto-reorder points, manage categories, and view real-time movement history.',
    icon: CubeIcon,
  },
  {
    name: 'Customer & Suppliers',
    description: 'Maintain a complete CRM with contact profiles, transaction history, and outstanding balance tracking.',
    icon: UserGroupIcon,
  },
];

const pricingPlans = [
  {
    name: 'Starter',
    price: 'Free',
    period: '',
    description: 'Perfect for solo entrepreneurs just getting started.',
    features: [
      'Up to 50 invoices/month',
      'Basic POS terminal',
      'Single user',
      'Income & expense tracking',
      'Ghana tax calculations',
      'Email support',
    ],
    cta: 'Start Free',
    popular: false,
  },
  {
    name: 'Business',
    price: 'GHS 99',
    period: '/month',
    description: 'For growing businesses that need full ERP capabilities.',
    features: [
      'Unlimited invoices',
      'Multi-terminal POS',
      'Up to 5 users',
      'Full accounting suite',
      'Payroll management',
      'Inventory tracking',
      'Financial reports (PDF/Excel)',
      'File attachments',
      'Priority support',
    ],
    cta: 'Start Free Trial',
    popular: true,
  },
  {
    name: 'Enterprise',
    price: 'GHS 299',
    period: '/month',
    description: 'For established businesses with advanced needs.',
    features: [
      'Everything in Business',
      'Unlimited users',
      'Multi-branch support',
      'Fixed asset management',
      'Advanced audit trail',
      'API access',
      'Custom integrations',
      'Dedicated account manager',
      'SLA guarantee',
    ],
    cta: 'Contact Sales',
    popular: false,
  },
];

const testimonials = [
  {
    name: 'Kwame Asante',
    role: 'CEO, Asante Trading Co.',
    text: 'CashBook transformed how we manage our books. Tax filing used to take days - now it takes minutes.',
  },
  {
    name: 'Abena Mensah',
    role: 'Owner, Mensah Supermarket',
    text: 'The POS system is incredibly fast. Our checkout lines are shorter and inventory is always accurate.',
  },
  {
    name: 'Yaw Boateng',
    role: 'CFO, GreenFields Ltd.',
    text: 'Finally, an ERP that understands Ghanaian tax laws. The PAYE and SSNIT calculations are spot-on.',
  },
];

const stats = [
  { value: '2,500+', label: 'Businesses' },
  { value: 'GHS 180M+', label: 'Transactions Processed' },
  { value: '99.9%', label: 'Uptime' },
  { value: '16', label: 'Regions Covered' },
];

/* ---- Dashboard Mockup Screenshots ---- */

function DashboardMockup({ variant }: { variant: number }) {
  const screens = [
    // Dashboard overview
    (
      <div key="dash" className="w-full h-full bg-[#fafafa] flex">
        <div className="w-[180px] bg-[#0a0a0a] p-4 flex flex-col gap-1">
          <div className="flex items-center gap-2 mb-6">
            <div className="w-6 h-6 rounded bg-white/20" />
            <div className="text-white text-xs font-semibold">CashBook</div>
          </div>
          {['Dashboard', 'Invoices', 'POS', 'Accounting', 'Payroll', 'Inventory', 'Reports', 'Settings'].map((item, i) => (
            <div key={item} className={`px-3 py-1.5 rounded text-[10px] ${i === 0 ? 'bg-white/10 text-white' : 'text-neutral-500'}`}>{item}</div>
          ))}
        </div>
        <div className="flex-1 p-5">
          <div className="text-sm font-semibold text-neutral-800 mb-4">Dashboard Overview</div>
          <div className="grid grid-cols-4 gap-3 mb-4">
            {[
              { l: 'Revenue', v: 'GHS 45,230', c: 'text-emerald-600' },
              { l: 'Expenses', v: 'GHS 18,400', c: 'text-red-500' },
              { l: 'Net Profit', v: 'GHS 26,830', c: 'text-emerald-600' },
              { l: 'Outstanding', v: 'GHS 8,500', c: 'text-amber-600' },
            ].map((card) => (
              <div key={card.l} className="bg-white rounded-lg border border-neutral-200 p-3">
                <div className="text-[9px] text-neutral-500 mb-1">{card.l}</div>
                <div className={`text-sm font-bold ${card.c}`}>{card.v}</div>
              </div>
            ))}
          </div>
          <div className="grid grid-cols-2 gap-3">
            <div className="bg-white rounded-lg border border-neutral-200 p-3 h-32">
              <div className="text-[10px] font-medium text-neutral-700 mb-2">Revenue Trend</div>
              <svg viewBox="0 0 200 60" className="w-full h-16">
                <polyline fill="none" stroke="#10b981" strokeWidth="2" points="0,50 30,40 60,45 90,30 120,25 150,15 180,20 200,10" />
                <polyline fill="rgba(16,185,129,0.1)" stroke="none" points="0,60 0,50 30,40 60,45 90,30 120,25 150,15 180,20 200,10 200,60" />
              </svg>
            </div>
            <div className="bg-white rounded-lg border border-neutral-200 p-3 h-32">
              <div className="text-[10px] font-medium text-neutral-700 mb-2">Recent Transactions</div>
              {['Invoice #INV-2026-0012', 'POS Sale #POS-0045', 'Expense: Office Rent'].map((t, i) => (
                <div key={t} className="flex justify-between text-[9px] py-1 border-b border-neutral-100 last:border-0">
                  <span className="text-neutral-600">{t}</span>
                  <span className={i === 2 ? 'text-red-500' : 'text-emerald-600'}>{i === 2 ? '-GHS 2,500' : `+GHS ${(i + 1) * 340}`}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    ),
    // POS screen
    (
      <div key="pos" className="w-full h-full bg-[#fafafa] flex">
        <div className="w-[180px] bg-[#0a0a0a] p-4 flex flex-col gap-1">
          <div className="flex items-center gap-2 mb-6">
            <div className="w-6 h-6 rounded bg-white/20" />
            <div className="text-white text-xs font-semibold">CashBook</div>
          </div>
          {['Dashboard', 'Invoices', 'POS', 'Accounting', 'Payroll', 'Inventory', 'Reports', 'Settings'].map((item, i) => (
            <div key={item} className={`px-3 py-1.5 rounded text-[10px] ${i === 2 ? 'bg-white/10 text-white' : 'text-neutral-500'}`}>{item}</div>
          ))}
        </div>
        <div className="flex-1 p-5 flex gap-4">
          <div className="flex-1">
            <div className="text-sm font-semibold text-neutral-800 mb-3">Point of Sale</div>
            <div className="grid grid-cols-3 gap-2">
              {['Rice (5kg)', 'Cooking Oil', 'Sugar (1kg)', 'Milk Powder', 'Bread', 'Water (1.5L)', 'Detergent', 'Tissue Pack', 'Batteries'].map((p, i) => (
                <div key={p} className="bg-white rounded-lg border border-neutral-200 p-2 text-center">
                  <div className="w-8 h-8 mx-auto mb-1 rounded bg-neutral-100 flex items-center justify-center text-[8px]">
                    <CubeIcon className="h-4 w-4 text-neutral-400" />
                  </div>
                  <div className="text-[9px] font-medium text-neutral-800 truncate">{p}</div>
                  <div className="text-[8px] text-neutral-500">GHS {((i + 1) * 5.5).toFixed(2)}</div>
                </div>
              ))}
            </div>
          </div>
          <div className="w-44 bg-white rounded-lg border border-neutral-200 p-3">
            <div className="text-[10px] font-semibold text-neutral-800 mb-2">Cart (3 items)</div>
            {['Rice (5kg)', 'Cooking Oil', 'Sugar (1kg)'].map((item, i) => (
              <div key={item} className="flex justify-between text-[8px] py-1 border-b border-neutral-100">
                <span className="text-neutral-600">{item}</span>
                <span className="font-medium">GHS {((i + 1) * 5.5).toFixed(2)}</span>
              </div>
            ))}
            <div className="flex justify-between text-[10px] font-bold mt-2 pt-1 border-t border-neutral-300">
              <span>Total</span><span>GHS 33.00</span>
            </div>
            <div className="mt-2 bg-neutral-900 text-white text-[9px] font-medium text-center py-1.5 rounded">Charge GHS 33.00</div>
          </div>
        </div>
      </div>
    ),
    // Reports screen
    (
      <div key="reports" className="w-full h-full bg-[#fafafa] flex">
        <div className="w-[180px] bg-[#0a0a0a] p-4 flex flex-col gap-1">
          <div className="flex items-center gap-2 mb-6">
            <div className="w-6 h-6 rounded bg-white/20" />
            <div className="text-white text-xs font-semibold">CashBook</div>
          </div>
          {['Dashboard', 'Invoices', 'POS', 'Accounting', 'Payroll', 'Inventory', 'Reports', 'Settings'].map((item, i) => (
            <div key={item} className={`px-3 py-1.5 rounded text-[10px] ${i === 6 ? 'bg-white/10 text-white' : 'text-neutral-500'}`}>{item}</div>
          ))}
        </div>
        <div className="flex-1 p-5">
          <div className="text-sm font-semibold text-neutral-800 mb-1">Income Statement</div>
          <div className="text-[9px] text-neutral-500 mb-4">January 2026 - February 2026</div>
          <div className="bg-white rounded-lg border border-neutral-200 p-4 max-w-md">
            <div className="text-[10px] font-bold text-neutral-700 border-b pb-1 mb-2">Revenue</div>
            {[{ n: 'Sales Revenue', v: '45,230.00' }, { n: 'Service Income', v: '8,400.00' }].map(r => (
              <div key={r.n} className="flex justify-between text-[9px] py-0.5">
                <span className="text-neutral-600 pl-3">{r.n}</span>
                <span>GHS {r.v}</span>
              </div>
            ))}
            <div className="flex justify-between text-[10px] font-bold border-t mt-1 pt-1">
              <span>Total Revenue</span><span>GHS 53,630.00</span>
            </div>

            <div className="text-[10px] font-bold text-neutral-700 border-b pb-1 mb-2 mt-4">Expenses</div>
            {[{ n: 'Cost of Goods Sold', v: '22,100.00' }, { n: 'Salaries & Wages', v: '12,000.00' }, { n: 'Rent', v: '3,500.00' }].map(r => (
              <div key={r.n} className="flex justify-between text-[9px] py-0.5">
                <span className="text-neutral-600 pl-3">{r.n}</span>
                <span>GHS {r.v}</span>
              </div>
            ))}
            <div className="flex justify-between text-[10px] font-bold border-t mt-1 pt-1">
              <span>Total Expenses</span><span>GHS 37,600.00</span>
            </div>

            <div className="flex justify-between text-xs font-bold border-t-2 border-neutral-800 mt-3 pt-2 text-emerald-600">
              <span>Net Profit</span><span>GHS 16,030.00</span>
            </div>
          </div>
        </div>
      </div>
    ),
  ];

  return (
    <div className="w-full aspect-[16/10] rounded-xl overflow-hidden border border-neutral-200/60 shadow-2xl bg-white">
      {/* Browser chrome */}
      <div className="h-7 bg-neutral-100 border-b border-neutral-200 flex items-center px-3 gap-1.5">
        <div className="w-2.5 h-2.5 rounded-full bg-red-400" />
        <div className="w-2.5 h-2.5 rounded-full bg-yellow-400" />
        <div className="w-2.5 h-2.5 rounded-full bg-green-400" />
        <div className="ml-4 flex-1 max-w-xs h-4 bg-white rounded border border-neutral-200 flex items-center px-2">
          <span className="text-[8px] text-neutral-400">cashbook.app/dashboard</span>
        </div>
      </div>
      {/* Screen content */}
      <div className="h-[calc(100%-28px)] overflow-hidden">
        {screens[variant % screens.length]}
      </div>
    </div>
  );
}

/* ---- COMPONENT ---- */

export default function HomePage() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const [emblaRef, emblaApi] = useEmblaCarousel(
    { loop: true, align: 'center' },
    [Autoplay({ delay: 5000, stopOnInteraction: false })]
  );
  const [selectedSlide, setSelectedSlide] = useState(0);

  useEffect(() => {
    const onScroll = () => setScrolled(window.scrollY > 10);
    window.addEventListener('scroll', onScroll);
    return () => window.removeEventListener('scroll', onScroll);
  }, []);

  const onSelect = useCallback(() => {
    if (!emblaApi) return;
    setSelectedSlide(emblaApi.selectedScrollSnap());
  }, [emblaApi]);

  useEffect(() => {
    if (!emblaApi) return;
    emblaApi.on('select', onSelect);
    onSelect();
  }, [emblaApi, onSelect]);

  const navLinks = [
    { label: 'Features', href: '#features' },
    { label: 'Pricing', href: '#pricing' },
    { label: 'Testimonials', href: '#testimonials' },
  ];

  return (
    <div className="min-h-screen bg-white">

      {/* ---- NAVBAR (Vercel-style) ---- */}
      <nav className={`fixed top-0 left-0 right-0 z-50 transition-all duration-200 ${
        scrolled
          ? 'bg-white/80 backdrop-blur-xl border-b border-neutral-200/70 shadow-sm'
          : 'bg-transparent border-b border-transparent'
      }`}>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="h-16 flex items-center justify-between">
            {/* Logo */}
            <Link href="/" className="flex items-center gap-2.5 group">
              <div className="h-8 w-8 bg-neutral-900 rounded-lg flex items-center justify-center group-hover:bg-neutral-800 transition-colors">
                <BookOpenIcon className="h-5 w-5 text-white" />
              </div>
              <span className="text-lg font-semibold text-neutral-900 tracking-tight">CashBook</span>
            </Link>

            {/* Desktop Nav */}
            <div className="hidden md:flex items-center gap-1">
              {navLinks.map((link) => (
                <a key={link.label} href={link.href} className="px-3 py-2 text-sm text-neutral-600 hover:text-neutral-900 transition-colors rounded-md hover:bg-neutral-100">
                  {link.label}
                </a>
              ))}
            </div>

            {/* Desktop Auth */}
            <div className="hidden md:flex items-center gap-3">
              <Link href="/login" className="text-sm font-medium text-neutral-600 hover:text-neutral-900 transition-colors px-3 py-2">
                Log In
              </Link>
              <Link href="/register" className="inline-flex items-center gap-1.5 px-4 py-2 text-sm font-medium text-white bg-neutral-900 rounded-lg hover:bg-neutral-800 transition-colors shadow-sm">
                Get Started <ArrowRightIcon className="h-3.5 w-3.5" />
              </Link>
            </div>

            {/* Mobile toggle */}
            <button className="md:hidden p-2 text-neutral-600" onClick={() => setMobileMenuOpen(!mobileMenuOpen)}>
              {mobileMenuOpen ? <XMarkIcon className="h-6 w-6" /> : <Bars3Icon className="h-6 w-6" />}
            </button>
          </div>
        </div>

        {/* Mobile menu */}
        {mobileMenuOpen && (
          <div className="md:hidden bg-white border-t border-neutral-200 px-4 pb-4 shadow-lg">
            {navLinks.map((link) => (
              <a key={link.label} href={link.href} className="block py-2.5 text-sm text-neutral-700 hover:text-neutral-900" onClick={() => setMobileMenuOpen(false)}>
                {link.label}
              </a>
            ))}
            <div className="mt-3 pt-3 border-t border-neutral-100 flex flex-col gap-2">
              <Link href="/login" className="btn-secondary w-full text-center">Log In</Link>
              <Link href="/register" className="btn-primary w-full text-center">Get Started Free</Link>
            </div>
          </div>
        )}
      </nav>

      {/* ---- HERO SECTION ---- */}
      <section className="pt-32 pb-20 lg:pt-40 lg:pb-28 relative overflow-hidden">
        {/* Subtle gradient background */}
        <div className="absolute inset-0 bg-gradient-to-b from-neutral-50/80 via-white to-white pointer-events-none" />
        <div className="absolute top-0 left-1/2 -translate-x-1/2 w-[800px] h-[600px] bg-gradient-to-br from-emerald-50/40 via-transparent to-blue-50/30 rounded-full blur-3xl pointer-events-none" />

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
          <div className="text-center max-w-4xl mx-auto">
            <div className="inline-flex items-center gap-2 px-4 py-1.5 rounded-full border border-neutral-200 bg-white text-neutral-600 text-sm font-medium mb-8 shadow-sm">
              <span className="relative flex h-2 w-2">
                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-emerald-400 opacity-75" />
                <span className="relative inline-flex rounded-full h-2 w-2 bg-emerald-500" />
              </span>
              Trusted by 2,500+ Ghana businesses
            </div>

            <h1 className="text-5xl sm:text-6xl lg:text-7xl font-bold text-neutral-900 tracking-tight leading-[1.08]">
              The Operating System
              <br />
              <span className="bg-gradient-to-r from-neutral-900 via-neutral-600 to-neutral-400 bg-clip-text text-transparent">for Your Business</span>
            </h1>

            <p className="mt-6 max-w-2xl mx-auto text-lg sm:text-xl text-neutral-500 leading-relaxed">
              CashBook is a complete cloud ERP &mdash; accounting, POS, invoicing, payroll, and tax compliance &mdash; built specifically for Ghanaian SMEs.
            </p>

            <div className="mt-10 flex flex-col sm:flex-row items-center justify-center gap-4">
              <Link href="/register" className="inline-flex items-center gap-2 px-8 py-3.5 text-base font-semibold text-white bg-neutral-900 rounded-xl hover:bg-neutral-800 transition-all shadow-lg shadow-neutral-900/20 hover:shadow-xl hover:shadow-neutral-900/25">
                Start Free Trial <ArrowRightIcon className="h-4 w-4" />
              </Link>
              <Link href="/login" className="inline-flex items-center gap-2 px-8 py-3.5 text-base font-semibold text-neutral-700 bg-white border border-neutral-300 rounded-xl hover:bg-neutral-50 hover:border-neutral-400 transition-all">
                Live Demo
              </Link>
            </div>

            <div className="mt-6 flex items-center justify-center gap-6 text-sm text-neutral-400">
              <span className="flex items-center gap-1"><CheckCircleIcon className="h-4 w-4 text-emerald-500" /> No credit card</span>
              <span className="flex items-center gap-1"><CheckCircleIcon className="h-4 w-4 text-emerald-500" /> 14-day trial</span>
              <span className="flex items-center gap-1"><CheckCircleIcon className="h-4 w-4 text-emerald-500" /> GRA compliant</span>
            </div>
          </div>

          {/* -- CAROUSEL -- */}
          <div className="mt-16 lg:mt-20 relative">
            <div className="overflow-hidden rounded-2xl" ref={emblaRef}>
              <div className="flex gap-6" style={{ backfaceVisibility: 'hidden' }}>
                {[0, 1, 2].map((idx) => (
                  <div key={idx} className="flex-[0_0_90%] sm:flex-[0_0_85%] lg:flex-[0_0_80%] min-w-0 pl-2 first:pl-0">
                    <DashboardMockup variant={idx} />
                  </div>
                ))}
              </div>
            </div>
            {/* Dots */}
            <div className="flex justify-center gap-2 mt-6">
              {[0, 1, 2].map((idx) => (
                <button
                  key={idx}
                  onClick={() => emblaApi?.scrollTo(idx)}
                  className={`h-2 rounded-full transition-all duration-300 ${
                    selectedSlide === idx ? 'w-8 bg-neutral-900' : 'w-2 bg-neutral-300 hover:bg-neutral-400'
                  }`}
                />
              ))}
            </div>
            {/* Edge fades */}
            <div className="absolute inset-y-0 left-0 w-12 bg-gradient-to-r from-white to-transparent pointer-events-none z-10" />
            <div className="absolute inset-y-0 right-0 w-12 bg-gradient-to-l from-white to-transparent pointer-events-none z-10" />
          </div>
        </div>
      </section>

      {/* ---- STATS BAR ---- */}
      <section className="py-12 border-y border-neutral-200 bg-neutral-50/50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-2 lg:grid-cols-4 gap-8">
            {stats.map((stat) => (
              <div key={stat.label} className="text-center">
                <div className="text-3xl lg:text-4xl font-bold text-neutral-900 tracking-tight">{stat.value}</div>
                <div className="mt-1 text-sm text-neutral-500">{stat.label}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ---- FEATURES SECTION ---- */}
      <section id="features" className="py-24 lg:py-32">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center max-w-3xl mx-auto mb-16">
            <p className="text-sm font-semibold text-emerald-600 uppercase tracking-wider mb-3">Features</p>
            <h2 className="text-3xl sm:text-4xl font-bold text-neutral-900 tracking-tight">
              Everything your business needs,<br className="hidden sm:block" /> nothing it doesn&apos;t
            </h2>
            <p className="mt-4 text-lg text-neutral-500">
              From daily bookkeeping to year-end tax filing &mdash; CashBook covers the full financial lifecycle.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-5">
            {features.map((feature) => (
              <div key={feature.name} className="group relative bg-white rounded-xl border border-neutral-200 p-6 hover:border-neutral-300 hover:shadow-md transition-all duration-200">
                <div className="h-10 w-10 rounded-lg bg-neutral-900 flex items-center justify-center mb-4 group-hover:bg-neutral-800 transition-colors">
                  <feature.icon className="h-5 w-5 text-white" />
                </div>
                <h3 className="text-sm font-semibold text-neutral-900">{feature.name}</h3>
                <p className="mt-2 text-sm text-neutral-500 leading-relaxed">{feature.description}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ---- GHANA COMPLIANCE ---- */}
      <section className="py-24 bg-neutral-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 bg-[radial-gradient(ellipse_at_top,_#1a1a2e_0%,_#0a0a0a_70%)]" />
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
          <div className="grid lg:grid-cols-2 gap-16 items-center">
            <div>
              <p className="text-sm font-semibold text-emerald-400 uppercase tracking-wider mb-3">Compliance</p>
              <h2 className="text-3xl sm:text-4xl font-bold tracking-tight">
                100% Ghana<br />Tax Compliant
              </h2>
              <p className="mt-4 text-neutral-400 leading-relaxed text-lg">
                Built from the ground up around GRA requirements. Every calculation, every form, every filing &mdash; done right.
              </p>
              <ul className="mt-8 space-y-3">
                {[
                  'VAT - 15% Standard / 4% Flat Rate',
                  'NHIL Levy - 2.5%',
                  'GETFund Levy - 2.5%',
                  'PAYE with all 7 Ghana brackets',
                  'SSNIT Tier 1, 2 & 3 deductions',
                  'Withholding Tax tracking',
                  'GRA e-filing ready',
                ].map((item) => (
                  <li key={item} className="flex items-center gap-3 text-sm text-neutral-300">
                    <CheckCircleIcon className="h-5 w-5 text-emerald-400 flex-shrink-0" />
                    {item}
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-white/5 backdrop-blur-sm rounded-2xl border border-white/10 p-8">
              <h3 className="text-base font-semibold text-white mb-6">Supported Payment Methods</h3>
              <div className="grid grid-cols-2 gap-4">
                {[
                  { name: 'MTN Mobile Money', icon: 'M' },
                  { name: 'Vodafone Cash', icon: 'V' },
                  { name: 'AirtelTigo Money', icon: 'A' },
                  { name: 'Bank Transfer', icon: 'B' },
                  { name: 'Paystack', icon: 'P' },
                  { name: 'Cash', icon: 'C' },
                ].map((method) => (
                  <div key={method.name} className="flex items-center gap-3 text-sm text-neutral-300 bg-white/5 rounded-lg px-3 py-2.5">
                    <span className="w-7 h-7 rounded-md bg-white/10 flex items-center justify-center text-xs font-bold text-white">{method.icon}</span>
                    {method.name}
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* ---- PRICING SECTION ---- */}
      <section id="pricing" className="py-24 lg:py-32">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center max-w-3xl mx-auto mb-16">
            <p className="text-sm font-semibold text-emerald-600 uppercase tracking-wider mb-3">Pricing</p>
            <h2 className="text-3xl sm:text-4xl font-bold text-neutral-900 tracking-tight">
              Simple, transparent pricing
            </h2>
            <p className="mt-4 text-lg text-neutral-500">
              Start free. Upgrade when you&apos;re ready. No hidden fees, no surprises.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 max-w-5xl mx-auto">
            {pricingPlans.map((plan) => (
              <div
                key={plan.name}
                className={`relative rounded-2xl p-8 transition-all duration-200 ${
                  plan.popular
                    ? 'bg-neutral-900 text-white border-2 border-neutral-900 shadow-2xl shadow-neutral-900/20 scale-[1.02]'
                    : 'bg-white border border-neutral-200 hover:border-neutral-300 hover:shadow-lg'
                }`}
              >
                {plan.popular && (
                  <div className="absolute -top-3.5 left-1/2 -translate-x-1/2 px-4 py-1 bg-emerald-500 text-white text-xs font-semibold rounded-full shadow">
                    Most Popular
                  </div>
                )}
                <h3 className={`text-lg font-semibold ${plan.popular ? 'text-white' : 'text-neutral-900'}`}>{plan.name}</h3>
                <div className="mt-4 flex items-baseline gap-1">
                  <span className={`text-4xl font-bold tracking-tight ${plan.popular ? 'text-white' : 'text-neutral-900'}`}>{plan.price}</span>
                  {plan.period && <span className={`text-sm ${plan.popular ? 'text-neutral-400' : 'text-neutral-500'}`}>{plan.period}</span>}
                </div>
                <p className={`mt-3 text-sm ${plan.popular ? 'text-neutral-400' : 'text-neutral-500'}`}>{plan.description}</p>

                <ul className="mt-6 space-y-2.5">
                  {plan.features.map((f) => (
                    <li key={f} className="flex items-start gap-2.5 text-sm">
                      <CheckIcon className={`h-4 w-4 mt-0.5 flex-shrink-0 ${plan.popular ? 'text-emerald-400' : 'text-emerald-500'}`} />
                      <span className={plan.popular ? 'text-neutral-300' : 'text-neutral-600'}>{f}</span>
                    </li>
                  ))}
                </ul>

                <Link
                  href="/register"
                  className={`mt-8 block w-full py-3 text-center text-sm font-semibold rounded-xl transition-all ${
                    plan.popular
                      ? 'bg-white text-neutral-900 hover:bg-neutral-100'
                      : 'bg-neutral-900 text-white hover:bg-neutral-800'
                  }`}
                >
                  {plan.cta}
                </Link>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ---- TESTIMONIALS ---- */}
      <section id="testimonials" className="py-24 bg-neutral-50 border-y border-neutral-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center max-w-3xl mx-auto mb-16">
            <p className="text-sm font-semibold text-emerald-600 uppercase tracking-wider mb-3">Testimonials</p>
            <h2 className="text-3xl sm:text-4xl font-bold text-neutral-900 tracking-tight">
              Loved by businesses across Ghana
            </h2>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 max-w-5xl mx-auto">
            {testimonials.map((t) => (
              <div key={t.name} className="bg-white rounded-xl border border-neutral-200 p-6 hover:shadow-md transition-shadow">
                <div className="flex gap-1 mb-4">
                  {[...Array(5)].map((_, i) => (
                    <StarIcon key={i} className="h-4 w-4 text-amber-400" />
                  ))}
                </div>
                <p className="text-sm text-neutral-600 leading-relaxed mb-6">&ldquo;{t.text}&rdquo;</p>
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 rounded-full bg-neutral-900 flex items-center justify-center text-white text-xs font-bold">
                    {t.name.split(' ').map(n => n[0]).join('')}
                  </div>
                  <div>
                    <div className="text-sm font-semibold text-neutral-900">{t.name}</div>
                    <div className="text-xs text-neutral-500">{t.role}</div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* ---- CTA SECTION ---- */}
      <section className="py-24 lg:py-32">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="relative bg-neutral-900 rounded-3xl p-12 lg:p-20 text-center overflow-hidden">
            <div className="absolute inset-0 bg-[radial-gradient(ellipse_at_center,_#1a1a2e_0%,_#0a0a0a_70%)]" />
            <div className="relative">
              <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-white tracking-tight">
                Ready to simplify<br />your finances?
              </h2>
              <p className="mt-4 text-lg text-neutral-400 max-w-xl mx-auto">
                Join thousands of Ghana businesses already using CashBook to save time, stay compliant, and grow faster.
              </p>
              <div className="mt-10 flex flex-col sm:flex-row items-center justify-center gap-4">
                <Link href="/register" className="inline-flex items-center gap-2 px-8 py-4 text-base font-semibold text-neutral-900 bg-white rounded-xl hover:bg-neutral-100 transition-all shadow-lg">
                  Start Your Free Trial <ArrowRightIcon className="h-4 w-4" />
                </Link>
              </div>
              <p className="mt-6 text-sm text-neutral-500">No credit card required. 14-day free trial.</p>
            </div>
          </div>
        </div>
      </section>

      {/* ---- FOOTER ---- */}
      <footer className="bg-neutral-900 text-neutral-400 border-t border-neutral-800">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 mb-12">
            {/* Brand */}
            <div className="col-span-2 md:col-span-1">
              <div className="flex items-center gap-2 mb-4">
                <div className="h-8 w-8 bg-white rounded-lg flex items-center justify-center">
                  <BookOpenIcon className="h-5 w-5 text-neutral-900" />
                </div>
                <span className="text-lg font-semibold text-white">CashBook</span>
              </div>
              <p className="text-sm text-neutral-500 leading-relaxed">
                Cloud-based ERP built for Ghanaian businesses. Accounting, POS, payroll, and tax &mdash; simplified.
              </p>
            </div>

            {/* Product */}
            <div>
              <h4 className="text-sm font-semibold text-white mb-4">Product</h4>
              <ul className="space-y-2.5">
                {['Accounting', 'Invoicing', 'Point of Sale', 'Payroll', 'Inventory', 'Reports'].map(l => (
                  <li key={l}><a href="#features" className="text-sm text-neutral-500 hover:text-white transition-colors">{l}</a></li>
                ))}
              </ul>
            </div>

            {/* Company */}
            <div>
              <h4 className="text-sm font-semibold text-white mb-4">Company</h4>
              <ul className="space-y-2.5">
                {['About', 'Blog', 'Careers', 'Contact', 'Partners'].map(l => (
                  <li key={l}><a href="#" className="text-sm text-neutral-500 hover:text-white transition-colors">{l}</a></li>
                ))}
              </ul>
            </div>

            {/* Legal */}
            <div>
              <h4 className="text-sm font-semibold text-white mb-4">Legal</h4>
              <ul className="space-y-2.5">
                {['Privacy Policy', 'Terms of Service', 'Security', 'GDPR'].map(l => (
                  <li key={l}><a href="#" className="text-sm text-neutral-500 hover:text-white transition-colors">{l}</a></li>
                ))}
              </ul>
            </div>
          </div>

          {/* Bottom bar */}
          <div className="pt-8 border-t border-neutral-800 flex flex-col md:flex-row items-center justify-between gap-4">
            <p className="text-sm text-neutral-500">&copy; {new Date().getFullYear()} PineWeb Consult. All rights reserved.</p>
            <div className="flex items-center gap-4 text-neutral-500">
              <span className="text-sm">Built for Ghana businesses</span>
            </div>
          </div>
        </div>
      </footer>
    </div>
  );
}
