'use client';

import React, { useState } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import { useAuth } from '@/context/AuthContext';
import toast from 'react-hot-toast';
import { BookOpenIcon } from '@heroicons/react/24/outline';
import { ghanaRegions } from '@/lib/utils';

export default function RegisterPage() {
  const [form, setForm] = useState({
    company_name: '',
    business_type: 'sole_proprietorship',
    first_name: '',
    last_name: '',
    email: '',
    password: '',
    password_confirm: '',
    tin_number: '',
    phone: '',
    region: '',
    city: '',
  });
  const [loading, setLoading] = useState(false);
  const { register } = useAuth();
  const router = useRouter();

  const onChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    setForm({ ...form, [e.target.name]: e.target.value });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (form.password !== form.password_confirm) {
      toast.error('Passwords do not match');
      return;
    }
    if (form.password.length < 8) {
      toast.error('Password must be at least 8 characters');
      return;
    }
    setLoading(true);
    try {
      await register(form);
      toast.success('Account created! Welcome to CashBook.');
      router.push('/dashboard');
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Registration failed');
    }
    setLoading(false);
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-white py-12 px-4">
      <div className="max-w-lg w-full">
        <div className="text-center mb-8">
          <Link href="/" className="inline-flex items-center gap-2">
            <div className="h-10 w-10 bg-neutral-900 rounded-md flex items-center justify-center">
              <BookOpenIcon className="h-6 w-6 text-white" />
            </div>
            <span className="text-2xl font-semibold text-neutral-900">CashBook</span>
          </Link>
          <p className="mt-2 text-sm text-neutral-500">Create your free account</p>
        </div>

        <div className="card p-8">
          <form onSubmit={handleSubmit} className="space-y-5">
            {/* Company Info */}
            <h3 className="text-sm font-semibold text-gray-900 uppercase tracking-wide">Company Information</h3>
            <div>
              <label className="label">Company / Business Name *</label>
              <input name="company_name" className="input" value={form.company_name} onChange={onChange} required />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="label">Business Type *</label>
                <select name="business_type" className="select" value={form.business_type} onChange={onChange}>
                  <option value="sole_proprietorship">Sole Proprietorship</option>
                  <option value="partnership">Partnership</option>
                  <option value="limited_company">Limited Company</option>
                  <option value="ngo">NGO / Non-Profit</option>
                </select>
              </div>
              <div>
                <label className="label">TIN Number</label>
                <input name="tin_number" className="input" value={form.tin_number} onChange={onChange} placeholder="GRA TIN" />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="label">Region</label>
                <select name="region" className="select" value={form.region} onChange={onChange}>
                  <option value="">Select region</option>
                  {ghanaRegions.map((r) => (
                    <option key={r} value={r}>{r}</option>
                  ))}
                </select>
              </div>
              <div>
                <label className="label">City / Town</label>
                <input name="city" className="input" value={form.city} onChange={onChange} />
              </div>
            </div>

            <hr className="my-2" />

            {/* User Info */}
            <h3 className="text-sm font-semibold text-gray-900 uppercase tracking-wide">Your Account</h3>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="label">First Name *</label>
                <input name="first_name" className="input" value={form.first_name} onChange={onChange} required />
              </div>
              <div>
                <label className="label">Last Name *</label>
                <input name="last_name" className="input" value={form.last_name} onChange={onChange} required />
              </div>
            </div>

            <div>
              <label className="label">Email Address *</label>
              <input name="email" type="email" className="input" value={form.email} onChange={onChange} required />
            </div>

            <div>
              <label className="label">Phone Number</label>
              <input name="phone" className="input" value={form.phone} onChange={onChange} placeholder="+233" />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="label">Password *</label>
                <input name="password" type="password" className="input" value={form.password} onChange={onChange} minLength={8} required />
              </div>
              <div>
                <label className="label">Confirm Password *</label>
                <input name="password_confirm" type="password" className="input" value={form.password_confirm} onChange={onChange} required />
              </div>
            </div>

            <button type="submit" className="btn-primary w-full" disabled={loading}>
              {loading ? 'Creating Account...' : 'Create Free Account'}
            </button>
          </form>
        </div>

        <p className="text-center text-sm text-neutral-500 mt-6">
          Already have an account?{' '}
          <Link href="/login" className="text-neutral-900 font-medium hover:underline">
            Sign in
          </Link>
        </p>
      </div>
    </div>
  );
}
