'use client';

import React, { useRef, useEffect, useState } from 'react';
import { XMarkIcon, PrinterIcon } from '@heroicons/react/24/outline';
import { formatCurrency } from '@/lib/utils';
import QRCode from 'qrcode';

interface ReceiptItem {
  name: string;
  quantity: number;
  unit_price: number;
  total: number;
}

interface ReceiptData {
  receipt_number: string;
  date: string;
  items: ReceiptItem[];
  subtotal: number;
  nhil: number;
  getfund: number;
  vat: number;
  total_tax: number;
  grand_total: number;
  payment_method: string;
  amount_tendered: number;
  change: number;
  company_name: string;
  cashier_name: string;
}

interface ReceiptPopupProps {
  open: boolean;
  onClose: () => void;
  receipt: ReceiptData | null;
}

export default function ReceiptPopup({ open, onClose, receipt }: ReceiptPopupProps) {
  const receiptRef = useRef<HTMLDivElement>(null);
  const [qrDataUrl, setQrDataUrl] = useState<string>('');

  useEffect(() => {
    if (receipt) {
      const qrPayload = JSON.stringify({
        rcpt: receipt.receipt_number,
        date: receipt.date,
        total: receipt.grand_total,
        items: receipt.items.map(i => ({ n: i.name, q: i.quantity, p: i.unit_price })),
        tax: receipt.total_tax,
        co: receipt.company_name,
      });
      QRCode.toDataURL(qrPayload, { width: 120, margin: 1, errorCorrectionLevel: 'M' })
        .then((url: string) => setQrDataUrl(url))
        .catch(() => setQrDataUrl(''));
    }
  }, [receipt]);

  if (!open || !receipt) return null;

  const thermalCSS = `
    *{margin:0;padding:0;box-sizing:border-box}
    body{font-family:'Segoe UI','Arial Narrow',Arial,sans-serif;width:72mm;max-width:72mm;margin:0 auto;padding:2mm 0;color:#111;font-size:11px;line-height:1.4;-webkit-print-color-adjust:exact;print-color-adjust:exact}
    hr.dash{border:none;border-top:1px dashed #aaa;margin:5px 0}
    hr.solid{border:none;border-top:1.5px solid #333;margin:5px 0}
    .row{display:flex;justify-content:space-between;padding:1px 0}
    .row .lbl{color:#666;font-size:10px}
    .row .val{font-weight:600;font-size:10px}
    .row.tax{font-size:9px;color:#777}
    .row.total{font-size:14px;font-weight:800;padding:3px 0}
    .hdr{text-align:center;margin-bottom:6px}
    .hdr h1{font-size:15px;font-weight:800;letter-spacing:.4px;margin-bottom:1px}
    .hdr .sub{font-size:8px;color:#666;text-transform:uppercase;letter-spacing:1px}
    .cols{display:flex;justify-content:space-between;font-size:8px;font-weight:700;text-transform:uppercase;color:#555;padding:2px 0;border-bottom:1px solid #ddd;margin-bottom:3px}
    .itm{margin-bottom:3px}
    .itm-n{font-size:10px;font-weight:600}
    .itm-d{display:flex;justify-content:space-between;font-size:9px;color:#555}
    .itm-d .amt{color:#111;font-weight:600}
    .qr{text-align:center;margin:5px 0}
    .qr img{width:80px;height:80px;display:block;margin:0 auto}
    .qr p{font-size:7px;color:#999;margin-top:2px}
    .ftr{text-align:center;margin-top:5px}
    .ftr .ty{font-size:10px;font-weight:600}
    .ftr .pw{font-size:7px;color:#999;margin-top:3px}
    @media print{@page{size:80mm auto;margin:0}body{width:72mm;padding:2mm 4mm}}
  `;

  const handlePrint = () => {
    if (!receipt) return;
    const pw = window.open('', '_blank', 'width=320,height=600');
    if (!pw) return;

    const itemsHtml = receipt.items.map(item =>
      `<div class="itm"><div class="itm-n">${item.name}</div><div class="itm-d"><span>${item.quantity} &times; ${formatCurrency(item.unit_price)}</span><span class="amt">${formatCurrency(item.total)}</span></div></div>`
    ).join('');

    const cashSection = receipt.payment_method === 'cash' ? `
      <div class="row"><span class="lbl">Tendered</span><span class="val">${formatCurrency(receipt.amount_tendered)}</span></div>
      <div class="row" style="font-weight:700;font-size:11px"><span>Change</span><span>${formatCurrency(receipt.change)}</span></div>
    ` : '';

    const qrSection = qrDataUrl ? `
      <hr class="dash" />
      <div class="qr"><img src="${qrDataUrl}" alt="QR" /><p>Scan for receipt details</p></div>
    ` : '';

    const html = `<!DOCTYPE html><html><head><meta charset="utf-8"><title>Receipt ${receipt.receipt_number}</title><style>${thermalCSS}</style></head><body>
      <div class="hdr"><h1>${receipt.company_name}</h1><p class="sub">Tax Invoice / Receipt</p></div>
      <hr class="dash" />
      <div class="row"><span class="lbl">Receipt #</span><span class="val">${receipt.receipt_number}</span></div>
      <div class="row"><span class="lbl">Date</span><span class="val">${receipt.date}</span></div>
      <div class="row"><span class="lbl">Cashier</span><span class="val">${receipt.cashier_name}</span></div>
      <hr class="dash" />
      <div class="cols"><span>Item</span><span>Amount</span></div>
      ${itemsHtml}
      <hr class="dash" />
      <div class="row"><span>Subtotal</span><span>${formatCurrency(receipt.subtotal)}</span></div>
      <div class="row tax"><span>NHIL (2.5%)</span><span>${formatCurrency(receipt.nhil)}</span></div>
      <div class="row tax"><span>GETFund (2.5%)</span><span>${formatCurrency(receipt.getfund)}</span></div>
      <div class="row tax"><span>VAT (15%)</span><span>${formatCurrency(receipt.vat)}</span></div>
      <hr class="solid" />
      <div class="row total"><span>TOTAL</span><span>${formatCurrency(receipt.grand_total)}</span></div>
      <hr class="dash" />
      <div class="row"><span class="lbl">Payment</span><span class="val" style="text-transform:capitalize">${receipt.payment_method.replace('_', ' ')}</span></div>
      ${cashSection}
      ${qrSection}
      <hr class="dash" />
      <div class="ftr"><p class="ty">Thank you for your purchase!</p><p class="pw">Powered by CashBook ERP</p></div>
    </body><script>window.onload=function(){window.print();}<\/script></html>`;

    pw.document.write(html);
    pw.document.close();
  };

  const S = {
    flex: { display: 'flex', justifyContent: 'space-between' } as React.CSSProperties,
    info: { fontSize: '10px', padding: '1px 0' } as React.CSSProperties,
    lbl: { color: '#666' } as React.CSSProperties,
    val: { fontWeight: 600 } as React.CSSProperties,
    dash: { border: 'none', borderTop: '1px dashed #aaa', margin: '6px 0' } as React.CSSProperties,
    solid: { border: 'none', borderTop: '1.5px solid #333', margin: '6px 0' } as React.CSSProperties,
    tax: { fontSize: '9px', padding: '1px 0', color: '#777' } as React.CSSProperties,
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50">
      <div className="bg-white rounded-lg border border-neutral-200 w-full max-w-[340px] mx-4 overflow-hidden shadow-2xl">
        {/* Header */}
        <div className="flex items-center justify-between px-4 py-2.5 bg-neutral-900 text-white">
          <h3 className="text-xs font-semibold tracking-wide uppercase">Sale Receipt</h3>
          <div className="flex items-center gap-1.5">
            <button onClick={handlePrint} className="p-1 rounded hover:bg-neutral-700 transition-colors" title="Print"><PrinterIcon className="h-4 w-4" /></button>
            <button onClick={onClose} className="p-1 rounded hover:bg-neutral-700 transition-colors"><XMarkIcon className="h-4 w-4" /></button>
          </div>
        </div>

        {/* Receipt */}
        <div className="px-3 py-3 max-h-[75vh] overflow-y-auto bg-neutral-50">
          <div ref={receiptRef} className="bg-white border border-neutral-200 rounded px-4 py-3 shadow-sm" style={{ fontFamily: "'Segoe UI','Arial Narrow',Arial,sans-serif", fontSize: '11px', lineHeight: '1.4', color: '#111', width: '72mm', maxWidth: '72mm', margin: '0 auto' }}>

            <div className="r-hdr" style={{ textAlign: 'center', marginBottom: '8px' }}>
              <h1 style={{ fontSize: '16px', fontWeight: 800, letterSpacing: '0.5px', marginBottom: '2px' }}>{receipt.company_name}</h1>
              <p className="sub" style={{ fontSize: '9px', color: '#666', textTransform: 'uppercase', letterSpacing: '1px' }}>Tax Invoice / Receipt</p>
            </div>

            <hr style={S.dash} />

            <div style={{ padding: '2px 0' }}>
              <div style={{ ...S.flex, ...S.info }}><span style={S.lbl}>Receipt #</span><span style={S.val}>{receipt.receipt_number}</span></div>
              <div style={{ ...S.flex, ...S.info }}><span style={S.lbl}>Date</span><span style={S.val}>{receipt.date}</span></div>
              <div style={{ ...S.flex, ...S.info }}><span style={S.lbl}>Cashier</span><span style={S.val}>{receipt.cashier_name}</span></div>
            </div>

            <hr style={S.dash} />

            <div style={{ ...S.flex, fontSize: '9px', fontWeight: 700, textTransform: 'uppercase', color: '#555', padding: '3px 0', borderBottom: '1px solid #ddd', marginBottom: '4px' }}>
              <span>Item</span><span>Amount</span>
            </div>

            {receipt.items.map((item, idx) => (
              <div key={idx} style={{ marginBottom: '4px' }}>
                <div style={{ fontSize: '11px', fontWeight: 600 }}>{item.name}</div>
                <div style={{ ...S.flex, fontSize: '10px', color: '#555' }}>
                  <span>{item.quantity} &times; {formatCurrency(item.unit_price)}</span>
                  <span style={{ color: '#111', fontWeight: 600 }}>{formatCurrency(item.total)}</span>
                </div>
              </div>
            ))}

            <hr style={S.dash} />

            <div style={{ ...S.flex, fontSize: '10px', padding: '1.5px 0' }}><span>Subtotal</span><span>{formatCurrency(receipt.subtotal)}</span></div>
            <div style={{ ...S.flex, ...S.tax }}><span>NHIL (2.5%)</span><span>{formatCurrency(receipt.nhil)}</span></div>
            <div style={{ ...S.flex, ...S.tax }}><span>GETFund (2.5%)</span><span>{formatCurrency(receipt.getfund)}</span></div>
            <div style={{ ...S.flex, ...S.tax }}><span>VAT (15%)</span><span>{formatCurrency(receipt.vat)}</span></div>

            <hr style={S.solid} />

            <div style={{ ...S.flex, fontSize: '15px', fontWeight: 800, padding: '4px 0' }}>
              <span>TOTAL</span><span>{formatCurrency(receipt.grand_total)}</span>
            </div>

            <hr style={S.dash} />

            <div style={{ ...S.flex, fontSize: '10px', padding: '1.5px 0' }}>
              <span style={S.lbl}>Payment</span>
              <span style={{ fontWeight: 600, textTransform: 'capitalize' }}>{receipt.payment_method.replace('_', ' ')}</span>
            </div>
            {receipt.payment_method === 'cash' && (
              <>
                <div style={{ ...S.flex, fontSize: '10px', padding: '1.5px 0' }}>
                  <span style={S.lbl}>Tendered</span><span>{formatCurrency(receipt.amount_tendered)}</span>
                </div>
                <div style={{ ...S.flex, fontSize: '11px', fontWeight: 700, padding: '1.5px 0' }}>
                  <span>Change</span><span>{formatCurrency(receipt.change)}</span>
                </div>
              </>
            )}

            {qrDataUrl && (
              <>
                <hr style={S.dash} />
                <div style={{ textAlign: 'center', margin: '6px 0' }}>
                  <img src={qrDataUrl} alt="QR" style={{ width: '100px', height: '100px', display: 'block', margin: '0 auto' }} />
                  <p style={{ fontSize: '8px', color: '#999', marginTop: '3px' }}>Scan for receipt details</p>
                </div>
              </>
            )}

            <hr style={S.dash} />

            <div style={{ textAlign: 'center', marginTop: '6px' }}>
              <p style={{ fontSize: '11px', fontWeight: 600 }}>Thank you for your purchase!</p>
              <p style={{ fontSize: '8px', color: '#999', marginTop: '4px' }}>Powered by CashBook ERP</p>
            </div>
          </div>
        </div>

        {/* Actions */}
        <div className="px-3 py-2.5 border-t border-neutral-200 flex gap-2 bg-white">
          <button onClick={handlePrint} className="btn-primary flex-1 flex items-center justify-center gap-2 text-sm">
            <PrinterIcon className="h-4 w-4" /> Print
          </button>
          <button onClick={onClose} className="btn-secondary flex-1 text-sm">New Sale</button>
        </div>
      </div>
    </div>
  );
}
