'use client';

import React, { createContext, useContext, useState, useEffect, useCallback } from 'react';
import Cookies from 'js-cookie';
import { authAPI, businessAPI } from '@/lib/api';

interface Company {
  id: string;
  name: string;
  business_type?: string;
  currency: string;
}

interface User {
  id: string;
  email: string;
  first_name: string;
  last_name: string;
  role: string;
  tenant_id?: string;
  company: Company;
}

interface AuthContextType {
  user: User | null;
  loading: boolean;
  login: (email: string, password: string) => Promise<void>;
  register: (data: any) => Promise<void>;
  logout: () => void;
  switchBusiness: (companyId: string) => Promise<void>;
  isAuthenticated: boolean;
  isAdmin: boolean;
  isManager: boolean;
}

const AuthContext = createContext<AuthContextType>({
  user: null,
  loading: true,
  login: async () => {},
  register: async () => {},
  logout: () => {},
  switchBusiness: async () => {},
  isAuthenticated: false,
  isAdmin: false,
  isManager: false,
});

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);

  // Normalize flat profile response into nested User shape
  const normalizeUser = (raw: any): User | null => {
    if (!raw) return null;
    // If the profile endpoint returns flat data (company_id, company_name at top level)
    // restructure it into the nested shape the frontend expects
    if (raw.company_id && !raw.company) {
      return {
        id: raw.id,
        email: raw.email,
        first_name: raw.first_name,
        last_name: raw.last_name,
        role: raw.role,
        tenant_id: raw.tenant_id,
        company: {
          id: raw.company_id,
          name: raw.company_name || '',
          business_type: raw.business_type || '',
          currency: raw.currency || 'GHS',
        },
      };
    }
    return raw as User;
  };

  const loadUser = useCallback(async () => {
    const token = Cookies.get('access_token');
    if (!token) {
      setLoading(false);
      return;
    }
    try {
      const { data } = await authAPI.profile();
      if (data?.data) {
        setUser(normalizeUser(data.data));
      }
    } catch {
      Cookies.remove('access_token');
      Cookies.remove('refresh_token');
    }
    setLoading(false);
  }, []);

  useEffect(() => {
    loadUser();
  }, [loadUser]);

  const login = async (email: string, password: string) => {
    const { data } = await authAPI.login({ email, password });
    if (data?.data) {
      Cookies.set('access_token', data.data.token, { expires: 1 });
      Cookies.set('refresh_token', data.data.refresh_token, { expires: 30 });
      setUser(data.data.user);
    }
  };

  const register = async (regData: any) => {
    const { data } = await authAPI.register(regData);
    if (data?.data) {
      Cookies.set('access_token', data.data.token, { expires: 1 });
      Cookies.set('refresh_token', data.data.refresh_token, { expires: 30 });
      setUser(data.data.user);
    }
  };

  const logout = () => {
    Cookies.remove('access_token');
    Cookies.remove('refresh_token');
    setUser(null);
    window.location.href = '/login';
  };

  const switchBusiness = async (companyId: string) => {
    const { data } = await businessAPI.switch(companyId);
    if (data?.data) {
      Cookies.set('access_token', data.data.token, { expires: 1 });
      Cookies.set('refresh_token', data.data.refresh_token, { expires: 30 });
      setUser(data.data.user);
      // Reload the page to refresh all data for the new business context
      window.location.href = '/dashboard';
    }
  };

  const isAdmin = user?.role === 'admin' || user?.role === 'super_admin';
  const isManager = isAdmin || user?.role === 'manager';

  return (
    <AuthContext.Provider
      value={{
        user,
        loading,
        login,
        register,
        logout,
        switchBusiness,
        isAuthenticated: !!user,
        isAdmin,
        isManager,
      }}
    >
      {children}
    </AuthContext.Provider>
  );
}

export const useAuth = () => useContext(AuthContext);
