import axios, { AxiosError, InternalAxiosRequestConfig } from 'axios';
import Cookies from 'js-cookie';

// In production (same domain), API_URL is just '/api'.
// In development, it can be overridden to point at the PHP backend directly.
const API_URL = process.env.NEXT_PUBLIC_API_URL || '/api';

const api = axios.create({
  baseURL: API_URL,
  headers: {
    'Content-Type': 'application/json',
  },
  withCredentials: true,
});

// Attach JWT token to every request
api.interceptors.request.use((config: InternalAxiosRequestConfig) => {
  const token = Cookies.get('access_token');
  if (token && config.headers) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

// Handle 401 responses - attempt token refresh
api.interceptors.response.use(
  (response) => response,
  async (error: AxiosError) => {
    const originalRequest = error.config as InternalAxiosRequestConfig & { _retry?: boolean };
    
    if (error.response?.status === 401 && !originalRequest._retry) {
      originalRequest._retry = true;
      const refreshToken = Cookies.get('refresh_token');

      if (refreshToken) {
        try {
          const { data } = await axios.post(`${API_URL}/auth/refresh`, {
            refresh_token: refreshToken,
          });

          if (data?.data?.token) {
            Cookies.set('access_token', data.data.token, { expires: 1 });
            if (originalRequest.headers) {
              originalRequest.headers.Authorization = `Bearer ${data.data.token}`;
            }
            return api(originalRequest);
          }
        } catch {
          // Refresh failed - redirect to login
          Cookies.remove('access_token');
          Cookies.remove('refresh_token');
          if (typeof window !== 'undefined') {
            window.location.href = '/login';
          }
        }
      } else {
        Cookies.remove('access_token');
        if (typeof window !== 'undefined') {
          window.location.href = '/login';
        }
      }
    }
    return Promise.reject(error);
  }
);

export default api;

// ───── Typed API helpers ─────

export interface ApiResponse<T = any> {
  success: boolean;
  data: T;
  message?: string;
}

export interface PaginatedResponse<T = any> {
  success: boolean;
  data: T[];
  pagination: {
    total: number;
    page: number;
    limit: number;
    pages: number;
  };
}

// Auth
export const authAPI = {
  register: (data: any) => api.post<ApiResponse>('/auth/register', data),
  login: (data: { email: string; password: string }) => api.post<ApiResponse>('/auth/login', data),
  profile: () => api.get<ApiResponse>('/auth/profile'),
  changePassword: (data: any) => api.put<ApiResponse>('/auth/password', data),
};

// Accounting
export const accountingAPI = {
  getAccounts: () => api.get<ApiResponse>('/accounts'),
  createAccount: (data: any) => api.post<ApiResponse>('/accounts', data),
  updateAccount: (id: string, data: any) => api.put<ApiResponse>(`/accounts/${id}`, data),
  getJournalEntries: (params?: any) => api.get<PaginatedResponse>('/journal-entries', { params }),
  createJournalEntry: (data: any) => api.post<ApiResponse>('/journal-entries', data),
  getJournalEntry: (id: string) => api.get<ApiResponse>(`/journal-entries/${id}`),
  postJournalEntry: (id: string) => api.post<ApiResponse>(`/journal-entries/${id}/post`),
  getTrialBalance: (params?: any) => api.get<ApiResponse>('/trial-balance', { params }),
  getGeneralLedger: (params?: any) => api.get<ApiResponse>('/general-ledger', { params }),
};

// Invoicing
export const invoiceAPI = {
  list: (params?: any) => api.get<PaginatedResponse>('/invoices', { params }),
  create: (data: any) => api.post<ApiResponse>('/invoices', data),
  get: (id: string) => api.get<ApiResponse>(`/invoices/${id}`),
  update: (id: string, data: any) => api.put<ApiResponse>(`/invoices/${id}`, data),
  void: (id: string) => api.delete<ApiResponse>(`/invoices/${id}`),
  listPayments: (params?: any) => api.get<PaginatedResponse>('/payments', { params }),
  recordPayment: (data: any) => api.post<ApiResponse>('/payments', data),
};

// POS
export const posAPI = {
  getTerminals: () => api.get<ApiResponse>('/pos/terminals'),
  createTerminal: (data: any) => api.post<ApiResponse>('/pos/terminals', data),
  openSession: (data: any) => api.post<ApiResponse>('/pos/sessions/open', data),
  closeSession: (id: string, data: any) => api.post<ApiResponse>(`/pos/sessions/${id}/close`, data),
  currentSession: () => api.get<ApiResponse>('/pos/sessions/current'),
  createSale: (data: any) => api.post<ApiResponse>('/pos/sales', data),
  listSales: (params?: any) => api.get<PaginatedResponse>('/pos/sales', { params }),
  getSale: (id: string) => api.get<ApiResponse>(`/pos/sales/${id}`),
  refund: (id: string, data: any) => api.post<ApiResponse>(`/pos/sales/${id}/refund`, data),
  getReturns: (params?: any) => api.get<PaginatedResponse>('/pos/returns', { params }),
  dailyReport: (params?: any) => api.get<ApiResponse>('/pos/daily-report', { params }),
};

// Transactions (Income & Expense)
export const transactionAPI = {
  list: (params?: any) => api.get<PaginatedResponse>('/transactions', { params }),
  create: (data: any) => api.post<ApiResponse>('/transactions', data),
  get: (id: string) => api.get<ApiResponse>(`/transactions/${id}`),
  void: (id: string) => api.delete<ApiResponse>(`/transactions/${id}`),
  summary: (params?: any) => api.get<ApiResponse>('/transactions/summary', { params }),
};

// Payroll
export const payrollAPI = {
  listEmployees: () => api.get<ApiResponse>('/employees'),
  createEmployee: (data: any) => api.post<ApiResponse>('/employees', data),
  getEmployee: (id: string) => api.get<ApiResponse>(`/employees/${id}`),
  updateEmployee: (id: string, data: any) => api.put<ApiResponse>(`/employees/${id}`, data),
  runPayroll: (data: any) => api.post<ApiResponse>('/payroll/run', data),
  listPeriods: () => api.get<ApiResponse>('/payroll/periods'),
  getPayslips: (periodId: string) => api.get<ApiResponse>(`/payroll/periods/${periodId}/payslips`),
  approvePayroll: (periodId: string) => api.post<ApiResponse>(`/payroll/periods/${periodId}/approve`),
};

// Inventory
export const inventoryAPI = {
  listProducts: (params?: any) => api.get<PaginatedResponse>('/products', { params }),
  createProduct: (data: any) => api.post<ApiResponse>('/products', data),
  getProduct: (id: string) => api.get<ApiResponse>(`/products/${id}`),
  updateProduct: (id: string, data: any) => api.put<ApiResponse>(`/products/${id}`, data),
  barcodeLookup: (code: string) => api.get<ApiResponse>(`/products/barcode/${code}`),
  adjustStock: (id: string, data: any) => api.post<ApiResponse>(`/products/${id}/adjust-stock`, data),
  importProducts: (products: any[]) => api.post<ApiResponse>('/products/import', { products }),
  lowStock: () => api.get<ApiResponse>('/inventory/low-stock'),
  movements: (params?: any) => api.get<PaginatedResponse>('/inventory/movements', { params }),
  getCategories: () => api.get<ApiResponse>('/product-categories'),
  createCategory: (data: any) => api.post<ApiResponse>('/product-categories', data),
  updateCategory: (id: string, data: any) => api.put<ApiResponse>(`/product-categories/${id}`, data),
  deleteCategory: (id: string) => api.delete<ApiResponse>(`/product-categories/${id}`),
};

// Contacts
export const contactAPI = {
  list: (params?: any) => api.get<PaginatedResponse>('/contacts', { params }),
  create: (data: any) => api.post<ApiResponse>('/contacts', data),
  get: (id: string) => api.get<ApiResponse>(`/contacts/${id}`),
  update: (id: string, data: any) => api.put<ApiResponse>(`/contacts/${id}`, data),
};

// Assets
export const assetAPI = {
  list: (params?: any) => api.get<ApiResponse>('/assets', { params }),
  create: (data: any) => api.post<ApiResponse>('/assets', data),
  get: (id: string) => api.get<ApiResponse>(`/assets/${id}`),
  depreciate: (id: string) => api.post<ApiResponse>(`/assets/${id}/depreciate`),
  dispose: (id: string, data: any) => api.post<ApiResponse>(`/assets/${id}/dispose`, data),
};

// Tax
export const taxAPI = {
  getRates: () => api.get<ApiResponse>('/tax/rates'),
  calculateTax: (data: any) => api.post<ApiResponse>('/tax/calculate', data),
  calculatePAYE: (data: any) => api.post<ApiResponse>('/tax/calculate-paye', data),
  getFilings: () => api.get<ApiResponse>('/tax/filings'),
  createFiling: (data: any) => api.post<ApiResponse>('/tax/filings', data),
  withholdingSummary: (params?: any) => api.get<ApiResponse>('/tax/withholding-summary', { params }),
};

// Reports
export const reportAPI = {
  incomeStatement: (params?: any) => api.get<ApiResponse>('/reports/income-statement', { params }),
  balanceSheet: (params?: any) => api.get<ApiResponse>('/reports/balance-sheet', { params }),
  cashFlow: (params?: any) => api.get<ApiResponse>('/reports/cash-flow', { params }),
  taxSummary: (params?: any) => api.get<ApiResponse>('/reports/tax-summary', { params }),
  arAging: () => api.get<ApiResponse>('/reports/aging'),
  dashboard: () => api.get<ApiResponse>('/reports/dashboard'),
};

// Admin
export const adminAPI = {
  listUsers: () => api.get<ApiResponse>('/admin/users'),
  createUser: (data: any) => api.post<ApiResponse>('/admin/users', data),
  updateUser: (id: string, data: any) => api.put<ApiResponse>(`/admin/users/${id}`, data),
  deactivateUser: (id: string) => api.delete<ApiResponse>(`/admin/users/${id}`),
  getCompany: () => api.get<ApiResponse>('/admin/company'),
  updateCompany: (data: any) => api.put<ApiResponse>('/admin/company', data),
  getAuditLog: (params?: any) => api.get<PaginatedResponse>('/admin/audit-log', { params }),
  listBankAccounts: () => api.get<ApiResponse>('/admin/bank-accounts'),
  createBankAccount: (data: any) => api.post<ApiResponse>('/admin/bank-accounts', data),
  getNotifications: () => api.get<ApiResponse>('/admin/notifications'),
  markNotificationRead: (id: string) => api.put<ApiResponse>(`/admin/notifications/${id}/read`),
  exportData: () => api.get('/admin/backup', { responseType: 'blob' }),
};

// Tenant Management
export const tenantAPI = {
  get: () => api.get<ApiResponse>('/tenant'),
  update: (data: any) => api.put<ApiResponse>('/tenant', data),
};

// Business Management
export const businessAPI = {
  list: () => api.get<ApiResponse>('/businesses'),
  create: (data: any) => api.post<ApiResponse>('/businesses', data),
  get: (id: string) => api.get<ApiResponse>(`/businesses/${id}`),
  update: (id: string, data: any) => api.put<ApiResponse>(`/businesses/${id}`, data),
  deactivate: (id: string) => api.delete<ApiResponse>(`/businesses/${id}`),
  assignUser: (id: string, data: { user_id: string; role: string }) => api.post<ApiResponse>(`/businesses/${id}/assign-user`, data),
  removeUser: (id: string, data: { user_id: string }) => api.post<ApiResponse>(`/businesses/${id}/remove-user`, data),
  switch: (companyId: string) => api.post<ApiResponse>('/businesses/switch', { company_id: companyId }),
};

// File Attachments
export const attachmentAPI = {
  list: (params: { entity_type: string; entity_id: string }) => api.get<ApiResponse>('/attachments', { params }),
  upload: (data: FormData) => api.post<ApiResponse>('/attachments', data, { headers: { 'Content-Type': 'multipart/form-data' } }),
  download: (id: string) => api.get(`/attachments/${id}/download`, { responseType: 'blob' }),
  delete: (id: string) => api.delete<ApiResponse>(`/attachments/${id}`),
};
