import numeral from 'numeral';
import { format, parseISO } from 'date-fns';

/** Currency symbol map */
const currencySymbols: Record<string, string> = {
  GHS: 'GH₵', USD: '$', EUR: '€', GBP: '£', NGN: '₦',
  XOF: 'CFA', XAF: 'CFA', KES: 'KSh', ZAR: 'R',
  TZS: 'TSh', UGX: 'USh', RWF: 'RF', EGP: 'E£',
  MAD: 'MAD', GMD: 'D', SLL: 'Le', LRD: 'L$',
  CNY: '¥', JPY: '¥', INR: '₹', AED: 'AED',
  CAD: 'C$', AUD: 'A$',
};

/** Format amount in given currency (defaults to GHS) */
export function formatCurrency(amount: number | string, currency: string = 'GHS'): string {
  const num = typeof amount === 'string' ? parseFloat(amount) : amount;
  const symbol = currencySymbols[currency] || currency;
  return `${symbol} ${numeral(num).format('0,0.00')}`;
}

/** Format a date string */
export function formatDate(date: string | Date, fmt: string = 'dd MMM yyyy'): string {
  if (!date) return '';
  const d = typeof date === 'string' ? parseISO(date) : date;
  return format(d, fmt);
}

/** Format a datetime string */
export function formatDateTime(date: string | Date): string {
  return formatDate(date, 'dd MMM yyyy HH:mm');
}

/** Format percentage */
export function formatPercent(value: number | string): string {
  const num = typeof value === 'string' ? parseFloat(value) : value;
  return `${numeral(num).format('0,0.00')}%`;
}

/** Get initials from name */
export function getInitials(firstName: string, lastName: string): string {
  return `${firstName?.[0] || ''}${lastName?.[0] || ''}`.toUpperCase();
}

/** Status badge color mapping */
export function getStatusColor(status: string): string {
  const colors: Record<string, string> = {
    draft: 'bg-gray-100 text-gray-800',
    sent: 'bg-blue-100 text-blue-800',
    paid: 'bg-green-100 text-green-800',
    partial: 'bg-yellow-100 text-yellow-800',
    overdue: 'bg-red-100 text-red-800',
    voided: 'bg-red-100 text-red-800',
    active: 'bg-green-100 text-green-800',
    inactive: 'bg-gray-100 text-gray-800',
    posted: 'bg-green-100 text-green-800',
    approved: 'bg-green-100 text-green-800',
    pending: 'bg-yellow-100 text-yellow-800',
    open: 'bg-blue-100 text-blue-800',
    closed: 'bg-gray-100 text-gray-800',
    completed: 'bg-green-100 text-green-800',
    refunded: 'bg-orange-100 text-orange-800',
    disposed: 'bg-red-100 text-red-800',
    fully_depreciated: 'bg-yellow-100 text-yellow-800',
  };
  return colors[status] || 'bg-gray-100 text-gray-800';
}

/** Truncate text */
export function truncate(text: string, length: number = 50): string {
  if (!text || text.length <= length) return text;
  return text.substring(0, length) + '...';
}

/** Ghana regions list */
export const ghanaRegions = [
  'Greater Accra', 'Ashanti', 'Central', 'Eastern', 'Western', 'Western North',
  'Northern', 'Savannah', 'North East', 'Upper East', 'Upper West',
  'Volta', 'Oti', 'Bono', 'Bono East', 'Ahafo',
];
